import logging

import torch
from src.metrics.ece import ECELoss
from torch import nn, optim
from tqdm import tqdm

from src.losses.kde_ece import get_ece_reg, get_bandwidth


# from: https://github.com/gpleiss/temperature_scaling/blob/master/temperature_scaling.py
class ModelWithTemperature(nn.Module):
    """
    A thin decorator, which wraps a model with temperature scaling model (nn.Module): A classification neural network
    NB: Output of the neural network should be the classification logits, NOT the softmax (or log softmax)!
    """
    def __init__(self, model, config):
        super(ModelWithTemperature, self).__init__()
        self.model = model
        self.config = config
        self.temperature = nn.Parameter(torch.ones(1) * 1.5)

    def forward(self, input):
        logits = self.model(input)
        return self.temperature_scale(logits)

    def temperature_scale(self, logits):
        """
        Perform temperature scaling on logits
        """
        # Expand temperature to match the size of logits
        temperature = self.temperature.unsqueeze(1).expand(logits.size(0), logits.size(1))
        return logits / temperature

    # This function probably should live outside of this class, but whatever
    def set_temperature(self, valid_loader, criterion, reg_weight):
        """
        Tune the temperature of the model (using the validation set).
        We're going to set it to optimize NLL, KDE-ECE or NLL-KDE-ECE (NLL + lambda * KDE-ECE).
        valid_loader (DataLoader): validation set loader
        """
        device = self.config['device']
        self.to(device)
        nll_criterion = nn.CrossEntropyLoss().to(device)
        ece_criterion = ECELoss().to(device)

        if criterion == 'nll':
            # First: collect all the logits and labels for the validation set
            logits_list = []
            labels_list = []
            with torch.no_grad():
                for input, label in tqdm(valid_loader):
                    input = input.to(device)
                    logits = self.model(input)
                    logits_list.append(logits)
                    labels_list.append(label)
                logits = torch.cat(logits_list).to(device)
                labels = torch.cat(labels_list).to(device)

            # Calculate NLL and ECE before temperature scaling
            self.log_temp_before_scaling(nll_criterion, ece_criterion, logits, labels)

            # Next: optimize the temperature w.r.t. NLL
            optimizer = optim.LBFGS([self.temperature], lr=0.01, max_iter=50)

            def eval():
                optimizer.zero_grad()
                loss = nll_criterion(self.temperature_scale(logits), labels)
                loss.backward()
                return loss
            optimizer.step(eval)

            self.log_temp_after_scaling(nll_criterion, ece_criterion, self.temperature_scale(logits), labels)

        return self

    def log_temp_after_scaling(self, nll_criterion, ece_criterion, logits, labels):
        # Calculate NLL and ECE after temperature scaling
        after_temperature_nll = nll_criterion(logits, labels).item()
        after_temperature_ece = ece_criterion(logits, labels).item()
        logging.info(f"Optimal temeprature: {self.temperature.item()}")
        logging.info(f"After temperature - NLL: {after_temperature_nll}, ECE: {after_temperature_ece}")

    @staticmethod
    def log_temp_before_scaling(nll_criterion, ece_criterion, logits, labels):
        before_temperature_nll = nll_criterion(logits, labels).item()
        before_temperature_ece = ece_criterion(logits, labels).item()
        logging.info(f"Before temperature - NLL: {before_temperature_nll}, ECE: {before_temperature_ece}")
