import torch
import torch.nn as nn
import torch.nn.functional as F


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1):
        super(BasicBlock, self).__init__()
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)

        self.downsample = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.downsample = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes, kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion*planes)
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out += self.downsample(x)
        out = F.relu(out)
        return out


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, in_planes, planes, stride=1):
        super(Bottleneck, self).__init__()
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv3 = nn.Conv2d(planes, self.expansion*planes, kernel_size=1, bias=False)
        self.bn3 = nn.BatchNorm2d(self.expansion*planes)

        self.downsample = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.downsample = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes, kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion*planes)
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = F.relu(self.bn2(self.conv2(out)))
        out = self.bn3(self.conv3(out))
        out += self.downsample(x)
        out = F.relu(out)
        return out


class ResNet(nn.Module):
    def __init__(self, block, num_blocks, num_layers, widen_factor=4, num_classes=10, num_input_channels=3):
        super(ResNet, self).__init__()
        self.in_planes = 64
        self.num_layers = num_layers

        width = 16 * widen_factor

        self.conv1 = nn.Conv2d(num_input_channels, 64, kernel_size=7, stride=2, padding=3, bias=False)
        self.bn1 = nn.BatchNorm2d(64)
        if self.num_layers >= 1:
            self.layer1 = self._make_layer(block, width, num_blocks[0], stride=1)
        if self.num_layers >= 2:
            self.layer2 = self._make_layer(block, width*2, num_blocks[1], stride=2)
        if self.num_layers >= 3:
            self.layer3 = self._make_layer(block, width*4, num_blocks[2], stride=2)
        if self.num_layers >= 4:
            self.layer4 = self._make_layer(block, width*8, num_blocks[3], stride=2)
        self.linear = nn.Linear(width*8*block.expansion, num_classes)
        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))
        #self.linear = nn.Linear(128 * block.expansion, num_classes)
        #import IPython; IPython.embed()

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1]*(num_blocks-1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layers)

    def forward(self, input_dict):
        x = input_dict['inputs']
        out = F.relu(self.bn1(self.conv1(x)))
        if self.num_layers == 1:
            out1 = self.layer1(out)
            out = self.avgpool(out1)
            emb = torch.flatten(out, 1)
            out = self.linear(emb)
            output_dict = {
                'logits': out, 'features': [out1], 'embedding': emb
            }
        elif self.num_layers == 2:
            out1 = self.layer1(out)
            out2 = self.layer2(out1)
            out = self.avgpool(out2)
            emb = torch.flatten(out, 1)
            out = self.linear(emb)
            output_dict = {
                'logits': out, 'features': [out1, out2], 'embedding': emb
            }
        elif self.num_layers == 3:
            out1 = self.layer1(out)
            out2 = self.layer2(out1)
            out3 = self.layer3(out2)
            out = self.avgpool(out3)
            emb = torch.flatten(out, 1)
            out = self.linear(emb)
            output_dict = {
                'logits': out, 'features': [out1, out2, out3], 'embedding': emb
            }
        else:
            out1 = self.layer1(out)
            out2 = self.layer2(out1)
            out3 = self.layer3(out2)
            out4 = self.layer4(out3)
            out = self.avgpool(out4)
            emb = torch.flatten(out, 1)
            out = self.linear(emb)
            output_dict = {
                'logits': out, 'features': [out1, out2, out3, out4], 'embedding': emb
            }
        # out3 = self.layer3(out2)
        # out4 = self.layer4(out3)
        # out = F.avg_pool2d(out4, 4)
        # out = out.view(out.size(0), -1)
        #out = self.avgpool(out2)
        #out = torch.flatten(out, 1)
        #out = self.linear(out)
        #output_dict = {
        #    'logits': out,
        #}
        # return out, [out1, out2, out3, out4]
        return output_dict


def ResNet18(num_layers, depth, widen_factor, dropout_rate, num_classes, num_input_channels=3, norm_layer=None):
    return ResNet(BasicBlock, [2,2,2,2], num_layers, num_classes=num_classes,
                  widen_factor=widen_factor, num_input_channels=num_input_channels)


def ResNet34(num_layers, depth, widen_factor, dropout_rate, num_classes, num_input_channels=3, norm_layer=None):
    return ResNet(BasicBlock, [3,4,6,3], num_layers, num_classes=num_classes,
                  widen_factor=widen_factor, num_input_channels=num_input_channels)


def ResNet50(num_layers, depth, widen_factor, dropout_rate, num_classes, num_input_channels=3, norm_layer=None):
    return ResNet(Bottleneck, [3,4,6,3], num_layers, num_classes=num_classes,
                  widen_factor=widen_factor, num_input_channels=num_input_channels)


def ResNet101(num_layers, depth, widen_factor, dropout_rate, num_classes, num_input_channels=3, norm_layer=None):
    return ResNet(Bottleneck, [3,4,23,3], num_layers, num_classes=num_classes,
                  widen_factor=widen_factor, num_input_channels=num_input_channels)


def ResNet152(num_layers, depth, widen_factor, dropout_rate, num_classes, num_input_channels=3, norm_layer=None):
    return ResNet(Bottleneck, [3,8,36,3], num_layers, num_classes=num_classes, widen_factor=widen_factor)


def ResNet200(num_layers, depth, widen_factor, dropout_rate, num_classes, num_input_channels=3, norm_layer=None):
    return ResNet(Bottleneck, [3, 24, 36, 3], num_layers, num_classes=num_classes, widen_factor=widen_factor)


def WideResNet50(num_layers, depth, widen_factor, dropout_rate, num_classes, num_input_channels=3, norm_layer=None):
    return ResNet(Bottleneck, [3, 4, 6, 3], num_layers, num_classes=num_classes, widen_factor=widen_factor)


def WideResNet101(num_layers, depth, widen_factor, dropout_rate, num_classes, num_input_channels=3, norm_layer=None):
    return ResNet(Bottleneck, [3, 4, 23, 3], num_layers, num_classes=num_classes, widen_factor=widen_factor)