import torchvision.models as models
import torch.nn as nn
import torch.nn.functional as F
import torchvision
import torch

import gen_BEV.utils as utils

def conv3x3(in_planes, out_planes, stride=1, groups=1, dilation=1):
    """3x3 convolution with padding"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride,
                     padding=dilation, groups=groups, bias=False, dilation=dilation)


def conv1x1(in_planes, out_planes, stride=1):
    """1x1 convolution"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=1, stride=stride, bias=False)

class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, inplanes, planes, stride=1, downsample=None, groups=1,
                 base_width=64, dilation=1, norm_layer=None):
        super(BasicBlock, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        if groups != 1 or base_width != 64:
            raise ValueError('BasicBlock only supports groups=1 and base_width=64')
        if dilation > 1:
            raise NotImplementedError("Dilation > 1 not supported in BasicBlock")
        # Both self.conv1 and self.downsample layers downsample the input when stride != 1
        self.conv1 = conv3x3(inplanes, planes, stride)
        self.bn1 = norm_layer(planes)
        self.relu = nn.ReLU(inplace=True)
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = norm_layer(planes)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        identity = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)

        if self.downsample is not None:
            identity = self.downsample(x)

        out += identity
        out = self.relu(out)

        return out

def _make_layer(block, inplanes, planes, blocks, stride=1):
        norm_layer = nn.BatchNorm2d
        downsample = nn.Sequential(
                conv1x1(inplanes, planes * block.expansion, stride),
                norm_layer(planes * block.expansion),
            )
        previous_dilation = 1
        layers = []
        layers.append(block(inplanes, planes, stride, downsample, 1,
                            64, previous_dilation, norm_layer))
        inplanes = planes * block.expansion
        for _ in range(1, blocks):
            layers.append(block(inplanes, planes, groups=1,
                                base_width=64, dilation=1,
                                norm_layer=norm_layer))

        return nn.Sequential(*layers)

class ResNet18(nn.Module):
    def __init__(self):
        super(ResNet18, self).__init__()
        # print('estimate_depth: ', estimate_depth)

        self.resnet18 = torchvision.models.resnet18(pretrained=True)
        # print(self.resnet18)

        self.conv1 = self.resnet18.conv1
        self.bn1 = self.resnet18.bn1
        self.relu = self.resnet18.relu
        self.maxpool = self.resnet18.maxpool
        
        self.layer1 = self.resnet18.layer1
        self.layer2 = self.resnet18.layer2
        self.layer3 = _make_layer(BasicBlock, 128, 256, 2)


    def forward(self, x):
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)
        x = self.maxpool(x)

        x = self.layer1(x)
        x = self.layer2(x)

        x = self.layer3(x)


        return x
    
    
class ResNet18_oxford(nn.Module):
    def __init__(self):
        super(ResNet18_oxford, self).__init__()
        # print('estimate_depth: ', estimate_depth)

        self.resnet18 = torchvision.models.resnet18(pretrained=True)
        # print(self.resnet18)

        self.conv1 = self.resnet18.conv1
        self.bn1 = self.resnet18.bn1
        self.relu = self.resnet18.relu
        self.maxpool = self.resnet18.maxpool
        
        self.layer1 = self.resnet18.layer1
        self.layer2 = _make_layer(BasicBlock, 64, 128, 2)
        self.layer3 = _make_layer(BasicBlock, 128, 256, 2)


    def forward(self, x):
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)
        # x = self.maxpool(x)

        x = self.layer1(x)
        x = self.layer2(x)

        x = self.layer3(x)


        return x


class ResNet50(nn.Module):
    def __init__(self):
        super(ResNet50, self).__init__()
        # print('estimate_depth: ', estimate_depth)

        self.resnet50 = torchvision.models.resnet50(pretrained=True)
        # print(self.resnet50)

        self.conv1 = self.resnet50.conv1
        self.bn1 = self.resnet50.bn1
        self.relu = self.resnet50.relu
        self.maxpool = self.resnet50.maxpool
        
        self.layer1 = self.resnet50.layer1
        self.layer2 = self.resnet50.layer2
        self.layer3 = _make_layer(BasicBlock, 512, 256, 3)

        

    def forward(self, x):
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)
        x = self.maxpool(x)

        x = self.layer1(x)
        x = self.layer2(x)

        x = self.layer3(x)


        return x


# a = ResNet50()
# b = torch.rand((1,3,512,512))
# c = a(b)
# print(c)
