import os
import json
import sys
import torch
from PIL import Image
import random
import uuid
import numpy as np

# Set environment variables
os.environ["NCCL_P2P_DISABLE"] = "1"
os.environ["NCCL_IB_DISABLE"] = "1"
os.environ["CUDA_VISIBLE_DEVICES"] = "3"

# Append LLaVA path
base_dir = '/home/user/llava/LLaVA'
sys.path.append(base_dir)

# Import seed control utility
from utils import setup_seeds

# Initialize seeds
setup_seeds(42)

# Directories
pope_file = "./src/cal_set/coco/coco_pope_random.json"
coco_annotations_file = "/hdd/user/vlm/coco/val2014/anno/instances_val2014.json"
image_folder = "/hdd/user/vlm/coco/val2014/"
output_dir = "/hdd/user/vlm/at/coco_100/"
output_name = "llava_data.json"
# output_dir = "/home/user/llava/LLaVA/playground/data/pope/"

# Load POPE JSON data
with open(pope_file, "r") as f:
    pope_data = [json.loads(line) for line in f]
# with open(pope_file, "r") as f:
#     pope_data = json.load(f) 
# Load COCO annotations
with open(coco_annotations_file, "r") as f:
    coco_data = json.load(f)

category_map = {cat['id']: cat['name'] for cat in coco_data['categories']}

# Extract unique images
unique_images = list({item["image"] for item in pope_data})

# Ensure output directory exists
os.makedirs(output_dir, exist_ok=True)

# Parameters
num_images_to_process = 100  # Number of unique images to process
num_crops_per_object = 10  # Number of augmented images per original image

# Function to crop and place objects
def crop_and_place(image, bbox, scale_factor, canvas_size):
    left, top, width, height = bbox
    min_size = min(canvas_size[0], canvas_size[1]) // 14  # Minimum size is original image size divided by 14
    scaled_width = max(min_size, min(canvas_size[0] // 2, int(width * scale_factor)))
    scaled_height = max(min_size, min(canvas_size[1] // 2, int(height * scale_factor)))
    cropped = image.crop((int(left), int(top), int(left + width), int(top + height)))
    resized = cropped.resize((scaled_width, scaled_height), Image.Resampling.LANCZOS)
    blank_image = Image.new("RGB", canvas_size, color=(255, 255, 255))
    rand_x = random.randint(0, canvas_size[0] - scaled_width)
    rand_y = random.randint(0, canvas_size[1] - scaled_height)
    blank_image.paste(resized, (rand_x, rand_y))
    return blank_image

# Process each image
final_output = []

for idx, image_name in enumerate(unique_images[:num_images_to_process]):
    print(f"Processing image {idx + 1}/{len(unique_images)}: {image_name}")

    # Get the POPE questions for the image
    pope_questions = [item for item in pope_data if item["image"] == image_name]

    image_path = os.path.join(image_folder, image_name)
    original_image = Image.open(image_path).convert("RGB")
    canvas_size = original_image.size

    # Get annotations for the current image
    image_id = int(image_name.split('_')[-1].split('.')[0])
    annotations = [anno for anno in coco_data['annotations'] if anno['image_id'] == image_id]

    object_counter = 0  # To keep track of object indices

    for question in pope_questions:
        if question["label"] == "yes":
            # Extract object name from the question text
            object_name = None
            if "Is there a " in question["text"]:
                object_name = question["text"].split("Is there a ")[1].split(" in the image?")[0]
            elif "Is there an " in question["text"]:
                object_name = question["text"].split("Is there an ")[1].split(" in the image?")[0]
            else:
                print(f"Unexpected question format: {question['text']}")
                continue

            object_annotations = [anno for anno in annotations if category_map[anno['category_id']].lower() == object_name.lower()]

            if not object_annotations:
                print(f"No annotations found for '{object_name}' in {image_name}. Skipping this question.")
                continue

            # Process crops for this object
            for crop_idx in range(num_crops_per_object):
                bbox = random.choice(object_annotations)['bbox']
                scaled_image = crop_and_place(original_image, bbox, random.uniform(1, 3.0), canvas_size)

                output_file_name = f"{image_name.split('_')[-1].split('.')[0]}-{object_counter}-{crop_idx}.jpg"
                output_file_path = os.path.join(output_dir, output_file_name)
                scaled_image.save(output_file_path)

                # Save JSON entry for "yes"
                final_output.append({
                    "id": str(uuid.uuid4()),
                    "image": output_file_name,
                    "conversations": [
                        {"from": "human", "value": f"<image>\n{question['text']}"},
                        {"from": "gpt", "value": "Yes"}
                    ],
                    "object_name": object_name,
                    "bbox": bbox,
                })

                # Add corresponding "no" entry
                no_questions = [q for q in pope_questions if q["label"] == "no"]
                if no_questions:
                    no_question = random.choice(no_questions)
                    final_output.append({
                        "id": str(uuid.uuid4()),
                        "image": output_file_name,
                        "conversations": [
                            {"from": "human", "value": f"<image>\n{no_question['text']}"},
                            {"from": "gpt", "value": "No"}
                        ],
                        "object_name": object_name,
                        "bbox": bbox,
                    })

            object_counter += 1

# Save all entries to a JSON file
output_json_path = os.path.join(output_dir, output_name)
with open(output_json_path, "w") as f:
    json.dump(final_output, f, indent=4)

print(f"Processing complete. JSON saved to {output_json_path}.")
