import re
from abc import ABC
from typing import Any, List, Tuple, Dict
import random
import difflib
from tenacity import retry, stop_after_attempt, wait_exponential, retry_if_exception_type

from .FL_prompt import *
from .FL_tools import *
from .util.api_requests import num_tokens_from_messages
from .util.postprocess_data import extract_code_blocks
from .util.preprocess_data import line_wrap_content, transfer_arb_locs_to_locs
from .util.model import make_model
from .ts_structure import CodeStructure
from .location import ComponentType, CodeLocation, CodeLocationGroup

class TypeOfMessage:
    file_localization_based_on_tree_request = "file_localization_based_on_tree_request"
    file_localization_based_on_tree_result = "file_localization_based_on_tree_result"
    file_localization_based_on_dependecy_request = "file_localization_based_on_dependecy_request"
    file_localization_based_on_dependecy_result = "file_localization_based_on_dependecy_result"
    file_localization_based_on_skeleton_request = "file_localization_based_on_skeleton_request"
    file_localization_based_on_skeleton_result = "file_localization_based_on_skeleton_result"

    initial_function_localize_request = "initial_function_localize_request"
    initial_function_localize_result = "initial_function_localize_result"
    function_call_result = "function_call_result"
    function_call_request_stressing_format = "function_call_request_stressing_format"
    function_execute_error = "function_execute_error"
    function_call_request = "function_call_request"
    code_relavance_judgement = "code_relevance_judgement"
    code_content = "code_content"
    summary_localization_request = "summary_localization_request"
    summary_localization_result = "summary_localization_result"
    exit_function_call = "exit_function_call"

class LLMService:
    """Encapsulates all interactions with language models."""
    def __init__(self, model_name: str, backend: str, logger, max_length=100000, max_tokens=4096, base_url=None, **kwargs):
        self.model_name = model_name
        self.backend = backend
        self.logger = logger
        self.base_url = base_url
        self.default_params = {
            "max_tokens": max_tokens,
            "temperature": 0.0,
            "batch_size": 1,
            **kwargs,
        }
        # Set max context length based on model name
        self.max_context_length = max_length
        self.max_new_tokens = max_tokens

    
    @retry(
        stop=stop_after_attempt(10), 
        wait=wait_exponential(multiplier=1, min=10, max=120),  
        retry=retry_if_exception_type(Exception) 
    )        
    def single_generate(self, messages: List[Dict[str, str]], num_samples: int = 1, **override_params) -> Dict:
        """Call model to generate content."""
        params = {**self.default_params, **override_params}
        model = make_model(
            model=self.model_name,
            backend=self.backend,
            logger=self.logger,
            base_url=self.base_url,
            **params
        )
        # In production, more complex exception handling and retry logic can be added here
        try:
            traj = model.codegen(messages, num_samples=num_samples)[0]
            traj["prompt"] = messages
            self.logger.info(f"API Return Successfully")
            return traj
        except Exception as e:
            self.logger.error(f"Model call failed: {e}")
            print(f"Model call failed: {e}")
            # Can return empty or failure traj based on exception type
            raise e

    def generate(self, messages: List[Dict[str, str]], num_samples: int = 1, **override_params) -> Dict:
        """Generate content with retry mechanism."""
        try:
            return self.single_generate(messages, num_samples, **override_params)
        except Exception as e:
            self.logger.error(f"Model call failed: {e}")
            return {
                "response": "Error: Model generation failed.",
                "usage": {"completion_tokens": 0, "prompt_tokens": 0},
            }
        

    def get_token_count(self, messages: List[Dict[str, str]]) -> int:
        """Calculate token count for messages."""
        return num_tokens_from_messages(messages, self.model_name)

class OutputParser:
    """Provides static methods for parsing model outputs."""
    @staticmethod
    def extract_code_files(content: str, suffix_set) -> List[str]:
        """Extract Python file name list from formatted text."""
        if content.count("```") % 2 != 0:
            return []
        
        extracted_blocks = re.findall(r'```(?:.*?\n)?(.*?)```', content, re.DOTALL)
        if not extracted_blocks:
            return []
        
        lines = "\n".join(extracted_blocks).strip().split('\n')
        parsed_list = [line.strip() for line in lines if line.strip().endswith(suffix_set)]
        return parsed_list

    @staticmethod
    def extract_code_from_block(content: str) -> str:
        """Extract code from Markdown code blocks."""
        match = re.search(r'```(?:[a-zA-Z]+\n)?(.*?)```', content, re.DOTALL)
        return match.group(1).strip() if match else ""

    @staticmethod
    def parse_function_call(content: str) -> Tuple[str, List[str]]:
        """Parse model-generated function call strings."""
        clean_content = content.replace("```python", "").replace("`", "").strip()
        match = re.match(r'(\w+)\((.*)\)', clean_content)
        if not match:
            return "", []

        func_name = match.group(1).strip()
        # Use more robust way to split arguments, handle commas in strings
        args_str = match.group(2)
        # Simple implementation, for more complex cases consider using ast.parse
        args = [arg.strip().strip("'\"") for arg in re.split(r",\s*(?![^()]*\))", args_str)]
        return func_name, args

class CodebaseService:
    """Encapsulates all interactions with codebase files and structures."""
    def __init__(self, instance_id: str, structure: CodeStructure):
        self.instance_id = instance_id
        self.structure = structure

    def get_file_summary(self, file_path: str) -> str:
        """Get class and function structure summary for a single file."""
        return self.structure.build_file_summary(file_path)

    def build_files_summary(self, files: List[str]) -> str:
        """Build structure summary for multiple files."""
        return "\n".join(self.get_file_summary(name) for name in files)

    def execute_code_retrieval(self, func_name: str, args: List[str]) -> Tuple[CodeLocation, str]:
        padded_args = (args + [None] * 3)[:3]
        if func_name == 'get_code_of_class':
            location = CodeLocation.for_class(padded_args[0], padded_args[1])
        elif func_name == 'get_code_of_class_function':
            location = CodeLocation.for_method(padded_args[0], padded_args[1], padded_args[2])
        elif func_name == 'get_code_of_file_function':
            location = CodeLocation.for_function(padded_args[0], padded_args[1])
        elif func_name == 'get_toplevel_code':
            location = CodeLocation.for_global(padded_args[0])
        else:
            return None, "Unknown function call."
        return location, self.structure.find(location=location)['text']

    def command_of_view_code(self, location: CodeLocation) -> str:
        if location.component_type == ComponentType.CLASS:
            return f"get_code_of_class({location.file_path}, {location.class_name})"
        elif location.component_type == ComponentType.METHOD:
            return f"get_code_of_class_function({location.file_path}, {location.class_name}, {location.member_name})"
        elif location.component_type == ComponentType.FUNCTION:
            return f"get_code_of_file_function({location.file_path}, {location.member_name})"
        else:
            return f"get_toplevel_code({location.file_path})"

class DialogueManager:
    """Manage dialogue history and trajectory."""
    def __init__(self, logger, system_prompt: str = ""):
        self.trajectory = []
        self.logger = logger
        if system_prompt:
            self.add_message("system", system_prompt)

    def add_message(self, role: str, content: str, **kwargs):
        """Add a message to dialogue history and trajectory."""
        self.trajectory.append({"role": role, "content": content, **kwargs})

    def add_user_message(self, content: str, **kwargs):
        self.add_message("user", content, **kwargs)

    def add_assistant_message(self, content: str, **kwargs):
        self.add_message("assistant", content, **kwargs)
        
    def get_messages(self) -> List[Dict[str, str]]:
        return self.trajectory.copy()

    def pop_last_message(self):
        """Remove the last message from dialogue trajectory."""
        if self.trajectory:
            return self.trajectory.pop()

    def present_dialogue(self):
        """Format and log the entire dialogue trajectory."""
        present = "Presentation of dialogue:\n"
        for item in self.trajectory:
            present += f">>>>>>>>>>>>>>{item['role']}\n"
            present += f"{item['content']}\n"
        self.logger.info(present)

class FL(ABC):
    """Base class remains unchanged."""
    def __init__(self, instance_id, structure: CodeStructure, problem_statement, **kwargs):
        self.structure = structure
        self.instance_id = instance_id
        self.problem_statement = problem_statement

class AFL(FL):
    def __init__(
            self,
            instance_id,
            structure: CodeStructure,
            problem_statement,
            model_name,
            model_max_length,
            model_max_new_tokens,
            backend,
            logger,
            base_url=None,
            **kwargs,
    ):
        super().__init__(instance_id, structure, problem_statement)
        self.logger = logger
        # Initialize all services
        self.llm = LLMService(model_name, backend, logger, max_length=model_max_length, max_tokens=model_max_new_tokens, base_url=base_url, **kwargs)
        self.parser = OutputParser()
        self.codebase = CodebaseService(instance_id, structure)

    def _get_best_file_match(self, file: str, all_files: list[str], cutoff: float = 0.8) -> str:
        """Use difflib to find best file match."""
        if file in all_files:
            return file
        matches = difflib.get_close_matches(file, all_files, n=1, cutoff=cutoff)
        return matches[0] if matches else file

    def localize_file(self, max_files=5, max_retry=10, mock=False):
        # If recursive call (with history), generate final result directly
        all_files = self.structure.get_all_files()
        system_msg = file_system_prompt_without_tool
        model_dialogue = DialogueManager(self.logger, system_prompt=system_msg)
        log_trajectory = DialogueManager(self.logger, system_prompt=system_msg)

        # Step 1: Initial file localization
        bug_report = bug_report_template.format(problem_statement=self.problem_statement, structure=self.structure.get_structure_string().strip())
        guidence_msg = file_guidence_prmpt_without_tool.format(pre_select_num=int(max_retry * 0.75), top_n=int(max_retry / 2))
        user_msg = f"{bug_report}\n{guidence_msg}\n{file_summary}"
        model_dialogue.add_user_message(user_msg)
        log_trajectory.add_user_message(user_msg)

        traj = self.llm.generate(model_dialogue.get_messages(), temperature=0)
        raw_output = traj["response"]
        model_dialogue.add_assistant_message(raw_output)
        log_trajectory.add_assistant_message(raw_output)
        
        found_files = self.parser.extract_code_files(raw_output, self.structure.language_manager.code_suffix_set())
        
        # Step 2: Format correction reflection
        if not found_files:
            self.logger.info("No files found, attempting format correction.")
            corrcted_tpl = format_correct_prompt.format(res=raw_output)
            format_traj = self.llm.generate([{"role": "user", "content": corrcted_tpl}])
            formated_res = format_traj["response"]
            self.logger.info(formated_res)
            # Update last message in model context and log
            model_dialogue.trajectory[-1]['content'] = formated_res
            log_trajectory.trajectory[-1]['content'] = formated_res
            found_files = self.parser.extract_code_files(formated_res, self.structure.language_manager.code_suffix_set())

        matched_files = [self._get_best_file_match(f, all_files) for f in found_files]
        matched_files = [f for f in matched_files if f in all_files] # Filter out unmatched files

        # Step 3: Reflection based on dependency relations
        import_content = "\n".join([f"file: {fi}\n {self.structure.get_imports_of_file(fi)}\n" for fi in matched_files])
        reflection_query = file_reflection_prompt_dependency.format(
            problem_statement=self.problem_statement,
            structure=self.structure.get_structure_string().strip(),
            import_content=import_content,
            pre_files=matched_files
        )

        # TODO: No-memory
        model_dialogue = DialogueManager(self.logger, system_prompt=system_msg)

        model_dialogue.add_user_message(reflection_query)
        log_trajectory.add_user_message(reflection_query)

        traj = self.llm.generate(model_dialogue.get_messages(), temperature=0)
        reflection_result = traj["response"]
        self.logger.info(reflection_result)
        model_dialogue.add_assistant_message(reflection_result)
        log_trajectory.add_assistant_message(reflection_result)
        reflection_files = self.parser.extract_code_files(reflection_result, self.structure.language_manager.code_suffix_set())
        reflection_files = [self._get_best_file_match(f, all_files) for f in reflection_files]
        reflection_files = [f for f in reflection_files if f in all_files]

        # Step 4: 基于骨架code进row最终Sort
        file_internal_structure = self.codebase.build_files_summary(reflection_files)
        final_query = final_rerank_based_on_skeleton_prompt.format(
            problem_statement=self.problem_statement,
            import_content=import_content,
            file_internal_structure=file_internal_structure
        )

        # TODO: No-memory
        model_dialogue = DialogueManager(self.logger, system_prompt=system_msg)
        
        model_dialogue.add_user_message(final_query)
        log_trajectory.add_user_message(final_query)

        traj = self.llm.generate(model_dialogue.get_messages(), temperature=0)
        final_results = traj["response"]
        self.logger.info(final_results)
        model_dialogue.add_assistant_message(final_results)
        log_trajectory.add_assistant_message(final_results)
        
        final_files = self.parser.extract_code_files(final_results, self.structure.language_manager.code_suffix_set())
        final_files = [self._get_best_file_match(f, all_files) for f in final_files]
        final_files = [f for f in final_files if f in all_files]

        log_trajectory.present_dialogue()

        self.logger.info(
            "Final predicted files:\n" + "\n".join(final_files) + '\n'
        )
        return final_files, {"raw_output_files": raw_output}, log_trajectory.trajectory

    def localize_func(self, max_files=5, max_functions=10, max_retry=10, files=None, mock=False, with_global=True):
        """Localize related functions (complete implementation)."""
        if with_global:
            localize_instruction = location_tool_prompt
        else:
            localize_instruction = location_tool_prompt_wo_toplevel

        if with_global:
            summary_instruction = location_summary
        else:
            summary_instruction = location_summary_wo_toplevel

        system_msg = location_system_prompt.format(functions=localize_instruction, max_try=max_retry)
        model_dialogue = DialogueManager(self.logger, system_prompt=system_msg)
        log_trajectory = DialogueManager(self.logger, system_prompt=system_msg)

        bug_report = bug_report_template_wo_repo_struct.format(problem_statement=self.problem_statement).strip()
        bug_file_content = self.codebase.build_files_summary(files)
        location_guidence_msg = location_guidence_prmpt.format(bug_file_list=bug_file_content, pre_select_num=7, top_n=5)
        user_msg = f"{bug_report}\n{location_guidence_msg}"
        model_dialogue.add_user_message(user_msg)
        log_trajectory.add_user_message(user_msg)

        output = self.llm.generate(model_dialogue.get_messages(), temperature=0.0) # 原code此处为0.85, 但其他地方为0.0, 统一为0.0
        reason = output['response']
        model_dialogue.add_assistant_message(reason)
        log_trajectory.add_assistant_message(reason)

        model_dialogue.add_user_message(call_function_prompt)
        log_trajectory.add_user_message(call_function_prompt)

        viewed_locations = CodeLocationGroup([])
        # 主Search循环
        for _ in range(max_retry):
            location_summary_tokens = self.llm.get_token_count([{"role": "user", "content": summary_instruction.format(bug_file_list=bug_file_content)}])
            current_tokens = self.llm.get_token_count(model_dialogue.get_messages())
            if current_tokens > self.llm.max_context_length - self.llm.max_new_tokens - 2 * location_summary_tokens:
                self.logger.warning("模型上下文长度接近极限, 修剪上下文.")
                model_dialogue.pop_last_message()
                log_trajectory.pop_last_message()
                break

            try:
                output = self.llm.generate(model_dialogue.get_messages(), temperature=0.0)
            except Exception as e:
                 if "Tokens" in str(e):
                    self.logger.warning("Model call failed due to token overflow, trimming context.")
                    model_dialogue.pop_last_message()
                    log_trajectory.pop_last_message()
                    break
                 raise e

            assistant_response = output['response']
            model_dialogue.add_assistant_message(assistant_response)
            log_trajectory.add_assistant_message(assistant_response)

            func_name, args = self.parser.parse_function_call(assistant_response)
            if not func_name:
                err_msg = "Please call functions in the right format to get enough information for your final answer." + localize_instruction
                model_dialogue.add_user_message(err_msg)
                log_trajectory.add_user_message(err_msg)
                continue

            # get the called locations and code content
            if func_name == "exit":
                break

            try:
                location, function_retval = self.codebase.execute_code_retrieval(func_name, args)
                viewed_locations = viewed_locations.union(CodeLocationGroup([location]))
            except Exception as e:
                self.logger.info(f"Retrivel code error: {e}")
                retrieval_err_msg = f"The call of {assistant_response} encountered error. Maybe you retrieved a location which does not exist."
                model_dialogue.add_user_message(retrieval_err_msg)
                model_dialogue.add_user_message(call_function_prompt)
                log_trajectory.add_user_message(retrieval_err_msg)
                log_trajectory.add_user_message(call_function_prompt)
                continue

            # Relationship judgement
            judge_query = check_func_retval_prompt.format(problem_statement=self.problem_statement, content=assistant_response, function_retval=function_retval)
            judge_output = self.llm.generate([{"role": "user", "content": judge_query}])
            is_related = "True" in self.parser.extract_code_from_block(judge_output['response'])
            print(f"Related judgement: {is_related}")
            
            if is_related:
                feedback = "I have already checked this function/class is related to the task."
                log_next_prompt = call_function_prompt + "\nYou can check the functions that this part of code calls.\n"
                model_next_prompt = log_next_prompt + "\nStructure:\n" + bug_file_content
            else:
                feedback = "I have already checked this function/class is not related to the task."
                log_next_prompt = call_function_prompt + "\nDon't check the functions that this part of code calls.\n"
                model_next_prompt = log_next_prompt

            model_dialogue.add_user_message(feedback)
            model_dialogue.add_user_message(function_retval)
            # TODO: agent
            model_dialogue.add_user_message(log_next_prompt)
            # model_dialogue.add_user_message(model_next_prompt)

            log_trajectory.add_user_message(feedback)
            log_trajectory.add_user_message(function_retval)
            log_trajectory.add_user_message(log_next_prompt)

#
        summary_prompt = summary_instruction.format(bug_file_list=bug_file_content)
        model_dialogue.add_user_message(summary_prompt)
        log_trajectory.add_user_message(summary_prompt)
        
        final_traj = self.llm.generate(model_dialogue.get_messages())
        raw_output = final_traj['response']
        self.logger.info("Original output ->\n" + raw_output)

        model_dialogue.add_assistant_message(raw_output)
        log_trajectory.add_assistant_message(raw_output)

        log_trajectory.present_dialogue()

        # 最终的Check
        model_found_locs_str = extract_code_blocks(raw_output)
        model_found_locs_standard = self.structure.map_output_locations_by_files(model_found_locs_str, files)
        try:
            model_found_locations = CodeLocationGroup.from_file_location_string_map(model_found_locs_standard)
        
            self.logger.info(
                "Predict locations:\n" + str(model_found_locations) + '\n'
            )
        except Exception as e:
            self.logger.error(f"解析模型输出的位置时出错: {e}")
            model_found_locations = CodeLocationGroup([])
        

        final_locs = self.structure.map_output_locations_by_files(extract_code_blocks(raw_output), files)
        return (
            final_locs,
            raw_output, # 原始output仍然是重要的
            log_trajectory.trajectory,
        )
    

    def localize_file_with_gt(self, gt_localization, max_files=5, max_retry=10, mock=False, history=[]):
        """本地化相关files (完整实现)."""
        # If recursive call (with history), generate final result directly
        all_files = self.structure.get_all_files()
        system_msg = file_system_prompt_without_tool
        model_dialogue = DialogueManager(self.logger, system_prompt=system_msg)
        log_trajectory = DialogueManager(self.logger, system_prompt=system_msg)

        if history:
            model_dialogue.trajectory = history

        # Step 1: 初始fileslocalization
        bug_report = bug_report_template.format(problem_statement=self.problem_statement, structure=self.structure.get_structure_string().strip())
        guidence_msg = file_guidence_prmpt_without_tool.format(pre_select_num=int(max_retry * 0.75), top_n=int(max_retry / 2))
        user_msg = f"{bug_report}\n{guidence_msg}\n{file_summary}"
        model_dialogue.add_user_message(user_msg)
        log_trajectory.add_user_message(user_msg, type=TypeOfMessage.file_localization_based_on_tree_request)

        traj = self.llm.generate(model_dialogue.get_messages(), temperature=0)
        raw_output = traj["response"]
        # model_dialogue.add_assistant_message(raw_output)
        log_trajectory.add_assistant_message(raw_output, type=TypeOfMessage.file_localization_based_on_tree_result)
        
        found_files = self.parser.extract_code_files(raw_output, self.structure.language_manager.code_suffix_set())
        
        # Step 2: format修正反思
        if not found_files:
            self.logger.info("No files found, attempting format correction.")
            corrcted_tpl = format_correct_prompt.format(res=raw_output)
            format_traj = self.llm.generate([{"role": "user", "content": corrcted_tpl}])
            formated_res = format_traj["response"]
            self.logger.info(formated_res)
            # Update last message in model context and log
            # model_dialogue.trajectory[-1]['content'] = formated_res
            log_trajectory.trajectory[-1]['content'] = formated_res
            found_files = self.parser.extract_code_files(formated_res, self.structure.language_manager.code_suffix_set())

        matched_files = [self._get_best_file_match(f, all_files) for f in found_files]
        matched_files = [f for f in matched_files if f in all_files] # Filter out unmatched files

        # Step 3: 基于依赖relation进row反思
        import_content = "\n".join([f"file: {fi}\n {self.structure.get_imports_of_file(fi)}\n" for fi in matched_files])
        reflection_query = file_reflection_prompt_dependency.format(
            problem_statement=self.problem_statement,
            structure=self.structure.get_structure_string().strip(),
            import_content=import_content,
            pre_files=matched_files
        )
        model_dialogue.trajectory[-1]['content'] = reflection_query
        log_trajectory.add_user_message(reflection_query, type=TypeOfMessage.file_localization_based_on_dependecy_request)

        traj = self.llm.generate(model_dialogue.get_messages(), temperature=0)
        reflection_result = traj["response"]
        self.logger.info(reflection_result)
        # model_dialogue.add_assistant_message(reflection_result)
        log_trajectory.add_assistant_message(reflection_result, type=TypeOfMessage.file_localization_based_on_dependecy_result)
        reflection_files = self.parser.extract_code_files(reflection_result, self.structure.language_manager.code_suffix_set())
        reflection_files = [self._get_best_file_match(f, all_files) for f in reflection_files]
        reflection_files = [f for f in reflection_files if f in all_files]

        # Step 4: 基于骨架code进row最终Sort
        file_internal_structure = self.codebase.build_files_summary(reflection_files)
        final_query = final_rerank_based_on_skeleton_prompt.format(
            problem_statement=self.problem_statement,
            import_content=import_content,
            file_internal_structure=file_internal_structure
        )
        model_dialogue.trajectory[-1]['content'] = final_query
        log_trajectory.add_user_message(final_query, type=TypeOfMessage.file_localization_based_on_skeleton_request)

        traj = self.llm.generate(model_dialogue.get_messages(), temperature=0)
        final_results = traj["response"]
        self.logger.info(final_results)
        # model_dialogue.add_assistant_message(final_results)
        log_trajectory.add_assistant_message(final_results, type=TypeOfMessage.file_localization_based_on_skeleton_result)
        
        final_files = self.parser.extract_code_files(final_results, self.structure.language_manager.code_suffix_set())
        final_files = [self._get_best_file_match(f, all_files) for f in final_files]
        final_files = [f for f in final_files if f in all_files]
        
        # Step 5: 与Ground Truth集成
        gt_files = gt_localization.get("modified_files", []) + gt_localization.get("removed_files", [])
        re_generate_flag = not all(gt_file in final_files for gt_file in gt_files)
        self.logger.info(f"GT files: {gt_files}, Final files: {final_files}, re_generate_flag: {re_generate_flag}")
        
        if re_generate_flag and not history: # 只在第一次运row时触发
            final_files_not_in_gt_files = [f for f in final_files if f not in gt_files]
            virtual_target_files = (gt_files + final_files_not_in_gt_files)[:max_files]
            file_answer = "```\n" + "\n".join(virtual_target_files) + "\n```"
            imitation_prompt = f"Well done. But finally we want to get these files as the final localization results. Please think again and imitate the dialogue above to give this final result: \n{file_answer}\nBut, you should act as if you don't know this information, and derive it step by step."
            
            return self.localize_file_with_gt(
                gt_localization, max_files, max_retry, mock, 
                history=log_trajectory.trajectory + [{"role": "user", "content": imitation_prompt}]
            )

        log_trajectory.present_dialogue()
        return final_files, {"raw_output_files": raw_output}, log_trajectory.trajectory

    def localize_func_with_gt(self, gt_localization, max_files=5, max_functions=10, max_retry=5, files=None, mock=False):
        """useGT构造localization相关functiondata"""

        gt_edit_locations = CodeLocationGroup.from_file_location_string_map(gt_localization["related_locs"])
        gt_edit_location_num = len(gt_edit_locations)

        system_msg = location_system_prompt.format(functions=location_tool_prompt, max_try=max_retry)
        model_dialogue = DialogueManager(self.logger, system_prompt=system_msg)
        log_trajectory = DialogueManager(self.logger, system_prompt=system_msg)

        gt_files = gt_localization.get("modified_files", []) + gt_localization.get("removed_files", [])
        if files is None:
            files = []
        if not all(file in files for file in gt_files):
            self.logger.warning(f"GT files {gt_files} are not all in the files {files}, will use GT files instead.")
            files_not_in_gt_files = [f for f in files if f not in gt_files]
            combined_files = (gt_files + files_not_in_gt_files)
            files = combined_files[:max_files]

        bug_report = bug_report_template_wo_repo_struct.format(problem_statement=self.problem_statement).strip()
        bug_file_content = self.codebase.build_files_summary(files)
        location_guidence_msg = location_guidence_prmpt.format(bug_file_list=bug_file_content, pre_select_num=7, top_n=5)
        user_msg = f"{bug_report}\n{location_guidence_msg}"
        model_dialogue.add_user_message(user_msg)
        log_trajectory.add_user_message(user_msg, type=TypeOfMessage.initial_function_localize_request)

        output = self.llm.generate(model_dialogue.get_messages(), temperature=0.0) # 原code此处为0.85, 但其他地方为0.0, 统一为0.0
        reason = output['response']
        model_dialogue.add_assistant_message(reason)
        log_trajectory.add_assistant_message(reason, type=TypeOfMessage.initial_function_localize_result)

        model_dialogue.add_user_message(call_function_prompt + bug_file_content)
        log_trajectory.add_user_message(call_function_prompt, fake=False, type=TypeOfMessage.function_call_request, round=0)

        viewed_locations = CodeLocationGroup([])

        # iterationepoch决定method
        # max_retry = min5, gt_edit_location_num, max10
        max_retry = min(max(gt_edit_location_num*2, 5), 10)
        print(f"Max retry time: {max_retry}")
        # 主Search循环
        for round_ in range(max_retry):
            location_summary_tokens = self.llm.get_token_count([{"role": "user", "content": location_summary.format(bug_file_list=bug_file_content)}])
            current_tokens = self.llm.get_token_count(model_dialogue.get_messages())
            if current_tokens > self.llm.max_context_length - self.llm.max_new_tokens - 2 * location_summary_tokens:
                self.logger.warning("模型上下文长度接近极限, 修剪上下文.")
print(", .")
                model_dialogue.pop_last_message()
                log_trajectory.pop_last_message()
                break

            try:
                output = self.llm.generate(model_dialogue.get_messages(), temperature=0.0)
            except Exception as e:
                if "Tokens" in str(e):
                    self.logger.warning("Model call failed due to token overflow, trimming context.")
                    model_dialogue.pop_last_message()
                    log_trajectory.pop_last_message()
                    break
                raise e

            assistant_response = output['response']
            model_dialogue.add_assistant_message(assistant_response)
            log_trajectory.add_assistant_message(assistant_response, fake=False, type=TypeOfMessage.function_call_result, round=round_)

            func_name, args = self.parser.parse_function_call(assistant_response)
            if not func_name:
                err_msg = "Please call functions in the right format to get enough information for your final answer." + location_tool_prompt
                model_dialogue.add_user_message(err_msg)
                log_trajectory.add_user_message(err_msg, fake=False, type=TypeOfMessage.function_call_request_stressing_format, round=round_+1)
                continue

            # get the called locations and code content
            if func_name == "exit":
                break

            try:
                location, function_retval = self.codebase.execute_code_retrieval(func_name, args)
                viewed_locations = viewed_locations.union(CodeLocationGroup([location]))
            except Exception as e:
                self.logger.info(f"Retrivel code error: {e}")
                retrieval_err_msg = f"The call of {assistant_response} encountered error. Maybe you retrieved a location which does not exist."
                model_dialogue.add_user_message(retrieval_err_msg)
                model_dialogue.add_user_message(call_function_prompt + "\nStructure:\n" + bug_file_content)
                log_trajectory.add_user_message(retrieval_err_msg, fake=False, type=TypeOfMessage.function_execute_error, round=round_)
                log_trajectory.add_user_message(call_function_prompt, fake=False, type=TypeOfMessage.function_call_request, round=round_+1)
                continue

            # Relationship judgement
            judge_query = check_func_retval_prompt.format(problem_statement=self.problem_statement, content=assistant_response, function_retval=function_retval)
            judge_output = self.llm.generate([{"role": "user", "content": judge_query}])
            is_related = "True" in self.parser.extract_code_from_block(judge_output['response'])
            print(f"Related judgement: {is_related}")
            
            if is_related:
                feedback = "I have already checked this function/class is related to the task."
                log_next_prompt = call_function_prompt + "\nYou can check the functions that this part of code calls.\n"
                model_next_prompt = log_next_prompt + "\nStructure:\n" + bug_file_content
            else:
                feedback = "I have already checked this function/class is not related to the task."
                log_next_prompt = call_function_prompt + "\nDon't check the functions that this part of code calls.\n"
                model_next_prompt = log_next_prompt

            model_dialogue.add_user_message(feedback)
            model_dialogue.add_user_message(function_retval)
            model_dialogue.add_user_message(model_next_prompt)

            log_trajectory.add_user_message(feedback, fake=False, type=TypeOfMessage.code_relavance_judgement, round=round_)
            log_trajectory.add_user_message(function_retval, fake=False, type=TypeOfMessage.code_content, round=round_)
            log_trajectory.add_user_message(log_next_prompt, fake=False, type=TypeOfMessage.function_call_request, round=round_+1)
            print("End of construct of iterative func localization")
            

        # GT集成: 只影响log轨迹
        left_components = []
        for gt_loc in gt_edit_locations:
            if gt_loc not in viewed_locations:
                left_components.append(gt_loc)

        round_ += 1
        for left_comp in left_components:
            function_retval = self.structure.find(left_comp)["text"]
            log_trajectory.add_message('assistant', content=self.codebase.command_of_view_code(left_comp), fake=True, type=TypeOfMessage.function_call_result, round=round_)
            log_trajectory.add_message("user", content="I have already checked this function/class is related to the task.", fake=True, type=TypeOfMessage.code_relavance_judgement, round=round_)
            log_trajectory.add_message("user", content=function_retval, fake=True, type=TypeOfMessage.code_content, round=round_)
            log_trajectory.add_message("user", content=call_function_prompt + "\nYou can check the functions that this part of code calls.\n", fake=True, type=TypeOfMessage.function_call_request, round=round_+1)
            round_ += 1
        
        # Exit
        log_trajectory.add_assistant_message("exit()", fake=False, type=TypeOfMessage.exit_function_call, round=round_)

#
        summary_prompt = location_summary.format(bug_file_list=bug_file_content)
        model_dialogue.add_user_message(summary_prompt)
        log_trajectory.add_user_message(summary_prompt, type=TypeOfMessage.summary_localization_request)
        
        final_traj = self.llm.generate(model_dialogue.get_messages())
        raw_output = final_traj['response']
        self.logger.info("Original output ->\n" + raw_output)

        # 最终的GTCheckandresult伪造
        model_found_locs_str = extract_code_blocks(raw_output)
        model_found_locs_standard = self.structure.map_output_locations_by_files(model_found_locs_str, files)
        model_found_locations = CodeLocationGroup.from_file_location_string_map(model_found_locs_standard)
        

        all_gt_in_model = all(gt_loc in model_found_locations for gt_loc in gt_edit_locations)
        left_locations = [found_loc for found_loc in model_found_locations if found_loc not in gt_edit_locations]

        self.logger.info(
            "Ground Truth Locations:\n" + str(gt_edit_locations) + '\n'
            "Predict locations:\n" + str(model_found_locations) + '\n'
            "Predict Locations that not in ground truth locations:\n" + str(CodeLocationGroup(left_locations))
        )
        
        if all_gt_in_model:
            final_assistant_output = raw_output
        else:
            virtual_locs_list = (gt_edit_locations.locations + left_locations)[:max_functions]
            virtual_locs_group = CodeLocationGroup(virtual_locs_list)
            virtual_locs_string = [loc.get_output_format() for loc in virtual_locs_group]
            # TODO: the order may not be ground truth - left after constructing group
            final_assistant_output = '```\n' + '\n\n'.join(virtual_locs_string) + '\n```'
            self.logger.info("Original output ->\n" + final_assistant_output)
        
        log_trajectory.add_assistant_message(final_assistant_output, type=TypeOfMessage.summary_localization_request)
        log_trajectory.present_dialogue()

        final_locs = self.structure.map_output_locations_by_files(extract_code_blocks(final_assistant_output), files)
        return (
            final_locs,
            raw_output, # 原始output仍然是重要的
            log_trajectory.trajectory,
        )

def construct_topn_file_context(
        file_to_locs,
        pred_files,
        file_contents,
        structure,
        context_window: int,
        loc_interval: bool = True,
        fine_grain_loc_only: bool = False,
        add_space: bool = False,
        sticky_scroll: bool = False,
        no_line_number: bool = True,
):
    file_loc_intervals = dict()
    topn_content = ""

    for pred_file, locs in file_to_locs.items():
        content = file_contents[pred_file]
        line_locs, context_intervals = transfer_arb_locs_to_locs(
            locs,
            structure,
            pred_file,
            context_window,
            loc_interval,
            fine_grain_loc_only,
            file_content=file_contents[pred_file] if pred_file in file_contents else "",
        )

        if len(line_locs) > 0:
            file_loc_content = line_wrap_content(
                content,
                context_intervals,
                add_space=add_space,
                no_line_number=no_line_number,
                sticky_scroll=sticky_scroll,
            )
            topn_content += f"### {pred_file}\n{file_loc_content}\n\n\n"
            file_loc_intervals[pred_file] = context_intervals

    return topn_content, file_loc_intervals

