import sys
sys.path.append('..')
from pathlib import Path
import requests
from typing import Optional

def get_openai_key():
    key_path = Path("..")
    return key_path.read_text(encoding="utf-8").strip()

def get_llama_endpoint():
    # For Llama, typically you run a local server
    return "http://localhost:11434"  # Default Ollama endpoint

class LlamaClient:
    def __init__(self, logger):
        self.logger = logger
        self.endpoint = f"{get_llama_endpoint()}/api/generate"
        self.model = "llama3:8b"  # Updated to use llama3
        self.headers = {"Content-Type": "application/json"}
        self.timeout = 60  # Increased timeout for complex reasoning
        
    def generate(self, prompt: str) -> Optional[str]:
        payload = {
            "model": self.model,
            "prompt": prompt,
            "stream": False,
            "options": {
                "temperature": 0.1,  # More deterministic than llama2
                "num_ctx": 4096,     # Matches your config
                "num_gpu": 1         # Utilize your GPU
            }
        }
        
        try:
            response = requests.post(
                self.endpoint,
                json=payload,
                headers=self.headers,
                timeout=self.timeout
            )
            response.raise_for_status()
            result = response.json().get("response", "").strip()
            if not result:
                raise ValueError("Empty response from Llama API")
            return result
        except requests.exceptions.RequestException as e:
            self.logger.error(f"Llama API request failed: {e}")
            raise RuntimeError(f"Llama API request failed: {e}")
        except ValueError as e:
            self.logger.error(f"Llama API returned invalid response: {e}")
            raise
        except Exception as e:
            self.logger.error(f"Unexpected error in LlamaClient: {e}")
            raise RuntimeError(f"Unexpected error in LlamaClient: {e}")

    