import pandas as pd
from typing import Tuple

def calculate_accuracy_stats(df: pd.DataFrame, logger=None) -> Tuple[pd.DataFrame, dict]:
    """
    Calculate verification accuracy statistics from results DataFrame.
    
    Args:
        df: DataFrame containing verification results with columns:
            - 'type': Rule type ('definite_rule', 'constraint', 'fact')
            - 'is_correct': Boolean verification result
        logger: Optional logger for output (default: None)
    
    Returns:
        Tuple of (stats_df, summary_dict) where:
        - stats_df: DataFrame with accuracy stats by rule type
        - summary_dict: Dictionary with overall statistics
    
    Example:
        >>> stats, summary = calculate_accuracy_stats(results_df, logger)
        >>> print(stats)
    """
    # Calculate by-type statistics
    stats_df = (
        df.groupby('type')['is_correct']
        .agg(['count', 'sum', 'mean'])
        .rename(columns={
            'count': 'total_rules',
            'sum': 'correct',
            'mean': 'accuracy'
        })
    )
    
    # Calculate overall statistics
    summary_dict = {
        'overall_accuracy': df['is_correct'].mean(),
        'total_rules': len(df),
        'correct_rules': df['is_correct'].sum()
    }
    
    # Log results if logger provided
    if logger:
        logger.info("\n=== Verification Statistics ===")
        for rule_type, row in stats_df.iterrows():
            logger.info(
                f"{rule_type:<15}: {row['correct']}/{row['total_rules']} "
                f"({row['accuracy']:.1%}) correct"
            )
        logger.info(f"\nOVERALL ACCURACY: {summary_dict['overall_accuracy']:.1%}")
    
    return stats_df, summary_dict