import sys
sys.path.append('..')
from config.predicates import get_predicate_defs

def generate_rule_verification_prompt(rule):
    predicate_defs = get_predicate_defs()
    cleaned_predicates = " ".join(predicate_defs.replace("\n", ". ").split())
    
    if ':-' in rule:
        head, body = rule.split(':-', 1)
        head = head.strip()
        body = body.strip().rstrip('.')
        body_atoms = split_body_atoms(body)  # Updated line
        
        if not head:  # Constraint case
            return {
                "type": "constraint",
                "prompt": (
                    "Consider these predicate definitions:\n"
                    f"{cleaned_predicates}\n\n"
                    "Given that all of the following atoms are true:\n"
                    f"{', '.join(body_atoms)}\n\n"
                    "Can this combination of facts logically exist?\n"
                    "Answer exactly one of:\n"
                    "[Possible] [Impossible] [Inevitable]"
                ),
                "expected": "Impossible"
            }
        else:  # Definite rule
            head_pred, head_vars = _parse_head(head)
            add_special_line = _check_special_case(head_vars, body_atoms)
            
            # Determine prompt based on arity
            if len(head_vars) == 1:  # Unary predicate
                relationship_prompt = (
                    f"Give a predicate 'rel' such that rel({','.join(head_vars)}) "
                    f'''describes a property of {', '.join(head_vars)} that must be true. the 'rel' that you can choose is a unary predicate, which takes only one argument.
                      For example: A response mother_of(X,Y) is not valid choice for you as mother_of is a binary predicate which takes two arguments X, and Y. However, 
                        is_gender(Y) is a valid choice as it is a unary predicate which takes a single argument Y. Your task is to find the correct valid response '''
                )
                format_instruction = f"rel({','.join(head_vars)})"
            else:  # Binary/N-ary predicate
                relationship_prompt = f"What is the relationship between {', '.join(head_vars)}?"
                format_instruction = f"rel({','.join(head_vars)})"
            
            prompt_lines = [
                "Consider these predicate definitions:",
                cleaned_predicates,
                "",
                "Given that all of the following atoms are true:",
                ', '.join(body_atoms),
                "",
                relationship_prompt,
                "Provide only the predicate with variables in exactly this format:",
                format_instruction,
                f"What is the predicate name that should replace 'rel'? "
                f'''Your response should be {format_instruction}, where rel is your guess. If you think multiple predicates could work, you must choose the most specific one. For example:
                 - If both brother and sibling are suitable, choose brother as it's more specific.'''
            ]
            
            if add_special_line:
                body_pred = body_atoms[0].split('(')[0].strip()
                prompt_lines.insert(6, 
                    f"SPECIAL NOTE: You cannot choose '{body_atoms[0]}' as your answer, "
                    f"we are looking for a more general version of the relationship '{body_pred}'"
                )
            return {
                        "type": "definite_rule",
                        "prompt": '\n'.join(prompt_lines),
                        "expected_head": head,
                        "head_pred": head_pred,
                        "head_vars": head_vars,
                        "is_unary": len(head_vars) == 1  # Optional: add this flag if useful for verification
                    }
    else:  # Fact
        return {
            "type": "fact",
            "prompt": (
                "Consider these predicate definitions:\n"
                f"{cleaned_predicates}\n\n"
                "Evaluate this statement:\n"
                f"{rule}\n\n"
                "Is this statement correct?\n"
                "Answer exactly one of:\n"
                "[True] [False]"
            ),
            "expected": "True"
        }
### utility methods
def _parse_head(head):
    """Extracts predicate and variables from rule head"""
    # Remove all whitespace for consistent parsing
    head = ''.join(head.split())
    pred = head.split('(')[0]
    vars_str = head.split('(')[1].split(')')[0]
    vars_list = [v.strip() for v in vars_str.split(',')]
    return pred, vars_list

def _check_special_case(head_vars, body_atoms):
    """Check if body has exactly one atom with vars matching head vars"""
    if len(body_atoms) != 1:
        return False    
    body_atom = body_atoms[0]
    try:
        body_pred, body_vars = _parse_head(body_atom)
        return body_vars == head_vars
    except:
        return False

def split_body_atoms(body):
    """Splits body atoms while respecting parentheses"""
    atoms = []
    current = []
    paren_level = 0
    
    for char in body:
        if char == '(':
            paren_level += 1
        elif char == ')':
            paren_level -= 1
        
        if char == ',' and paren_level == 0:
            atoms.append(''.join(current).strip())
            current = []
        else:
            current.append(char)
    
    if current:
        atoms.append(''.join(current).strip())
    
    return atoms

if __name__ == "__main__":
    file_path = f'../data/world_rules.txt'
    with open(file_path, 'r') as f:
            rules = [r.strip() for r in f.read().split('\n') 
                    if r.strip() and not r.strip().startswith('%')]
    rule_num =202
    prompt_info = generate_rule_verification_prompt(rules[rule_num])
    print(f'''   For rule {rules[rule_num]} ..\n we have {prompt_info}          ''')

