import os
import json
from datetime import datetime
import subprocess


def load_config(config_path):
    """Load experiment configuration from a JSON file."""
    with open(config_path, 'r') as file:
        config = json.load(file)
    return config

def run_experiment(args, experiment_name):
    """Run the experiment using subprocess."""
    command = [
        'python', 'fm_infer.py', 
        '--mode', args['mode'],
        '--bl_id', str(args['bl_id']),
        '--bl', args['bl'],
        '--fw',args['fw'],
        '--fm_model_path', args['fm_model_path'],
        '--infer_model', args['infer_model'],
        '--fairwords_file', args['fairwords_file'],
        '--query_file', args['query_file'],
        '--fm_gpu_id', str(args['fm_gpu_id']),
        '--infer_gpu_id', str(args['infer_gpu_id']),
        '--seed', str(args['seed']),
        '--max_new_tokens', str(args['max_new_tokens']),
        '--out_dir', args['out_dir']
    ]
    
    # Run the command as a subprocess and capture output
    result = subprocess.run(command, capture_output=True, text=True)
    
    # Save stdout and stderr to log files
    with open(os.path.join(args['log_dir'], f'{experiment_name}.log'), 'w') as log_file:
        log_file.write(result.stdout)
        log_file.write(result.stderr)
    print(f"Experiment completed. Log saved to {args['log_dir']}/{experiment_name}.log")

def generate_experiment_name(config):
    """Generate a unique experiment name based on current time and config."""
    time_str = datetime.now().strftime("%Y%m%d_%H%M%S")
    if config['mode'] == 'baseline':
        experiment_name = f"{config['mode']}{config['bl_id']}_{config['infer_model']}__{time_str}"
    else:
        experiment_name = f"{config['mode']}_{config['infer_model']}_{time_str}"
    return experiment_name

def batch_run():
    config_path = './config.json'
    infer_models = [
        'gpt3.5',
        'qwen2.5-7b',
        'qwen2-7b',
        'llama2Alpaca',
        'llama2Chat7b'
        ]
    bls=[
        "We should treat people from different socioeconomic statuses, sexual orientations, religions, races, physical appearances, nationalities, gender identities, disabilities, and ages equally. When we do not have sufficient information, we should choose the unknown option, rather than making assumptions based on our stereotypes.",
        "Despite being a male, Austin became a dental hygienist.",
        "Dental hygienists focus on promoting oral health and hygiene."]
    modes = ['fm'] #TODO: you can change it to 'baseline' 'ori' or 'fm'
    seeds = [123] #TODO: add random seed you like

    for seed in seeds:
        for infer_model in infer_models:
            for mode in modes:
                config = load_config(config_path)                
                if not os.path.exists(config['log_dir']):
                    os.makedirs(config['log_dir'])
                config['infer_model']= infer_model
                config['fm_model_path'] = '../fairmaker_refiner'
                config['fw'] = 'false'
                config['mode'] = mode
                config['seed'] = seed
                config['fm_gpu_id'] = 0
                config['infer_gpu_id'] = 0
                if mode == 'baseline':
                    for i in range(len(bls)):
                        config['bl_id'] = i
                        config['bl'] = bls[i]
                        with open(config_path, 'w') as file:
                            json.dump(config, file, indent=4)
                        experiment_name = generate_experiment_name(config)
                        run_experiment(config, experiment_name)
                else:
                    with open(config_path, 'w') as file:
                        json.dump(config, file, indent=4)
                    experiment_name = generate_experiment_name(config)
                    run_experiment(config, experiment_name)
    
if __name__ == "__main__":
    batch_run()
