# YOLOv3 🚀 by Ultralytics, GPL-3.0 license
"""
Train a  model on a custom dataset

Usage:
    $ python path/to/train.py --data coco128.yaml --weights yolov3.pt --img 640
"""
import argparse
import cv2
import os
import random
import sys
import time
from copy import deepcopy
from datetime import datetime
from pathlib import Path
from PIL import Image
import numpy as np
import torch
import torch.distributed as dist
import torch.nn as nn
import yaml
from torch.cuda import amp
from torch.nn.parallel import DistributedDataParallel as DDP
from torch.optim import SGD, Adam, lr_scheduler
from tqdm import tqdm
from models.attention import YOLOV3Attention
from models.yolov3_detector import YOLOV3ObjectDetector

FILE = Path(__file__).resolve()
ROOT = FILE.parents[0]  # root directory
if str(ROOT) not in sys.path:
    sys.path.append(str(ROOT))  # add ROOT to PATH
ROOT = Path(os.path.relpath(ROOT, Path.cwd()))  # relative

import val  # for end-of-epoch mAP
from models.experimental import attempt_load
from models.yolo import Model
from utils.autoanchor import check_anchors
from utils.autobatch import check_train_batch_size
from utils.callbacks import Callbacks
from utils.datasets_mfa import create_dataloader
from utils.downloads import attempt_download
from utils.general import (LOGGER, NCOLS, check_dataset, check_file, check_git_status, check_img_size,
                           check_requirements, check_suffix, check_yaml, colorstr, get_latest_run, increment_path,
                           init_seeds, intersect_dicts, labels_to_class_weights, labels_to_image_weights, methods,
                           one_cycle, print_args, print_mutation, strip_optimizer)
from utils.loggers import Loggers
from utils.loggers.wandb.wandb_utils import check_wandb_resume 
from utils.metrics import fitness
from utils.plots import plot_evolve, plot_labels
from utils.torch_utils import EarlyStopping, ModelEMA, de_parallel, select_device, torch_distributed_zero_first

import neural_renderer

LOCAL_RANK = int(os.getenv('LOCAL_RANK', -1))  # https://pytorch.org/docs/stable/elastic/run.html
RANK = int(os.getenv('RANK', -1))
WORLD_SIZE = int(os.getenv('WORLD_SIZE', 1))

target_layers = ['model_15_act', 'model_22_act', 'model_27_1_cv2_act'] 

def parse_opt(known=False):
    parser = argparse.ArgumentParser()
    parser.add_argument('--weights', type=str, default=ROOT / 'yolov3.pt', help='initial weights path')
    parser.add_argument('--cfg', type=str, default='', help='model.yaml path')
    parser.add_argument('--data', type=str, default = 'data/carla.yaml', help='dataset.yaml path')
    parser.add_argument('--hyp', type=str, default=ROOT / 'data/hyps/hyp.scratch.yaml', help='hyperparameters path')
    parser.add_argument('--batch-size', type=int, default=1, help='total batch size for all GPUs, -1 for autobatch')
    parser.add_argument('--imgsz', '--img', '--img-size', type=int, default=640, help='train, val image size (pixels)')
    parser.add_argument('--rect', action='store_true', help='rectangular training')
    parser.add_argument('--resume', nargs='?', const=True, default=False, help='resume most recent training')
    parser.add_argument('--nosave', action='store_true', help='only save final checkpoint')
    parser.add_argument('--noval', action='store_true', help='only validate final epoch')
    parser.add_argument('--noautoanchor', action='store_true', help='disable autoanchor check')
    parser.add_argument('--evolve', type=int, nargs='?', const=300, help='evolve hyperparameters for x generations')
    parser.add_argument('--bucket', type=str, default='', help='gsutil bucket')
    parser.add_argument('--cache', type=str, nargs='?', const='ram', help='--cache images in "ram" (default) or "disk"')
    parser.add_argument('--image-weights', action='store_true', help='use weighted image selection for training')
    parser.add_argument('--multi-scale', action='store_true', help='vary img-size +/- 50%%')
    parser.add_argument('--single-cls', action='store_true', help='train multi-class data as single-class')
    parser.add_argument('--adam', action='store_true', help='use torch.optim.Adam() optimizer')
    parser.add_argument('--sync-bn', action='store_true', help='use SyncBatchNorm, only available in DDP mode')
    parser.add_argument('--workers', type=int, default=8, help='max dataloader workers (per RANK in DDP mode)')
    parser.add_argument('--project', default=ROOT / 'runs/train', help='save to project/name')
    parser.add_argument('--name', default='exp', help='save to project/name')
    parser.add_argument('--exist-ok', action='store_true', help='existing project/name ok, do not increment')
    parser.add_argument('--quad', action='store_true', help='quad dataloader')
    parser.add_argument('--linear-lr', action='store_true', help='linear LR')
    parser.add_argument('--label-smoothing', type=float, default=0.0, help='Label smoothing epsilon')
    parser.add_argument('--patience', type=int, default=100, help='EarlyStopping patience (epochs without improvement)')
    parser.add_argument('--freeze', type=int, default=0, help='Number of layers to freeze. backbone=10, all=24')
    parser.add_argument('--save-period', type=int, default=-1, help='Save checkpoint every x epochs (disabled if < 1)')
    parser.add_argument('--local_rank', type=int, default=-1, help='DDP parameter, do not modify')
    # Weights & Biases arguments
    parser.add_argument('--entity', default=None, help='W&B: Entity')
    parser.add_argument('--upload_dataset', action='store_true', help='W&B: Upload dataset as artifact table')
    parser.add_argument('--bbox_interval', type=int, default=-1, help='W&B: Set bounding-box image logging interval')
    parser.add_argument('--artifact_alias', type=str, default='latest', help='W&B: Version of dataset artifact to use')

    # ------------------------------------#  
    parser.add_argument('--lr', type=float, default=0.01, help='learning rate for texture_param')
    parser.add_argument('--obj_file', type=str, default='carassets/audi_et_te.obj', help='3d car model obj')
    parser.add_argument('--faces', type=str, default='carassets/exterior_face.txt',
                        help='exterior_face file  (exterior_face, all_faces)')
    parser.add_argument('--datapath', type=str, default='/home/chenwen/Dataset/Carla/', help='data path')
    parser.add_argument('--device', default='0', help='cuda device, i.e. 0 or 0,1,2,3 or cpu') 
    parser.add_argument("--t1", type=float, default=0.0001, help='loss_smooth weights')   
    parser.add_argument("--t2", type=float, default=0.25, help='The threshold of the NMS')   
    parser.add_argument("--experiment", type=str, help='logs name')  # loss_smooth
    parser.add_argument('--epochs', type=int, default=5)
    opt = parser.parse_args()

    opt = parser.parse_known_args()[0] if known else parser.parse_args()
    return opt

def loss_smooth(img, mask):
    # 平滑损失 ==> 使得在边界的对抗性扰动不那么突兀，更加平滑
    # [1,3,223,223]
    s1 = torch.pow(img[:, :, 1:, :-1] - img[:, :, :-1, :-1], 2)
    s2 = torch.pow(img[:, :, :-1, 1:] - img[:, :, :-1, :-1], 2)
    # [3,223,223]
    mask = mask[:, :-1, :-1]
    mask = mask.unsqueeze(1)
    return T1 * torch.sum(mask * (s1 + s2))

def cal_texture(texture_param, texture_origin, texture_mask, texture_content=None, CONTENT=False,):
    # 计算纹理
    if CONTENT:
        textures = 0.5 * (torch.nn.Tanh()(texture_content) + 1)
    else:
        textures = 0.5 * (torch.nn.Tanh()(texture_param) + 1)
    return texture_origin * (1 - texture_mask) + texture_mask * textures

def train(hyp, opt, device):
    save_dir, epochs, batch_size, weights, single_cls, evolve, data, cfg, resume, noval, nosave, workers, freeze, = \
        Path(opt.save_dir), opt.epochs, opt.batch_size, opt.weights, opt.single_cls, opt.evolve, opt.data, opt.cfg, \
        opt.resume, opt.noval, opt.nosave, opt.workers, opt.freeze
    # ---------------------------------#
    # -------Load 3D model-------------#
    # ---------------------------------#
    texture_size = 6
    vertices, faces, texture_origin = neural_renderer.load_obj(filename_obj=opt.obj_file, texture_size=texture_size,
                                                               load_texture=True)

    texture_param = np.random.random((1, faces.shape[0], texture_size, texture_size, texture_size, 3)).astype('float32')
    # texture_param = np.zeros((1, faces.shape[0], texture_size, texture_size, texture_size, 3)).astype('float32')
    texture_param = torch.autograd.Variable(torch.from_numpy(texture_param).to(device), requires_grad=True)
    
    # load face points
    texture_mask = np.zeros((faces.shape[0], texture_size, texture_size, texture_size, 3), 'int8')
    with open(opt.faces, 'r') as f:
        face_ids = f.readlines()
        for face_id in face_ids:
            if face_id != '\n':
                texture_mask[int(face_id) - 1, :, :, :,
                :] = 1  # adversarial perturbation only allow painted on specific areas
    texture_mask = torch.from_numpy(texture_mask).to(device).unsqueeze(0)
    mask_dir = os.path.join(opt.datapath, 'masks/')

    # ---------------------------------#
    # -------Yolo-v3 setting-----------#
    # ---------------------------------# 
    # Directories
    w = save_dir / 'weights'  # weights dir
    (w.parent if evolve else w).mkdir(parents=True, exist_ok=True)  # make dir 

    # Hyperparameters
    if isinstance(hyp, str):
        with open(hyp, errors='ignore') as f:
            hyp = yaml.safe_load(f)  # load hyps dict
    LOGGER.info(colorstr('hyperparameters: ') + ', '.join(f'{k}={v}' for k, v in hyp.items()))

    # Save run settings
    with open(save_dir / 'hyp.yaml', 'w') as f:
        yaml.safe_dump(hyp, f, sort_keys=False)
    with open(save_dir / 'opt.yaml', 'w') as f:
        yaml.safe_dump(vars(opt), f, sort_keys=False)
    data_dict = None

    # Loggers
    if RANK in [-1, 0]:
        loggers = Loggers(save_dir, weights, opt, hyp, LOGGER)  # loggers instance
        if loggers.wandb:
            data_dict = loggers.wandb.data_dict
            if resume:
                weights, epochs, hyp = opt.weights, opt.epochs, opt.hyp

    # Config
    cuda = device.type != 'cpu'
    with torch_distributed_zero_first(LOCAL_RANK):
        data_dict = data_dict or check_dataset(data)  # check if None
    train_path, val_path = data_dict['train'], data_dict['val']
    nc = 1 if single_cls else int(data_dict['nc'])  # number of classes
    names = ['item'] if single_cls and len(data_dict['names']) != 1 else data_dict['names']  # class names
    assert len(names) == nc, f'{len(names)} names found for nc={nc} dataset in {data}'  # check

    # Image size
    gs = 32 # max(int(model.stride.max()), 32)  # grid size (max stride)
    imgsz = check_img_size(opt.imgsz, gs, floor=gs * 2)  # verify imgsz is gs-multiple

    # Optimizer
    optimizer = torch.optim.Adam([texture_param], lr=opt.lr) 
    scheduler = torch.optim.lr_scheduler.StepLR(optimizer, step_size=5, gamma = 0.5)

    # ---------------------------------#
    # -------Load dataset-------------#
    # ---------------------------------#
    dataloader, dataset = create_dataloader(train_path, imgsz, batch_size // WORLD_SIZE, gs, 
                                                faces, texture_size, vertices, opt,
                                                single_cls, hyp=hyp, augment=True, cache=opt.cache, rect=opt.rect, rank=LOCAL_RANK,
                                                workers=workers, image_weights=opt.image_weights, quad=opt.quad,
                                                prefix=colorstr('train: '), shuffle=False,
                                                mask_dir=mask_dir, ret_mask=True)
     
    nb = len(dataloader)  # number of batches
    if cuda and RANK != -1:
        model = DDP(model, device_ids=[LOCAL_RANK], output_device=LOCAL_RANK)

    # Model parameters
    nl = 3 # de_parallel(model).model[-1].nl  # number of detection layers (to scale hyps)
    hyp['box'] *= 3 / nl  # scale to layers
    hyp['cls'] *= nc / 80 * 3 / nl  # scale to classes and layers
    hyp['obj'] *= (imgsz / 640) ** 2 * 3 / nl  # scale to image size and layers

    # Start training  
    input_size = (imgsz, imgsz)
    Detector = YOLOV3ObjectDetector(opt.weights, device, input_size, names, confidence=opt.t2)
    saliency_method = YOLOV3Attention(Detector, target_layers, device, input_size)  
    # ---------------------------------#
    # ------------Training-------------#
    # ---------------------------------#
    for epoch in range(1, epochs+1):  # epoch ------------------------------------------------------------------
        pbar = enumerate(dataloader)
        textures = cal_texture(texture_param, texture_origin, texture_mask)
        dataset.set_textures(textures)
        LOGGER.info(('\n' + '%10s' * 4) % ('Epoch', 'smooth', 'adv', 'loss'))
        if RANK in [-1, 0]:
            pbar = tqdm(pbar, total=nb)  # progress bar
        mloss = torch.zeros(3, device=device)  # mean losses 
        for i, (imgs, texture_img, masks, targets, paths, _) in pbar:  # batch -------------------------------------------------------------
            imgs = imgs.to(device, non_blocking=True).float() / 255.0  # uint8 to float32, 0-255 to 0.0-1.0

            # Forward
            saliencys, inf_result, train_out, feature, L_adv = saliency_method(imgs)  # forward

            # compute loss 
            L_smooth = loss_smooth(texture_img, masks).unsqueeze(dim=0) 
            L_adv = L_adv.unsqueeze(dim=0) 
            loss =  L_smooth + L_adv
            
            # Backward
            optimizer.zero_grad()
            loss.backward() 
            optimizer.step()  
            loss_items = torch.cat((L_smooth, L_adv, loss)).detach()
         
            # try:
            #     for mask in saliencys:
            #         mask = mask.squeeze(0).mul(255).add_(0.5).clamp_(0, 255).permute(1, 2, 0).detach().cpu().numpy().astype(np.uint8)
            #         heatmap = cv2.applyColorMap(mask, cv2.COLORMAP_JET)
            #         Image.fromarray(cv2.cvtColor(heatmap, cv2.COLOR_BGR2RGB)).save(os.path.join(log_dir, 'hotmap.png')) 
            #     Image.fromarray(np.transpose(255 * imgs.data.cpu().numpy()[0], (1, 2, 0)).astype('uint8')).save(
            #         os.path.join(log_dir, 'test_total.png'))
            #     Image.fromarray(
            #         (255 * texture_img).data.cpu().numpy()[0].transpose((1, 2, 0)).astype('uint8')).save(
            #         os.path.join(log_dir, 'texture2.png')) 
            # except:
            #     pass

            if RANK in [-1, 0]:
                mloss = (mloss * i + loss_items) / (i + 1)  # update mean losses 
                s = ('%10s' + '%10.4g' * 3) % ('%g/%g' % (epoch, epochs), *mloss) 
                pbar.set_description(s)

            # update texture_param
            textures = cal_texture(texture_param, texture_origin, texture_mask)
            dataset.set_textures(textures)

        # end epoch ----------------------------------------------------------------------------------------------------
        np.save(os.path.join(log_dir, f'texture_{epoch}.npy'), texture_param.data.cpu().numpy())
    # end training
        
    np.save(os.path.join(log_dir, 'texture.npy'), texture_param.data.cpu().numpy())
    
    torch.cuda.empty_cache() 

log_dir = ""
def make_log_dir(logs_name):
    global log_dir
    dir_name = 'logs/' + logs_name
    print(dir_name)
    if not (os.path.exists(dir_name)):
        os.makedirs(dir_name)
    log_dir = dir_name

if __name__ == "__main__":
    opt = parse_opt()
    T1 = opt.t1
    T2 = opt.t2
    LR = opt.lr
    logs_name = opt.experiment 
    make_log_dir(logs_name)
    
    # Checks
    if RANK in [-1, 0]:
        print_args(FILE.stem, opt)
        check_git_status()
        check_requirements(exclude=['thop'])
    
    # Resume
    if opt.resume and not check_wandb_resume(opt) and not opt.evolve:  # resume an interrupted run
        ckpt = opt.resume if isinstance(opt.resume, str) else get_latest_run()  # specified or most recent path
        assert os.path.isfile(ckpt), 'ERROR: --resume checkpoint does not exist'
        with open(Path(ckpt).parent.parent / 'opt.yaml', errors='ignore') as f:
            opt = argparse.Namespace(**yaml.safe_load(f))  # replace
        opt.cfg, opt.weights, opt.resume = '', ckpt, True  # reinstate
        LOGGER.info(f'Resuming training from {ckpt}')
    else:
        opt.data, opt.cfg, opt.hyp, opt.weights, opt.project = \
            check_file(opt.data), check_yaml(opt.cfg), check_yaml(opt.hyp), str(opt.weights), str(opt.project)  # checks
        assert len(opt.cfg) or len(opt.weights), 'either --cfg or --weights must be specified'
        if opt.evolve:
            opt.project = str(ROOT / 'runs/evolve')
            opt.exist_ok, opt.resume = opt.resume, False  # pass resume to exist_ok and disable resume
        opt.save_dir = str(increment_path(Path(opt.project) / opt.name, exist_ok=opt.exist_ok))

    # DDP mode
    device = select_device(opt.device, batch_size=opt.batch_size)
    if LOCAL_RANK != -1:
        assert torch.cuda.device_count() > LOCAL_RANK, 'insufficient CUDA devices for DDP command'
        assert opt.batch_size % WORLD_SIZE == 0, '--batch-size must be multiple of CUDA device count'
        assert not opt.image_weights, '--image-weights argument is not compatible with DDP training'
        assert not opt.evolve, '--evolve argument is not compatible with DDP training'
        torch.cuda.set_device(LOCAL_RANK)
        device = torch.device('cuda', LOCAL_RANK)
        dist.init_process_group(backend="nccl" if dist.is_nccl_available() else "gloo")

    train(opt.hyp, opt, device)
