#!/usr/bin/env python
# coding=utf-8
# Copyright 2020 The HuggingFace Inc. team. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
""" Finetuning the library models for sequence classification on GLUE."""
# You can also adapt this script on your own text classification task. Pointers for this are left as comments.

import logging
import os
import random
import sys
import json
from dataclasses import dataclass, field, asdict
from typing import Optional
from pathlib import Path

import datasets
import numpy as np
from datasets import load_dataset

import torch
import evaluate
import transformers
from transformers import (
    AutoConfig,
    AutoTokenizer,
    AutoModel,
    DataCollatorWithPadding,
    EvalPrediction,
    HfArgumentParser,
    PretrainedConfig,
    Trainer,
    TrainingArguments,
    default_data_collator,
    set_seed,
)
from transformers.trainer_utils import get_last_checkpoint
from transformers.utils import check_min_version, send_example_telemetry
from transformers.utils.versions import require_version
from transformers import DapForSSL

from transformers.models.perceiver_dap.utils import KillCallback, EMATeacherCallback, WandbTextReconstructionVisualizer, GLUECallback, EMADistanceCallback, BioCallback
# Will error if the minimal version of Transformers is not installed. Remove at your own risks.
check_min_version("4.26.0.dev0")

require_version("datasets>=1.8.0", "To fix: pip install -r examples/pytorch/text-classification/requirements.txt")

logger = logging.getLogger(__name__)


@dataclass
class DataTrainingArguments:
    """
    Arguments pertaining to what data we are going to input our model for training and eval.

    Using `HfArgumentParser` we can turn this class
    into argparse arguments to be able to specify them on
    the command line.
    """
    dataset_name: Optional[str] = field(
        default=None, metadata={"help": "The name of the dataset to use (via the datasets library)."}
    )
    input_key: Optional[str] = field(
        default="text",
    )
    data_dir: Optional[str] = field(
        default=None,
    )
    dataset_config_name: Optional[str] = field(
        default=None, metadata={"help": "The configuration name of the dataset to use (via the datasets library)."}
    )
    max_seq_length: int = field(
        default=2048,
        metadata={
            "help": (
                "The maximum total input sequence length after tokenization. Sequences longer "
                "than this will be truncated, sequences shorter will be padded."
            )
        },
    )
    min_seq_length: int = field(
        default=1,
    )
    overwrite_cache: bool = field(
        default=False, metadata={"help": "Overwrite the cached preprocessed datasets or not."}
    )
    pad_to_max_length: bool = field(
        default=True,
        metadata={
            "help": (
                "Whether to pad all samples to `max_seq_length`. "
                "If False, will pad the samples dynamically when batching to the maximum length in the batch."
            )
        },
    )
    max_train_samples: Optional[int] = field(
        default=None,
        metadata={
            "help": (
                "For debugging purposes or quicker training, truncate the number of training examples to this "
                "value if set."
            )
        },
    )
    max_eval_samples: Optional[int] = field(
        default=None,
        metadata={
            "help": (
                "For debugging purposes or quicker training, truncate the number of evaluation examples to this "
                "value if set."
            )
        },
    )
    max_predict_samples: Optional[int] = field(
        default=None,
        metadata={
            "help": (
                "For debugging purposes or quicker training, truncate the number of prediction examples to this "
                "value if set."
            )
        },
    )
    train_split: Optional[str] = field(
        default="train",
    )
    eval_split: Optional[str] = field(
        default="validation",
    )
    train_file: Optional[str] = field(
        default=None, metadata={"help": "A csv or a json file containing the training data."}
    )
    validation_file: Optional[str] = field(
        default=None, metadata={"help": "A csv or a json file containing the validation data."}
    )
    test_file: Optional[str] = field(default=None, metadata={"help": "A csv or a json file containing the test data."})
    
    merge_method: Optional[str] = field(default=None, metadata={"help": "method for merging documents"})

    dynamic_padding: bool = field(default=False, metadata={"help": "If true will pad for each batch. Useful for reducing dataloading times"})
    no_pretokenization: bool = field(default=False, metadata={"help": "If true will tokenize on the fly to save RAM at the cost of runtime"})
@dataclass
class ModelArguments:
    """
    Arguments pertaining to which model/config/tokenizer we are going to fine-tune from.
    """

    model_name_or_path: str = field(
        metadata={"help": "Path to pretrained model or model identifier from huggingface.co/models"}
    )
    config_name: Optional[str] = field(
        default=None, metadata={"help": "Pretrained config name or path if not the same as model_name"}
    )
    tokenizer_name: Optional[str] = field(
        default=None, metadata={"help": "Pretrained tokenizer name or path if not the same as model_name"}
    )
    cache_dir: Optional[str] = field(
        default=None,
        metadata={"help": "Where do you want to store the pretrained models downloaded from huggingface.co"},
    )
    use_fast_tokenizer: bool = field(
        default=True,
        metadata={"help": "Whether to use one of the fast tokenizer (backed by the tokenizers library) or not."},
    )
    model_revision: str = field(
        default="main",
        metadata={"help": "The specific model version to use (can be a branch name, tag name or commit id)."},
    )
    use_auth_token: bool = field(
        default=False,
        metadata={
            "help": (
                "Will use the token generated when running `huggingface-cli login` (necessary to use this script "
                "with private models)."
            )
        },
    )
    ignore_mismatched_sizes: bool = field(
        default=False,
        metadata={"help": "Will enable to load a pretrained model whose head dimensions are different."},
    )
    masking_ratio: float = field(
        default=None,
    )

@dataclass
class CallbackArguments:   
    kill_steps: Optional[int] = field(
        default=None, metadata={"help": "Will kill run at num steps. Used for hyp search."}
    )
    
    ema_callback: Optional[bool] = field(
        default=False,
        metadata={
            "help": (
                "ema_callback"
            )
        },
    )

    reconstruction_visualization_callback: Optional[str] = field(
        default="text",
        metadata={
            "help": (
                "reconstruction_visualization_callback"
            )
        },
    )

    rvc_interval: Optional[int] = field(
        default=1000,
        metadata={
            "help": (
                "reconstruction_visualization_callback"
            )
        },
    )

    rvc_num_samples: Optional[int] = field(
        default=5,
        metadata={
            "help": (
                "reconstruction_visualization_callback"
            )
        },
    )
    glue_interval: Optional[int] = field(
        default=-1,
    )
    configs_path: Optional[str] = field(
        default=None,
    )
    bio_interval: Optional[int] = field(
        default=-1,
    )
    ema_distance_callback_interval: Optional[int] = field(
        default=0,
    )
def main():
    # See all possible arguments in src/transformers/training_args.py
    # or by passing the --help flag to this script.
    # We now keep distinct sets of args, for a cleaner separation of concerns.

    parser = HfArgumentParser((ModelArguments, DataTrainingArguments, TrainingArguments, CallbackArguments))
    if len(sys.argv) == 2 and sys.argv[1].endswith(".json"):
        # If we pass only one argument to the script and it's the path to a json file,
        # let's parse it to get our arguments.
        model_args, data_args, training_args, callback_args = parser.parse_json_file(json_file=os.path.abspath(sys.argv[1]))
    else:
        model_args, data_args, training_args, callback_args = parser.parse_args_into_dataclasses()


    all_args_dict = {}
    for args in (model_args, data_args, training_args):
        for k, v in asdict(args).items():
            all_args_dict[k] = v
    Path(training_args.output_dir).mkdir(parents=True, exist_ok=True)

    with open(os.path.join(training_args.output_dir, "all_args.json"), "w") as f:
        json.dump(all_args_dict, f, indent=4)
    # Sending telemetry. Tracking the example usage helps us better allocate resources to maintain them. The
    # information sent is the one passed as arguments along with your Python/PyTorch versions.
    send_example_telemetry("run_pretraining_discrete", model_args, data_args)

    # Setup logging
    logging.basicConfig(
        format="%(asctime)s - %(levelname)s - %(name)s - %(message)s",
        datefmt="%m/%d/%Y %H:%M:%S",
        handlers=[logging.StreamHandler(sys.stdout)],
    )

    log_level = training_args.get_process_log_level()
    logger.setLevel(log_level)
    datasets.utils.logging.set_verbosity(log_level)
    transformers.utils.logging.set_verbosity(log_level)
    transformers.utils.logging.enable_default_handler()
    transformers.utils.logging.enable_explicit_format()

    # Log on each process the small summary:
    logger.warning(
        f"Process rank: {training_args.local_rank}, device: {training_args.device}, n_gpu: {training_args.n_gpu}"
        + f"distributed training: {bool(training_args.local_rank != -1)}, 16-bits training: {training_args.fp16}"
    )
    logger.info(f"Training/evaluation parameters {training_args}")

    # Detecting last checkpoint.
    last_checkpoint = None
    if os.path.isdir(training_args.output_dir) and training_args.do_train and not training_args.overwrite_output_dir:
        last_checkpoint = get_last_checkpoint(training_args.output_dir)
        if last_checkpoint is None and len(os.listdir(training_args.output_dir)) > 0:
            raise ValueError(
                f"Output directory ({training_args.output_dir}) already exists and is not empty. "
                "Use --overwrite_output_dir to overcome."
            )
        elif last_checkpoint is not None and training_args.resume_from_checkpoint is None:
            logger.info(
                f"Checkpoint detected, resuming training at {last_checkpoint}. To avoid this behavior, change "
                "the `--output_dir` or add `--overwrite_output_dir` to train from scratch."
            )

    # Set seed before initializing model.
    set_seed(training_args.seed)

    # Get the datasets: you can either provide your own CSV/JSON training and evaluation files (see below)
    # or specify a GLUE benchmark task (the dataset will be downloaded automatically from the datasets Hub).
    #
    # For CSV/JSON files, this script will use as labels the column called 'label' and as pair of sentences the
    # sentences in columns called 'sentence1' and 'sentence2' if such column exists or the first two columns not named
    # label if at least two columns are provided.
    #
    # If the CSVs/JSONs contain only one non-label column, the script does single sentence classification on this
    # single column. You can easily tweak this behavior (see below)
    #
    # In distributed training, the load_dataset function guarantee that only one local process can concurrently
    # download the dataset.
    if data_args.dataset_name is not None:
        # Downloading and loading a dataset from the hub.
        if training_args.do_train:
            train_dataset = load_dataset(
                data_args.dataset_name,
                data_args.dataset_config_name,
                cache_dir=model_args.cache_dir,
                use_auth_token=True if model_args.use_auth_token else None,
                split=data_args.train_split
            )
            if "label" in train_dataset.column_names:
                train_dataset = train_dataset.remove_columns(["label"])
        if training_args.do_eval:
            eval_dataset = load_dataset(
                data_args.dataset_name,
                data_args.dataset_config_name,
                cache_dir=model_args.cache_dir,
                use_auth_token=True if model_args.use_auth_token else None,
                split=data_args.eval_split
            )
            if "label" in eval_dataset.column_names:
                eval_dataset = eval_dataset.remove_columns("label")
    
    # Load pretrained model and tokenizer
    #
    # In distributed training, the .from_pretrained methods guarantee that only one local process can concurrently
    # download model & vocab.
    config = AutoConfig.from_pretrained(
        model_args.config_name if model_args.config_name else model_args.model_name_or_path,
        cache_dir=model_args.cache_dir,
        revision=model_args.model_revision,
        use_auth_token=True if model_args.use_auth_token else None,
    )
    if model_args.masking_ratio is not None:
        if config.cross_train_noising_args is not None:
            config.cross_train_noising_args[0][1]["masking_ratio"] = model_args.masking_ratio

        elif config.train_transform_args is not None:
            config.train_transform_args[0][1]["masking_ratio"] = model_args.masking_ratio
    
    tokenizer = AutoTokenizer.from_pretrained(
        model_args.tokenizer_name if model_args.tokenizer_name else model_args.model_name_or_path,
        cache_dir=model_args.cache_dir,
        use_fast=model_args.use_fast_tokenizer,
        revision=model_args.model_revision,
        use_auth_token=True if model_args.use_auth_token else None,
    )
    if os.path.exists(os.path.join(model_args.model_name_or_path, "pytorch_model.bin")):
        model = DapForSSL.from_pretrained(
            model_args.model_name_or_path,
            from_tf=bool(".ckpt" in model_args.model_name_or_path),
            config=config,
            cache_dir=model_args.cache_dir,
            revision=model_args.model_revision,
            use_auth_token=True if model_args.use_auth_token else None,
            ignore_mismatched_sizes=model_args.ignore_mismatched_sizes,
        )
    else:
        model = DapForSSL._from_config(config)
    # Padding strategy
    if data_args.pad_to_max_length and not data_args.dynamic_padding:
        padding = "max_length"
    else:
        # We will pad later, dynamically at batch creation, to the max sequence length in each batch
        padding = False

    
    if data_args.max_seq_length > tokenizer.model_max_length:
        logger.warning(
            f"The max_seq_length passed ({data_args.max_seq_length}) is larger than the maximum length for the"
            f"model ({tokenizer.model_max_length}). Using max_seq_length={tokenizer.model_max_length}."
        )
    max_seq_length = min(data_args.max_seq_length, tokenizer.model_max_length)

    import multiprocessing
    placeholder_token = u"\u0003"
    placeholder_token_id = tokenizer.convert_tokens_to_ids(placeholder_token)
    sep_token_id = tokenizer.sep_token_id
    def merge_documents(sample, idx, dataset, min_start_length_doc=1, max_len=2048, input_key="text", add_sep_placeholder=True):
        if len(sample[input_key]) < min_start_length_doc:
            return sample
        curr_text = sample[input_key]
        last_blank = False
        curr_idx = idx + 1
        while curr_idx < len(dataset):
            new_text = dataset[curr_idx][input_key]
            if len(new_text) == 0:
                if last_blank:
                    break
                last_blank = True
                curr_idx += 1
                continue
            else:
                last_blank = False
            if len(curr_text) + len(new_text) >= max_len - 2:
                break
            if add_sep_placeholder:
                curr_text += placeholder_token
            curr_text += dataset[curr_idx][input_key]
            curr_idx += 1
        sample[input_key] = curr_text
        return sample

    def replace_placeholder(sample):
        sample["input_ids"] = [i if i != placeholder_token_id else sep_token_id for i in sample["input_ids"]]
        return sample
    
    if data_args.merge_method is not None:
        if training_args.do_train:
            train_dataset = train_dataset.map(
                merge_documents,
                fn_kwargs={"dataset": train_dataset, "max_len": max_seq_length, "input_key": data_args.input_key, "add_sep_placeholder": data_args.merge_method == "sep"},
                with_indices=True,
                num_proc=multiprocessing.cpu_count()
            )
        if training_args.do_eval:
            eval_dataset = eval_dataset.map(
                merge_documents,
                fn_kwargs={"dataset": eval_dataset, "max_len": max_seq_length, "input_key": data_args.input_key, "add_sep_placeholder": data_args.merge_method == "sep"},
                with_indices=True,
                num_proc=multiprocessing.cpu_count()
            )                

    def tokenize_function(examples):
        # Tokenize the texts
        return tokenizer(examples[data_args.input_key], padding=padding, max_length=max_seq_length, truncation=True)
        
    if training_args.do_train:
        train_dataset = train_dataset.filter(lambda sample: len(sample[data_args.input_key]) > data_args.min_seq_length)
    if training_args.do_eval:
        eval_dataset = eval_dataset.filter(lambda sample: len(sample[data_args.input_key]) > data_args.min_seq_length)
    
    if data_args.no_pretokenization:
        if training_args.do_train:
            train_dataset.set_transform(tokenize_function)
        if training_args.do_eval:
            eval_dataset.set_transform(tokenize_function)
    else:
        with training_args.main_process_first(desc="dataset map pre-processing"):
            if training_args.do_train:
                train_dataset = train_dataset.map(
                    tokenize_function,
                    batched=True,
                    load_from_cache_file=not data_args.overwrite_cache,
                    desc="Running tokenizer on dataset",
                    num_proc=multiprocessing.cpu_count(),
                )
            if training_args.do_eval:
                eval_dataset = eval_dataset.map(
                    tokenize_function,
                    batched=True,
                    load_from_cache_file=not data_args.overwrite_cache,
                    desc="Running tokenizer on dataset",
                    num_proc=multiprocessing.cpu_count(),
                )
        
    if data_args.merge_method == "sep":
        if training_args.do_train:
            train_dataset = train_dataset.map(replace_placeholder, num_proc=multiprocessing.cpu_count())
        if training_args.do_eval:
            eval_dataset = eval_dataset.map(replace_placeholder, num_proc=multiprocessing.cpu_count())
    
    print("finished processing dataset")
    if training_args.do_train and data_args.max_train_samples is not None:
        max_train_samples = min(len(train_dataset), data_args.max_train_samples)
        train_dataset = train_dataset.select(range(max_train_samples))

    if training_args.do_eval and data_args.max_eval_samples is not None:
        max_eval_samples = min(len(eval_dataset), data_args.max_eval_samples)
        eval_dataset = eval_dataset.select(range(max_eval_samples))

    # Log a few random samples from the training set:
    #if training_args.do_train:
    #    for index in random.sample(range(len(train_dataset)), 3):
    #        logger.info(f"Sample {index} of the training set: {train_dataset[index]}.")

    # You can define your custom compute_metrics function. It takes an `EvalPrediction` object (a namedtuple with a
    # predictions and label_ids field) and has to return a dictionary string to float.

    # Data collator will default to DataCollatorWithPadding when the tokenizer is passed to Trainer, so we change it if
    # we already did the padding.
    if data_args.pad_to_max_length and not data_args.dynamic_padding:
        data_collator = default_data_collator
    else:
        data_collator = DataCollatorWithPadding(tokenizer, padding="max_length", max_length=max_seq_length)
    #print(train_dataset, eval_dataset)
    if data_args.no_pretokenization:
        training_args.remove_unused_columns = False
    # Initialize our Trainer
    trainer = Trainer(
        model=model,
        args=training_args,
        train_dataset=train_dataset if training_args.do_train else None,
        eval_dataset=eval_dataset if training_args.do_eval else None,
        compute_metrics=None,
        tokenizer=tokenizer,
        data_collator=data_collator,
    )
    if callback_args.ema_callback:
        trainer.add_callback(EMATeacherCallback(resume=training_args.resume_from_checkpoint is not None))
    if callback_args.kill_steps is not None:
        trainer.add_callback(KillCallback(callback_args.kill_steps))
    
    if callback_args.reconstruction_visualization_callback == "text":
        trainer.add_callback(WandbTextReconstructionVisualizer(
            log_step_interval=callback_args.rvc_interval,
            num_samples=callback_args.rvc_num_samples
        ))
    if callback_args.ema_distance_callback_interval > 0:
        trainer.add_callback(EMADistanceCallback(
            log_step_interval=callback_args.ema_distance_callback_interval,
        ))
    #if callback_args.cc_script_path is not None:
    #    trainer.add_callback(CommandCallback(run_step_interval=callback_args.cc_interval, script_path=callback_args.cc_script_path))
    if callback_args.glue_interval > 0:
        trainer.add_callback(GLUECallback(run_step_interval=callback_args.glue_interval, configs_dir=callback_args.configs_path))
    
    if callback_args.bio_interval > 0:
        trainer.add_callback(BioCallback(run_step_interval=callback_args.bio_interval, configs_dir=callback_args.configs_path))
    # Training
    if training_args.do_train:
        checkpoint = None
        if training_args.resume_from_checkpoint is not None:
            checkpoint = training_args.resume_from_checkpoint
        elif last_checkpoint is not None:
            checkpoint = last_checkpoint
        train_result = trainer.train(resume_from_checkpoint=checkpoint)
        metrics = train_result.metrics
        max_train_samples = (
            data_args.max_train_samples if data_args.max_train_samples is not None else len(train_dataset)
        )
        metrics["train_samples"] = min(max_train_samples, len(train_dataset))

        trainer.save_model()  # Saves the tokenizer too for easy upload

        trainer.log_metrics("train", metrics)
        trainer.save_metrics("train", metrics)
        trainer.save_state()

    # Evaluation
    if training_args.do_eval:
        logger.info("*** Evaluate ***")

        # Loop to handle MNLI double evaluation (matched, mis-matched)
        tasks = [data_args.dataset_name]
        eval_datasets = [eval_dataset]

        for eval_dataset, task in zip(eval_datasets, tasks):
            metrics = trainer.evaluate(eval_dataset=eval_dataset)

            max_eval_samples = (
                data_args.max_eval_samples if data_args.max_eval_samples is not None else len(eval_dataset)
            )
            metrics["eval_samples"] = min(max_eval_samples, len(eval_dataset))

            trainer.log_metrics("eval", metrics)

    kwargs = {}
    if data_args.dataset_name is not None:
        kwargs["dataset_tags"] = data_args.dataset_name

    if training_args.push_to_hub:
        trainer.push_to_hub(**kwargs)
    else:
        trainer.create_model_card(**kwargs)


def _mp_fn(index):
    # For xla_spawn (TPUs)
    main()


if __name__ == "__main__":
    main()
