# coding=utf-8
# Copyright Deepmind and The HuggingFace Inc. team. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
""" Dap model configuration"""

from collections import OrderedDict
from typing import Any, Mapping, Optional, Union

from ...configuration_utils import PretrainedConfig
from ...feature_extraction_utils import FeatureExtractionMixin
from ...onnx import OnnxConfig
from ...onnx.utils import compute_effective_axis_dimension
from ...tokenization_utils_base import PreTrainedTokenizerBase
from ...utils import TensorType, logging


logger = logging.get_logger(__name__)

DAP_PRETRAINED_CONFIG_ARCHIVE_MAP = {
    
}


class DapConfig(PretrainedConfig):
    r"""
    This is the configuration class to store the configuration of a [`PerceiverModel`]. It is used to instantiate an
    Perceiver model according to the specified arguments, defining the model architecture. Instantiating a
    configuration with the defaults will yield a similar configuration to that of the Perceiver
    [deepmind/language-perceiver](https://huggingface.co/deepmind/language-perceiver) architecture.

    Configuration objects inherit from [`PretrainedConfig`] and can be used to control the model outputs. Read the
    documentation from [`PretrainedConfig`] for more information.

    Args:
        num_latents (`int`, *optional*, defaults to 256):
            The number of latents.
        latent_channels (`int`, *optional*, defaults to 1280):
            Dimension of the latent embeddings.
        cross_channels (`int`, *optional*, defaults to 768):
            Dimension of the inputs. Should only be provided in case [*PerceiverTextPreprocessor*] is used or no
            preprocessor is provided.
        num_blocks (`int`, *optional*, defaults to 1):
            Number of blocks in the Transformer encoder.
        num_self_attends_per_block (`int`, *optional*, defaults to 26):
            The number of self-attention layers per block.
        num_self_attention_heads (`int`, *optional*, defaults to 8):
            Number of attention heads for each self-attention layer in the Transformer encoder.
        num_cross_attention_heads (`int`, *optional*, defaults to 8):
            Number of attention heads for each cross-attention layer in the Transformer encoder.
        qk_channels (`int`, *optional*):
            Dimension to project the queries + keys before applying attention in the cross-attention and self-attention
            layers of the encoder. Will default to preserving the dimension of the queries if not specified.
        v_channels (`int`, *optional*):
            Dimension to project the values before applying attention in the cross-attention and self-attention layers
            of the encoder. Will default to preserving the dimension of the queries if not specified.
        cross_attention_shape_for_attention (`str`, *optional*, defaults to `'kv'`):
            Dimension to use when downsampling the queries and keys in the cross-attention layer of the encoder.
        self_attention_widening_factor (`int`, *optional*, defaults to 1):
            Dimension of the feed-forward layer in the cross-attention layer of the Transformer encoder.
        cross_attention_widening_factor (`int`, *optional*, defaults to 1):
            Dimension of the feed-forward layer in the self-attention layers of the Transformer encoder.
        hidden_act (`str` or `function`, *optional*, defaults to `"gelu"`):
            The non-linear activation function (function or string) in the encoder and pooler. If string, `"gelu"`,
            `"relu"`, `"selu"` and `"gelu_new"` are supported.
        attention_probs_dropout_prob (`float`, *optional*, defaults to 0.1):
            The dropout ratio for the attention probabilities.
        initializer_range (`float`, *optional*, defaults to 0.02):
            The standard deviation of the truncated_normal_initializer for initializing all weight matrices.
        layer_norm_eps (`float`, *optional*, defaults to 1e-12):
            The epsilon used by the layer normalization layers.
        use_query_residual (`float`, *optional*, defaults to `True`):
            Whether to add a query residual in the cross-attention layer of the encoder.
        vocab_size (`int`, *optional*, defaults to 262):
            Vocabulary size for the masked language modeling model.
        max_position_embeddings (`int`, *optional*, defaults to 2048):
            The maximum sequence length that the masked language modeling model might ever be used with. Typically set
            this to something large just in case (e.g., 512 or 1024 or 2048).

    Example:

    ```python
    >>> from transformers import DapModel, DapConfig

    >>> # Initializing a Perceiver deepmind/language-perceiver style configuration
    >>> configuration = DapConfig()

    >>> # Initializing a model from the deepmind/language-perceiver style configuration
    >>> model = DapModel(configuration)

    >>> # Accessing the model configuration
    >>> configuration = model.config
    ```"""
    model_type = "perceiver_dap"

    def __init__(
        self,
        num_latents=512,
        latent_channels=1024,
        embedded_channels=256,
        num_blocks=1,
        num_self_attends_per_block=6,
        num_self_attention_heads=8,
        num_cross_attention_heads=1,
        qk_channels=None,
        v_channels=None,
        dense_use_bias=False,
        encoder_type="cross_attention",
        encoder_cross_attention_channels=None,
        cross_train_noising_args=None,
        cross_eval_noising_args=None,
        self_attention_widening_factor=1,
        cross_attention_widening_factor=1,
        hidden_act="gelu",
        attention_dropout_prob=0.0,
        hidden_dropout_prob=0.0,
        drop_path_rate=0.0,
        initializer_range=0.02,
        pe_initializer_range=0.02,
        layernorm_eps=1e-12,
        use_query_residual=True,
        max_position_embeddings=50176,
        input_channels=3,
        input_type="continuous",
        project_after_concat=True,
        num_discrete_tokens=262,
        loss_fn="mse",
        use_decoder=False,
        #decoder_type="cross_attention",
        decoder_latent_channels=None,
        decoder_attention_channels=None,
        decoder_heads=1,
        post_decoder_layers=None,
        final_project=True,
        output_channels=3,
        num_outputs=1,
        share_decoder_queries=False,
        share_embedding_weights=False,
        teacher_name="D2VTeacher",
        teacher_args={
            "mask_replace": 3,
            "reconstruction_decoder_args": {
                "query_num_channels": None,
                "qk_channels": None,
                "v_channels": None,
                "num_heads": None,
                "use_query_residual": True,
                #"num_outputs": None,
                #"output_channels": None,
                "share_decoder_queries": True,
                "share_embedding_weights": True,
            },
            "reconstruction_weighted_loss": False,
            "reconstruction_loss_fn": "crossentropy",
            "reconstruction_loss_weight": 1.0,
            "ema_args": {
                "ema_decay_start": 0.998,
                "ema_decay_end": 0.9999,
            },
            "train_transform_args": [
                ("CrossAttentionMasking", {"masking_ratio": 0.5, "query_subsample_ratio": 1.0, "replace": 3}),
            ],
            "eval_transform_args": None,
            "auxiliary_loss_weight": 1.0,
            "auxiliary_loss_fn": "mse",
            "num_layer_target_avg": None,
            "target_normalization_fn": "layernorm"
        },
        **kwargs
    ):
        super().__init__(**kwargs)

        self.num_latents = num_latents
        self.latent_channels = latent_channels
        self.embedded_channels = embedded_channels
        self.num_blocks = num_blocks
        self.num_self_attends_per_block = num_self_attends_per_block
        self.num_self_attention_heads = num_self_attention_heads
        self.num_cross_attention_heads = num_cross_attention_heads
        self.cross_train_noising_args = cross_train_noising_args
        self.cross_eval_noising_args = cross_eval_noising_args
        self.qk_channels = qk_channels
        self.v_channels = v_channels
        self.dense_use_bias = dense_use_bias
        
        self.encoder_type = encoder_type
        self.encoder_cross_attention_channels = encoder_cross_attention_channels if encoder_cross_attention_channels else qk_channels
        self.self_attention_widening_factor = self_attention_widening_factor
        self.cross_attention_widening_factor = cross_attention_widening_factor
        self.hidden_act = hidden_act
        self.attention_dropout_prob = attention_dropout_prob
        self.hidden_dropout_prob = hidden_dropout_prob
        self.drop_path_rate = drop_path_rate
        self.initializer_range = initializer_range
        self.pe_initializer_range = pe_initializer_range
        self.layernorm_eps = layernorm_eps
        self.use_query_residual = use_query_residual
        
        self.input_type = input_type
        self.input_channels = input_channels
        self.max_position_embeddings = max_position_embeddings
        self.num_discrete_tokens = num_discrete_tokens

        self.loss_fn = loss_fn
        self.project_after_concat = project_after_concat

        self.use_decoder = use_decoder
        self.decoder_latent_channels = decoder_latent_channels if decoder_latent_channels else latent_channels
        self.decoder_heads = decoder_heads
        self.share_decoder_queries = share_decoder_queries
        self.share_embedding_weights = share_embedding_weights
        self.post_decoder_layers = post_decoder_layers
        self.final_project = final_project

        #self.decoder_type = decoder_type
        self.decoder_attention_channels = decoder_attention_channels
        
        self.output_channels = output_channels if output_channels else input_channels
        self.num_outputs = num_outputs if num_outputs else max_position_embeddings
        self.layer_norm_eps = 1e-12

        self.teacher_name = teacher_name
        self.teacher_args = teacher_args
    
class DapOnnxConfig(OnnxConfig):
    @property
    def inputs(self) -> Mapping[str, Mapping[int, str]]:
        if self.task == "multiple-choice":
            dynamic_axis = {0: "batch", 1: "choice", 2: "sequence"}
        else:
            dynamic_axis = {0: "batch", 1: "sequence"}
        return OrderedDict(
            [
                ("inputs", dynamic_axis),
                ("attention_mask", dynamic_axis),
            ]
        )

    @property
    def atol_for_validation(self) -> float:
        return 1e-4

    def generate_dummy_inputs(
        self,
        preprocessor: Union["PreTrainedTokenizerBase", "FeatureExtractionMixin"],
        batch_size: int = -1,
        seq_length: int = -1,
        num_choices: int = -1,
        is_pair: bool = False,
        framework: Optional[TensorType] = None,
        num_channels: int = 3,
        image_width: int = 40,
        image_height: int = 40,
    ) -> Mapping[str, Any]:
        # copied from `transformers.onnx.config.OnnxConfig` and slightly altered/simplified

        if isinstance(preprocessor, PreTrainedTokenizerBase):
            # If dynamic axis (-1) we forward with a fixed dimension of 2 samples to avoid optimizations made by ONNX
            batch_size = compute_effective_axis_dimension(
                batch_size, fixed_dimension=OnnxConfig.default_fixed_batch, num_token_to_add=0
            )
            # If dynamic axis (-1) we forward with a fixed dimension of 8 tokens to avoid optimizations made by ONNX
            token_to_add = preprocessor.num_special_tokens_to_add(is_pair)
            seq_length = compute_effective_axis_dimension(
                seq_length, fixed_dimension=OnnxConfig.default_fixed_sequence, num_token_to_add=token_to_add
            )
            # Generate dummy inputs according to compute batch and sequence
            dummy_input = [" ".join(["a"]) * seq_length] * batch_size
            inputs = dict(preprocessor(dummy_input, return_tensors=framework))
            inputs["inputs"] = inputs.pop("input_ids")
            return inputs
        elif isinstance(preprocessor, FeatureExtractionMixin) and preprocessor.model_input_names[0] == "pixel_values":
            # If dynamic axis (-1) we forward with a fixed dimension of 2 samples to avoid optimizations made by ONNX
            batch_size = compute_effective_axis_dimension(batch_size, fixed_dimension=OnnxConfig.default_fixed_batch)
            dummy_input = self._generate_dummy_images(batch_size, num_channels, image_height, image_width)
            inputs = dict(preprocessor(images=dummy_input, return_tensors=framework))
            inputs["inputs"] = inputs.pop("pixel_values")
            return inputs
        else:
            raise ValueError(
                "Unable to generate dummy inputs for the model. Please provide a tokenizer or a preprocessor."
            )
