# filepath: src/qa_pairs_generation/paths/process_single_file_visibility.py
import json, os, numpy as np
from pathlib import Path
from typing import Dict, Optional, List
from src.qa_pairs_generation.utils import generate_qa_pairs_with_subsampling, save_and_info, render_visibility_path, merge_objects_and_openings, _label

# in your runner
from functools import partial
from src.qa_pairs_generation.shortest_path.pathfinder import (
    find_path_room_constrained_blocked,
)

# Preferred labels per room type (used only if room_type is known)
CHOICE_OPTIONS = {
    "kitchens": ["stove", "fridge", "sink", "dishwasher", "door", "window", "table"],
    "living_rooms": ["door", "window", "sofa", "loveseat", "armchair", "coffee_table", "side_table", "tv_stand", "television", "bookshelf", "fireplace"],
    "bedrooms": ["door", "window", "bed", "dresser", "wardrobe", "desk", "bookshelf", "chair"],
}

# Explicit layout_id → (start_label, goal_label) overrides (ONLY those lines that had pairs)
MAP_OVERRIDES: Dict[str, tuple[str, str]] = {
    "102344022_room_0": ("cabinet_2", "chair_2"),
    "102344022_room_5": ("door", "wastafel"),
    "102344022_room_6": ("door", "table_3"),
    "102344022_room_11": ("door", "dresser"),
    "102344022_room_18": ("plant_1", "plant_2"),
    "102344022_room_19": ("fabric", "fridge"),
    "102344049_room_12": ("sofa", "chair_1"),
    "102344094_room_0": ("mirror", "chair"),
    "102344094_room_4": ("door_1", "rod"),
    "102344115_room_1": ("sink", "door_1"),
    "102344193_room_0": ("bench", "plant_1"),
    "102344193_room_2": ("laptop", "door"),
    "102344250_room_2": ("bench", "door_1"),
    "102344250_room_3": ("bin", "door"),
    "102344250_room_6": ("bookcase", "door_2"),
    "102344250_room_11": ("chest", "door_3"),
    "102344280_room_7": ("stool", "chest_2"),
    "102344280_room_10": ("window_1", "shelf_2"),
    "102344280_room_11": ("door", "mirror_2"),
    "102344307_room_0": ("bookcase_1", "armchair"),
    "102344307_room_3": ("door", "bathhtub"),
    "102344307_room_14": ("furniture_1", "cabinet_8"),
    "102344328_room_4": ("chest", "door"),
    "102344328_room_5": ("wardrobe", "chest"),
    "102344328_room_8": ("door", "bin"),
    "102344328_room_9": ("shower", "window"),
    "102344328_room_10": ("plant_1", "plant_2"),
    "102344439_room_2": ("plant", "nightstand_2"),
    "102344439_room_13": ("door_1", "door_2"),
    "102344457_room_8": ("fridge_1", "cabinet_3"),
    "102344457_room_10": ("vase_3", "chair_3"),
    "102815835_room_3": ("door", "curtain"),
    "102815835_room_8": ("door_2", "door_1"),
    "102815859_room_11": ("cabinet_1", "cabinet_4"),
    "102816036_room_1": ("nightstand", "chair"),
    "102816036_room_3": ("bookcase", "cabinet_1"),
    "102816036_room_31": ("door_2", "chair_2"),
    "102816051_room_1": ("shower_2", "door"),
    "102816051_room_3": ("door_3", "bed"),
    "102816066_room_0": ("window_2", "door"),
    "102816066_room_1": ("chair_2", "dresser"),
    "102816066_room_11": ("sofa_1", "shelves"),
    "102816066_room_13": ("door", "bookshelf"),
    "102816066_room_21": ("window_2", "artwork"),
    "102816066_room_22": ("window_3", "kitchen"),
    "102816114_room_2": ("table_1", "lamp_2"),
    "102816114_room_4": ("door_1", "chair"),
    "102816114_room_11": ("door", "island"),
    "102816216_room_0": ("door", "bath_2"),
    "102816216_room_2": ("door_2", "chair_1"),
    "102816216_room_6": ("door", "toilet_2"),
    "102816216_room_7": ("artwork_1", "bookcase"),
    "102816216_room_10": ("refrigerator", "stove"),
    "102816615_room_2": ("door", "dresser_1"),
    "102816615_room_4": ("door", "bath"),
    "102816615_room_6": ("credenza", "sofa_1"),
    "102816615_room_7": ("door_1", "dresser"),
    "102816627_room_6": ("door_1", "chest"),
    "102816729_room_8": ("door", "sofa"),
    "102816729_room_9": ("nightstand_1", "fabric_1"),
    "102816756_room_5": ("door", "wardrobe"),
    "102816786_room_6": ("artwork", "bookshelf_2"),
    "102816786_room_7": ("door", "mirror_2"),
    "102816852_room_0": ("door", "fabric"),
    "102816852_room_5": ("basket", "fabric"),
    "102817053_room_1": ("door", "poster"),
    "102817140_room_2": ("door", "vanity_2"),
    "102817140_room_9": ("bin_1", "highchair_1"),
    "102817200_room_1": ("door_2", "table_2"),
    "102817200_room_2": ("furniture", "window_2"),
    "102817200_room_5": ("door", "bin_2"),
    "103997424_171030444_room_0": ("door", "dresser"),
    "103997424_171030444_room_1": ("stool_1", "bin"),
    "103997424_171030444_room_3": ("door", "window"),
    "103997424_171030444_room_6": ("chair_1", "light"),
    "103997445_171030492_room_3": ("door", "bed"),
    "103997445_171030492_room_5": ("door", "basket"),
    "103997445_171030492_room_8": ("door_1", "table_1"),
    "103997445_171030492_room_10": ("door", "pouf"),
    "103997460_171030507_room_2": ("door", "window_1"),
    "103997460_171030507_room_5": ("door", "daybed_1"),
    "103997478_171030525_room_0": ("artwork_5", "artwork_7"),
    "103997478_171030525_room_5": ("door", "vanity"),
    "103997478_171030525_room_7": ("door", "bookcase"),
    "103997478_171030528_room_2": ("toilet", "bathhtub_1"),
    "103997541_171030615_room_4": ("door", "mirror"),
    "103997562_171030642_room_4": ("door", "television"),
    "103997562_171030642_room_5": ("door_1", "bookshelf"),
    "103997562_171030642_room_6": ("door", "toilet_2"),
    "103997562_171030642_room_13": ("door_2", "stool_1"),
    "103997586_171030666_room_5": ("stove", "cabinet_7"),
    "103997586_171030669_room_1": ("door", "picture"),
    "103997586_171030669_room_3": ("door", "bed"),
    "103997586_171030669_room_4": ("blind_3", "plant_1"),
    "103997613_171030702_room_7": ("door", "shower"),
    "103997613_171030702_room_8": ("door", "chair_2"),
    "103997613_171030702_room_9": ("door", "hamper"),
    "103997613_171030702_room_12": ("door", "chair"),
    "103997643_171030747_room_2": ("door", "bed_2"),
    "103997643_171030747_room_4": ("door_1", "basket"),
    "103997643_171030747_room_6": ("door", "wardrobe"),
    "103997781_171030978_room_4": ("chair", "tv"),
    "103997781_171030978_room_5": ("door", "toilet_1"),
    "103997781_171030978_room_8": ("door_1", "toilet"),
    "103997781_171030978_room_9": ("door_1", "wardrobe"),
    "103997781_171030978_room_10": ("aquarium", "window_1"),
    "103997799_171031002_room_3": ("door", "sofa_3"),
    "103997799_171031002_room_17": ("chair", "window_1"),
    "103997799_171031002_room_18": ("chair", "sidetable"),
    "103997895_171031182_room_0": ("door", "sofa_2"),
    "103997895_171031182_room_7": ("door", "bin"),
    "103997919_171031233_room_1": ("door", "pan"),
    "103997919_171031233_room_3": ("door", "curtain_1"),
    "103997940_171031257_room_4": ("door", "fabric_1"),
    "103997940_171031257_room_18": ("bin", "holder_1"),
    "103997940_171031257_room_21": ("chair", "nightstand_1"),
    "103997940_171031257_room_22": ("chair", "lamp_1"),
    "103997940_171031257_room_23": ("chest", "door"),
    "103997940_171031257_room_26": ("mirror", "artwork"),
    "103997940_171031257_room_27": ("door", "wardrobe"),
    "103997970_171031287_room_1": ("door_2", "tree"),
    "103997970_171031287_room_2": ("door_2", "plant"),
    "103997994_171031320_room_1": ("door", "toilet_2"),
    "103997994_171031320_room_2": ("door_2", "wardrobe"),
    "104348010_171512832_room_13": ("door_1", "plant"),
    "104348010_171512832_room_14": ("door_1", "painting"),
    "104348010_171512832_room_15": ("door", "toilet"),
    "104348010_171512832_room_19": ("mat_1", "mat_3"),
    "104348010_171512832_room_20": ("door", "painting"),
    "104348028_171512877_room_5": ("door", "chair"),
    "104348037_171512898_room_2": ("door", "bag"),
    "104348082_171512994_room_2": ("door", "nightstand"),
    "104348082_171512994_room_4": ("door", "curtain_1"),
    "104348160_171513093_room_2": ("door", "chair"),
    "104348181_171513120_room_14": ("door_2", "sofa_3"),
    "104348181_171513120_room_17": ("door", "island_3"),
    "104348202_171513150_room_0": ("plant_1", "door_1"),
    "104348202_171513150_room_2": ("sofa_1", "chair"),
    "104348202_171513150_room_13": ("island_1", "bin"),
    "104348328_171513363_room_2": ("door", "spray_2"),
    "104348328_171513363_room_6": ("door", "wardrobe"),
    "104348394_171513453_room_1": ("door_1", "rail"),
    "104348463_171513588_room_1": ("door_2", "dresser"),
    "104348511_171513654_room_15": ("fridge freezer", "cabinet_8"),
    "104862384_172226319_room_3": ("door", "wardrobe_2"),
    "104862396_172226349_room_1": ("door_1", "fireplace"),
    "104862396_172226349_room_3": ("door", "bath"),
    "104862396_172226349_room_5": ("door_2", "bookcase"),
    "104862396_172226349_room_7": ("door", "sofa_4"),
    "104862396_172226349_room_12": ("door", "bed_2"),
    "104862396_172226349_room_17": ("door", "plant_2"),
    "104862417_172226382_room_0": ("door_3", "bed"),
    "104862417_172226382_room_1": ("door_1", "nighstand"),
    "104862417_172226382_room_3": ("pouf_1", "table"),
    "104862417_172226382_room_25": ("stove", "cabinet_1"),
    "104862501_172226556_room_0": ("door_2", "chair_1"),
    "104862501_172226556_room_1": ("door", "bathtub"),
    "104862501_172226556_room_2": ("door", "shower"),
    "104862501_172226556_room_3": ("door_1", "pet"),
    "104862501_172226556_room_7": ("door", "pet"),
    "104862513_172226580_room_6": ("door", "trunk_2"),
    "104862534_172226625_room_2": ("door_2", "blind"),
    "104862534_172226625_room_3": ("door", "blind"),
    "104862558_172226664_room_6": ("stove", "unit"),
    "104862573_172226682_room_2": ("tv", "table"),
    "104862639_172226823_room_2": ("door_2", "fireplace"),
    "104862639_172226823_room_5": ("door_1", "curtain_2"),
    "104862660_172226844_room_5": ("sofa", "stool"),
    "104862669_172226853_room_5": ("door", "holder"),
    "104862681_172226874_room_10": ("door", "sofa"),
    "104862687_172226883_room_3": ("door", "bath"),
    "104862687_172226883_room_5": ("door", "island"),
    "105515184_173104128_room_5": ("door", "shelf"),
    "105515184_173104128_room_6": ("door", "shelf"),
    "105515184_173104128_room_12": ("door", "bathtub"),
    "105515184_173104128_room_15": ("door_1", "stool"),
    "105515235_173104215_room_0": ("door", "chair"),
    "105515235_173104215_room_17": ("door_2", "chair"),
    "105515307_173104317_room_7": ("door", "cabinet"),
    "105515337_173104347_room_2": ("door_2", "chair"),
    "105515403_173104449_room_1": ("door_2", "fridge"),
    "105515403_173104449_room_3": ("chair", "armchair"),
    "105515448_173104512_room_5": ("door", "shower"),
    "105515448_173104512_room_7": ("door_3", "chair"),
    "105515490_173104566_room_2": ("door_2", "unit"),
    "105515490_173104566_room_3": ("door_2", "wardrobe"),
    "105515505_173104584_room_0": ("door", "toilet_2"),
    "105515505_173104584_room_7": ("door", "painting"),
    "105515541_173104641_room_6": ("door", "tray"),
    "106366104_174226329_room_11": ("door", "refrigerator"),
    "106366104_174226332_room_3": ("door", "bench"),
    "106366104_174226332_room_4": ("bench_1", "sideboard"),
    "106366104_174226332_room_8": ("door", "toilet"),
    "106366104_174226332_room_9": ("door", "dresser"),
    "106366173_174226431_room_2": ("door", "bin"),
    "106366248_174226527_room_3": ("door", "locker_2"),
    "106366263_174226557_room_0": ("door_1", "chair_2"),
    "106366263_174226557_room_1": ("door", "toilet_2"),
    "106366263_174226557_room_4": ("door", "toilet_2"),
    "106366263_174226557_room_7": ("window_1", "sofa"),
}


def _eligible_labels(objects: List[Dict], room_type: Optional[str]) -> List[str]:
    """
    Filter labels by room_type pool (if provided), else return all.
    Rugs are excluded from candidates.
    """
    pool = CHOICE_OPTIONS.get((room_type or "").lower(), None)
    labels = []
    for o in objects:
        lab = _label(o)
        if not lab:
            continue
        if "rug" in lab.lower():
            continue
        if pool:
            if any(tok in lab.lower() for tok in pool):
                labels.append(lab)
        else:
            labels.append(lab)
    # fallback -> all non-rug labels if filtered empty
    if not labels:
        labels = [_label(o) for o in objects if "rug" not in _label(o).lower()]
    return labels


def process_single_file(file_path: str, file: str, room_type: str, out_dir: str) -> Optional[Dict]:

    with open(file_path, "r", encoding="utf-8") as f:
        data = json.load(f)

    layout_id = str(data.get("layout_id"))
    raw_file_name = os.path.splitext(file)[0]

    # objects = data.get("objects", []) or []
    rng = np.random.default_rng(seed=abs(hash(layout_id)))

    objects = merge_objects_and_openings(data)

    room_type_resolved = room_type
    if room_type_resolved == "unknown":
        room_type_resolved = data.get("room_type") or (data.get("room", {}) or {}).get("room_type") or "unknown"

    labels_all = [_label(o) for o in objects if _label(o)]
    if len(labels_all) < 2:
        return None

    # 1) Try explicit overrides first (by raw filename, then by layout_id)
    start_label = goal_label = None
    for key in (raw_file_name, layout_id):
        if key and key in MAP_OVERRIDES:
            print("Override found for", key)
            a, b = MAP_OVERRIDES[key]
            if a in labels_all and b in labels_all:
                start_label, goal_label = a, b
            # whether or not both exist, we don't try the other key once matched
            break

    # 2) If no valid override, pick from room-type pool; else random from all
    if start_label is None or goal_label is None:
        pool = _eligible_labels(objects, room_type_resolved)
        if len(pool) >= 2:
            i, j = rng.choice(len(pool), 2, replace=False)
            start_label, goal_label = pool[i], pool[j]
        else:
            i, j = rng.choice(len(labels_all), 2, replace=False)
            start_label, goal_label = labels_all[i], labels_all[j]

    print(f"Chosen pair: {start_label} → {goal_label} in room type '{room_type_resolved}')")

    # build path & graph (fully connected, no obstacles)
    clearance = 0.1  # meters
    path, room_poly, obj_polys, sp, gp, G = find_path_room_constrained_blocked(data, objects, start_label=start_label, goal_label=goal_label, clearance=clearance)

    # render (no try/except)
    out_png = Path(out_dir) / f"{layout_id}.png"
    out_png.parent.mkdir(parents=True, exist_ok=True)
    title = f"{layout_id}  |  {start_label} → {goal_label}  (ALL-POINTS GRAPH)"
    render_visibility_path(
        data=data,
        room_polygon=room_poly,
        obstacles=obj_polys,
        path_xy=path,
        start_pt=sp,
        goal_pt=gp,
        out_path=out_png,
        title=f"{layout_id}  |  {start_label} → {goal_label} (room-constrained, object-blocked)",
        show_grid=True,
        show_axes=True,
        dpi=150,
        objects=objects,
        start_label=start_label,
        goal_label=goal_label,
        vis_graph=G,
    )

    return {
        "layout_id": layout_id,
        "room_type": room_type_resolved,
        "object_1": start_label,
        "object_2": goal_label,
        "clearance": clearance,
        "path_len": len(path),
        "answer": path,
        "N_objects": len(objects),
    }


def main_shortest_path(
    input_dir: str = "data/hssd_data/new_format",
    output_csv: str = "benchmark/{parent_folder_name}/{parent_folder_name}_qa_hssd_data.csv",
    output_img: str = "benchmark/{parent_folder_name}/{parent_folder_name}_qa_hssd_images/",
    enable_subsampling: bool = False,
    bedrooms_count: int = 80,
    living_rooms_count: int = 80,
    kitchens_count: int = 40,
):
    parent_folder_name = os.path.basename(os.path.dirname(os.path.realpath(__file__)))
    output_csv = output_csv.format(parent_folder_name=parent_folder_name)
    output_img = output_img.format(parent_folder_name=parent_folder_name)

    # Create output directory if it doesn't exist
    os.makedirs(os.path.dirname(output_csv), exist_ok=True)

    # Configure subsampling
    subsample_config = None
    if enable_subsampling:
        subsample_config = {"bedrooms": bedrooms_count, "living_rooms": living_rooms_count, "kitchens": kitchens_count}
        print(f"Subsampling enabled: {subsample_config}")
    else:
        print("Processing all available files")

    qa_pairs = generate_qa_pairs_with_subsampling(input_dir=input_dir, process_single_file=partial(process_single_file, out_dir=output_img), subsample_config=subsample_config)

    save_and_info(qa_pairs, output_csv=output_csv)
