import base64
import os
from typing import List, Literal, Optional

from .prompts import PROMPT_EXTRACT_DATA_FROM_CHART, PROMPT_GENERATE_PLOTTING_CODE


def extract_block(text: str, block_type: Literal["csv", "python"]) -> Optional[str]:
    """
    Extract a block of text from llm response.

    Args:
        text: The llm response containing the text block
        block_type: The type of block to extract ("csv" or "python")

    Returns:
        The extracted text block or None if not found
    """
    start_marker = f"```{block_type}"
    end_marker = "```"

    start_idx = text.find(start_marker)
    end_idx = text.rfind(end_marker)

    if start_idx == -1 or end_idx == -1:
        return None

    # Add length of start marker to get actual start of content
    content_start = start_idx + len(start_marker)
    return text[content_start:end_idx].strip()


def _encode_image(image_path):
    """
    Encode an image file to base64.

    Args:
        image_path: Path to the image file

    Returns:

    """
    with open(image_path, "rb") as image_file:
        return base64.b64encode(image_file.read()).decode("utf-8")


def extract_data_from_chart(
    image: str,
    openai_client,
    model: str,
    output_file_path: str = None,
    prompt: str = PROMPT_EXTRACT_DATA_FROM_CHART,
    use_cache: bool = False,
    code_interpreter: bool = False,
    **kwargs,
) -> str:
    """
    Extract data from a chart image.

    Args:
        image: URL or local path of the chart image
        openai_client: OpenAI client instance
        model: Model name to use for extraction
        output_file_path: Path of CSV file to save the extracted data
        use_cache: Whether to directly return the cached data if it exists
        code_interpreter: Whether to use the code interpreter
        **kwargs: Additional arguments to pass to the openai_client.responses.create method
    Returns:
        Dictionary containing the extracted CSV data, input tokens, and output tokens

    Raises:
        ValueError: If no CSV data block is found in the response
    """
    if use_cache and os.path.exists(output_file_path):
        with open(output_file_path, "r") as f:
            return f.read()

    if image.startswith(("http://", "https://")):
        image_input = {"type": "input_image", "image_url": image, "detail": "high"}
    else:
        image_input = {
            "type": "input_image",
            "image_url": f"data:image/jpeg;base64,{_encode_image(image)}",
            "detail": "high",
        }

    response = openai_client.responses.create(
        model=model,
        tools=(
            [{"type": "code_interpreter", "container": {"type": "auto"}}]
            if code_interpreter
            else None
        ),
        input=[
            {
                "role": "user",
                "content": [
                    {"type": "input_text", "text": prompt},
                    image_input,
                ],
            }
        ],
        **kwargs,
    )

    # csv_data = extract_block(response.output_text, "csv")
    # if csv_data is None:
    #     raise ValueError("No CSV data block found in the response")
    csv_data = response.output_text
    if output_file_path:
        with open(output_file_path, "w") as f:
            f.write(csv_data)
    print(response)
    result = {
        "csv_data": csv_data,
        "input_tokens": response.usage.input_tokens,
        "output_tokens": response.usage.output_tokens,
    }

    return result
