#!/bin/bash
# DATE-GFN Environment Setup Script
# This script automates the setup of the DATE-GFN research environment

set -e  # Exit on any error

echo " Setting up DATE-GFN Research Environment"
echo "============================================"

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Function to print colored output
print_status() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

print_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

print_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# Check if conda is installed
check_conda() {
    if command -v conda &> /dev/null; then
        print_success "Conda is installed"
        conda --version
        return 0
    else
        print_error "Conda is not installed. Please install Anaconda or Miniconda first."
        echo "Download from: https://docs.conda.io/en/latest/miniconda.html"
        return 1
    fi
}

# Check if git is installed
check_git() {
    if command -v git &> /dev/null; then
        print_success "Git is installed"
        git --version
        return 0
    else
        print_error "Git is not installed. Please install Git first."
        return 1
    fi
}

# Create conda environment
create_environment() {
    ENV_NAME="date-gfn"
    PYTHON_VERSION="3.8"
    
    print_status "Creating conda environment: $ENV_NAME"
    
    if conda env list | grep -q "^$ENV_NAME "; then
        print_warning "Environment '$ENV_NAME' already exists"
        read -p "Do you want to remove and recreate it? (y/N): " -n 1 -r
        echo
        if [[ $REPLY =~ ^[Yy]$ ]]; then
            print_status "Removing existing environment..."
            conda env remove -n $ENV_NAME -y
        else
            print_status "Using existing environment"
            return 0
        fi
    fi
    
    print_status "Creating new environment with Python $PYTHON_VERSION..."
    conda create -n $ENV_NAME python=$PYTHON_VERSION -y
    print_success "Environment created successfully"
}

# Activate environment and install dependencies
install_dependencies() {
    ENV_NAME="date-gfn"
    
    print_status "Activating environment and installing dependencies..."
    
    # Source conda and activate environment
    eval "$(conda shell.bash hook)"
    conda activate $ENV_NAME
    
    print_status "Installing PyTorch..."
    # Install PyTorch (CPU version by default, user can modify for GPU)
    conda install pytorch torchvision torchaudio cpuonly -c pytorch -y
    
    print_status "Installing Python dependencies..."
    pip install --upgrade pip
    pip install -r requirements.txt
    
    print_status "Installing DATE-GFN in development mode..."
    pip install -e .
    
    print_success "All dependencies installed successfully"
}

# Setup Wandb (optional)
setup_wandb() {
    print_status "Setting up Weights & Biases (optional)..."
    
    read -p "Do you want to setup Wandb for experiment tracking? (y/N): " -n 1 -r
    echo
    if [[ $REPLY =~ ^[Yy]$ ]]; then
        print_status "Installing wandb..."
        pip install wandb
        
        print_status "Please login to Wandb..."
        wandb login
        
        read -p "Enter your Wandb entity/username: " WANDB_ENTITY
        if [ ! -z "$WANDB_ENTITY" ]; then
            echo "export WANDB_ENTITY=\"$WANDB_ENTITY\"" >> ~/.bashrc
            export WANDB_ENTITY="$WANDB_ENTITY"
            print_success "Wandb entity set to: $WANDB_ENTITY"
        fi
    else
        print_status "Skipping Wandb setup"
    fi
}

# Setup CUDA (optional)
setup_cuda() {
    print_status "Checking CUDA availability..."
    
    if command -v nvidia-smi &> /dev/null; then
        print_success "NVIDIA GPU detected"
        nvidia-smi --query-gpu=name,memory.total --format=csv,noheader
        
        read -p "Do you want to install CUDA-enabled PyTorch? (y/N): " -n 1 -r
        echo
        if [[ $REPLY =~ ^[Yy]$ ]]; then
            print_status "Installing CUDA-enabled PyTorch..."
            eval "$(conda shell.bash hook)"
            conda activate date-gfn
            
            # Remove CPU-only PyTorch
            pip uninstall torch torchvision torchaudio -y
            
            # Install CUDA PyTorch (CUDA 11.8 by default)
            conda install pytorch torchvision torchaudio pytorch-cuda=11.8 -c pytorch -c nvidia -y
            
            print_success "CUDA-enabled PyTorch installed"
            
            # Set environment variables
            echo "export CUDA_VISIBLE_DEVICES=0" >> ~/.bashrc
            echo "export KMP_DUPLICATE_LIB_OK=TRUE" >> ~/.bashrc
            export CUDA_VISIBLE_DEVICES=0
            export KMP_DUPLICATE_LIB_OK=TRUE
        fi
    else
        print_warning "No NVIDIA GPU detected, using CPU-only version"
    fi
}

# Run tests to verify installation
run_tests() {
    print_status "Running installation tests..."
    
    eval "$(conda shell.bash hook)"
    conda activate date-gfn
    
    # Test basic import
    python -c "from date_gfn import DATEGFN, HypergridEnvironment; print('✓ Basic imports successful')"
    
    # Test PyTorch
    python -c "import torch; print(f'✓ PyTorch {torch.__version__} available'); print(f'✓ CUDA available: {torch.cuda.is_available()}')"
    
    # Test environment creation
    python -c "from date_gfn import HypergridEnvironment; env = HypergridEnvironment(height=4, ndim=2); print(f'✓ Environment created with {len(env.get_all_modes())} modes')"
    
    # Test model creation
    python -c "from date_gfn import DATEGFN; model = DATEGFN(state_dim=2, action_dim=5, hidden_dim=32); print('✓ DATE-GFN model created')"
    
    print_success "All tests passed!"
}

# Create directory structure
create_directories() {
    print_status "Creating experiment directories..."
    
    mkdir -p results/{logs,checkpoints,plots}
    mkdir -p data
    
    print_success "Directory structure created"
}

# Generate activation script
create_activation_script() {
    print_status "Creating activation script..."
    
    cat > activate_date_gfn.sh << 'EOF'
#!/bin/bash
# DATE-GFN Environment Activation Script

echo "🧬 Activating DATE-GFN Research Environment"

# Activate conda environment
eval "$(conda shell.bash hook)"
conda activate date-gfn

# Set environment variables
export CUDA_VISIBLE_DEVICES=0
export KMP_DUPLICATE_LIB_OK=TRUE

# Set Python path
export PYTHONPATH="${PYTHONPATH}:$(pwd)/src"

echo "Environment activated!"
echo " Current directory: $(pwd)"
echo " Python: $(which python)"
echo " PyTorch: $(python -c 'import torch; print(torch.__version__)')"
echo " CUDA available: $(python -c 'import torch; print(torch.cuda.is_available())')"

if [ ! -z "$WANDB_ENTITY" ]; then
    echo " Wandb entity: $WANDB_ENTITY"
fi

echo ""
echo "Quick start commands:"
echo "  python examples/quick_start/demo.py              # Run basic demo"
echo "  python examples/hypergrid/simple_demo.py         # Hypergrid comparison"
echo "  python scripts/run_comparative_analysis.py       # Full comparison"
echo ""
EOF

    chmod +x activate_date_gfn.sh
    print_success "Activation script created: ./activate_date_gfn.sh"
}

# Main setup function
main() {
    echo ""
    print_status "Starting DATE-GFN setup process..."
    echo ""
    
    # Check prerequisites
    check_conda || exit 1
    check_git || exit 1
    
    # Setup process
    create_environment
    install_dependencies
    setup_wandb
    setup_cuda
    create_directories
    run_tests
    create_activation_script
    
    echo ""
    print_success "🎉 DATE-GFN setup completed successfully!"
    echo ""
    echo "Next steps:"
    echo "1. Activate the environment: source activate_date_gfn.sh"
    echo "2. Run a quick demo: python examples/quick_start/demo.py"
    echo "3. Read the documentation: docs/EXPERIMENTAL_GUIDE.md"
    echo ""
    print_status "Happy researching!"
}

# Help function
show_help() {
    echo "DATE-GFN Environment Setup Script"
    echo ""
    echo "Usage: $0 [OPTIONS]"
    echo ""
    echo "Options:"
    echo "  -h, --help     Show this help message"
    echo "  --no-cuda      Skip CUDA setup"
    echo "  --no-wandb     Skip Wandb setup"
    echo "  --env-name     Custom environment name (default: date-gfn)"
    echo ""
    echo "This script will:"
    echo "1. Create a conda environment with Python 3.8"
    echo "2. Install PyTorch and dependencies"
    echo "3. Install DATE-GFN in development mode"
    echo "4. Optionally setup CUDA and Wandb"
    echo "5. Run verification tests"
    echo "6. Create activation script"
}

# Parse command line arguments
SKIP_CUDA=false
SKIP_WANDB=false
ENV_NAME="date-gfn"

while [[ $# -gt 0 ]]; do
    case $1 in
        -h|--help)
            show_help
            exit 0
            ;;
        --no-cuda)
            SKIP_CUDA=true
            shift
            ;;
        --no-wandb)
            SKIP_WANDB=true
            shift
            ;;
        --env-name)
            ENV_NAME="$2"
            shift 2
            ;;
        *)
            print_error "Unknown option: $1"
            show_help
            exit 1
            ;;
    esac
done

# Run main setup
main
