import json
import logging
import os
from pathlib import Path
import sys
from functools import wraps

from tenacity import retry, stop_after_attempt, wait_exponential_jitter, wait_fixed


class MyNameSpace:
    pass


def setup_logger(folder_path, log_file_name="logger.log", console_output=False, logger_name="text2svg"):
    dir_root = Path(folder_path)
    full_path = dir_root.joinpath(log_file_name)

    already_exist = Path(full_path).exists()

    logger = logging.getLogger(logger_name)
    logger.setLevel(logging.INFO)

    formatter = logging.Formatter("%(asctime)s| %(message)s", "%m-%d|%H:%M:%S")

    file_hdl = logging.FileHandler(full_path)
    file_hdl.setFormatter(formatter)

    logger.addHandler(file_hdl)

    if console_output:
        console_hdl = logging.StreamHandler()
        console_hdl.setFormatter(formatter)
        logger.addHandler(console_hdl)

    logger.info("")
    logger.info("-*" * 30)
    logger.info("Logger ready")
    if already_exist:
        logger.info("")
        logger.info("")
        logger.info(f">>>>> Logger file {full_path} already exist, append to it. <<<<<")
        logger.info("")
        logger.info("")


def is_container_env():
    container_env_vars = ["KUBERNETES_SERVICE_HOST", "KUBERNETES_SERVICE_PORT"]
    for var in container_env_vars:
        if os.getenv(var):
            return True

    if not sys.stdout.isatty():
        return True

    return False


def load_jsonl_iter(file, limit=None, skip=None):
    if skip is None:
        skip = 0

    with Path(file).open("r") as f:
        for idx, line in enumerate(f):
            if idx < skip:
                continue
            if limit and idx >= skip + limit:
                return
            if line.strip():
                yield json.loads(line.strip())


def read_jsonl(fn):
    with open(fn) as f:
        return [json.loads(l) for l in f.readlines()]


def write_jsonl(fn, data):
    with open(fn, "w") as f:
        f.write("\n".join([json.dumps(l, ensure_ascii=False) for l in data]))


def on_retry_error(s):
    e = s.outcome.exception()
    # print(f"give up retrying. error: {e}")
    raise e


def before_retry_sleep(s):
    msg = f"function call error for {s.attempt_number} time(s), will retry... error: {s.outcome.exception()}"
    if s.attempt_number > 10:
        print(msg)
    else:
        print(msg)


def configurable_retry(max_attempts):
    def decorator(func):
        @wraps(func)
        @retry(
            # wait=wait_exponential_jitter(),
            wait=wait_fixed(2),
            stop=stop_after_attempt(max_attempts),
            # before_sleep=before_retry_sleep,
            retry_error_callback=on_retry_error,
        )
        def sync_wrapper(*args, **kwargs):
            return func(*args, **kwargs)

        return sync_wrapper

    return decorator


class CompactJSONEncoder(json.JSONEncoder):
    """A JSON Encoder that puts small containers on single lines."""

    CONTAINER_TYPES = (list, tuple, dict)
    """Container datatypes include primitives or other containers."""

    MAX_WIDTH = 500
    """Maximum width of a container that might be put on a single line."""

    MAX_ITEMS = 60
    """Maximum number of items in container that might be put on single line."""

    def __init__(self, *args, **kwargs):
        # using this class without indentation is pointless
        if kwargs.get("indent") is None:
            kwargs["indent"] = 4
        super().__init__(*args, **kwargs)
        self.indentation_level = 0

    def encode(self, o):
        """Encode JSON object *o* with respect to single line lists."""
        if isinstance(o, (list, tuple)):
            return self._encode_list(o)
        if isinstance(o, dict):
            return self._encode_object(o)
        if isinstance(o, float):  # Use scientific notation for floats
            return format(o, "g")
        return json.dumps(
            o,
            skipkeys=self.skipkeys,
            ensure_ascii=self.ensure_ascii,
            check_circular=self.check_circular,
            allow_nan=self.allow_nan,
            sort_keys=self.sort_keys,
            indent=self.indent,
            separators=(self.item_separator, self.key_separator),
            default=self.default if hasattr(self, "default") else None,
        )

    def _encode_list(self, o):
        if self._put_on_single_line(o):
            return "[" + ", ".join(self.encode(el) for el in o) + "]"
        self.indentation_level += 1
        output = [self.indent_str + self.encode(el) for el in o]
        self.indentation_level -= 1
        return "[\n" + ",\n".join(output) + "\n" + self.indent_str + "]"

    def _encode_object(self, o):
        if not o:
            return "{}"
        if self._put_on_single_line(o):
            return "{ " + ", ".join(f"{self.encode(k)}: {self.encode(el)}" for k, el in o.items()) + " }"
        self.indentation_level += 1
        output = [f"{self.indent_str}{json.dumps(k)}: {self.encode(v)}" for k, v in o.items()]

        self.indentation_level -= 1
        return "{\n" + ",\n".join(output) + "\n" + self.indent_str + "}"

    def iterencode(self, o, **kwargs):
        """Required to also work with `json.dump`."""
        return self.encode(o)

    def _put_on_single_line(self, o):
        return self._primitives_only(o) and len(o) <= self.MAX_ITEMS and len(str(o)) - 2 <= self.MAX_WIDTH

    def _primitives_only(self, o):
        if isinstance(o, (list, tuple)):
            return not any(isinstance(el, self.CONTAINER_TYPES) for el in o)
        elif isinstance(o, dict):
            return not any(isinstance(el, self.CONTAINER_TYPES) for el in o.values())

    @property
    def indent_str(self) -> str:
        if isinstance(self.indent, int):
            return " " * (self.indentation_level * self.indent)
        elif isinstance(self.indent, str):
            return self.indentation_level * self.indent
        else:
            raise ValueError(f"indent must either be of type int or str (is: {type(self.indent)})")
