import argparse
import torch
import torch.nn as nn
import torch.nn.functional as F
from huggingface_hub import PyTorchModelHubMixin, hf_hub_download
from depth_anything.blocks import FeatureFusionBlock, _make_scratch
import math 

def affinity(fea_a, fea_b):
    B, C, D, H, W = fea_b.shape

    anchor_fea = fea_a.flatten(start_dim=2)

    fea_b = fea_b.reshape((B, C, -1))
    a_sq = fea_b.pow(2).sum(1).unsqueeze(2)  # 4 576 1, where 1 is expand by unsqueeze
    ab = fea_b.transpose(1, 2) @ anchor_fea
    affinity = (2 * ab - a_sq) / math.sqrt(C)  # B, THW, HW

    maxes = torch.max(affinity, dim=1, keepdim=True)[0]
    x_exp = torch.exp(affinity - maxes)
    x_exp_sum = torch.sum(x_exp, dim=1, keepdim=True)
    affinity = x_exp / x_exp_sum

    fea_a = torch.bmm(fea_b, affinity)
    return fea_a.view(B, C, H, W)

def affinity_pattern(fea_a, fea_b):
    B, C, D, H, W = fea_b.shape

    anchor_fea = fea_a.flatten(start_dim=2).transpose(1, 2) ## b, N, c
    fea_b = fea_b.mean(dim=2)
    fea_b = fea_b.reshape((B, C, -1)) ## b, c, N
    fea_b = fea_b.transpose(1, 2) ## b, N, c

    a_sq = fea_b.pow(2).sum(1).unsqueeze(2)  # b, c, 1
    ab = (fea_b.transpose(1, 2) / (H*W)) @ (anchor_fea / (H*W))  ## b c N * b N c -> b c c 

    affinity = (2 * ab - a_sq) / math.sqrt(H*W) / 100  # B, c c 

    maxes = torch.max(affinity, dim=1, keepdim=True)[0]
    x_exp = torch.exp(affinity - maxes)
    x_exp_sum = torch.sum(x_exp, dim=1, keepdim=True)
    affinity = x_exp / x_exp_sum

    fea_a = torch.bmm(fea_b, affinity) ## b N c * b c c -> b N c
    fea_a = fea_a.transpose(1, 2)
    fea_a = fea_a.view(B, C, H, W)

    return fea_a

def _make_fusion_block(features, use_bn, size = None):
    return FeatureFusionBlock(
        features,
        nn.ReLU(False),
        deconv=False,
        bn=use_bn,
        expand=False,
        align_corners=True,
        size=size,
    )

class DPTHead(nn.Module):
    def __init__(self, nclass, in_channels, 
                 features=256, use_bn=False, 
                 out_channels=[256, 512, 1024, 1024], 
                 use_clstoken=False, num_frames=3):
        super(DPTHead, self).__init__()
        
        self.num_frames = num_frames
        self.nclass = nclass
        self.use_clstoken = use_clstoken
        
        self.projects = nn.ModuleList([
            nn.Conv2d(
                in_channels=in_channels,
                out_channels=out_channel,
                kernel_size=1,
                stride=1,
                padding=0,
            ) for out_channel in out_channels
        ])
        
        self.resize_layers = nn.ModuleList([
            nn.ConvTranspose2d(
                in_channels=out_channels[0],
                out_channels=out_channels[0],
                kernel_size=4,
                stride=4,
                padding=0),
            nn.ConvTranspose2d(
                in_channels=out_channels[1],
                out_channels=out_channels[1],
                kernel_size=2,
                stride=2,
                padding=0),
            nn.Identity(),
            nn.Conv2d(
                in_channels=out_channels[3],
                out_channels=out_channels[3],
                kernel_size=3,
                stride=2,
                padding=1)
        ])
        
        if use_clstoken:
            self.readout_projects = nn.ModuleList()
            for _ in range(len(self.projects)):
                self.readout_projects.append(
                    nn.Sequential(
                        nn.Linear(2 * in_channels, in_channels),
                        nn.GELU()))
        
        self.scratch = _make_scratch(
            out_channels,
            features,
            groups=1,
            expand=False,
        )

        self.scratch.stem_transpose = None
        
        self.scratch.refinenet1 = _make_fusion_block(features, use_bn)
        self.scratch.refinenet2 = _make_fusion_block(features, use_bn)
        self.scratch.refinenet3 = _make_fusion_block(features, use_bn)
        self.scratch.refinenet4 = _make_fusion_block(features, use_bn)

        head_features_1 = features
        head_features_2 = 32
        
        self.scratch.output_conv = nn.Sequential(
            nn.Conv2d(head_features_1, head_features_1, kernel_size=3, stride=1, padding=1),
            nn.ReLU(True),
            nn.Conv2d(head_features_1, nclass, kernel_size=1, stride=1, padding=0),
        )
        
        self.fuse_layer_1 = nn.Conv2d(2*in_channels, in_channels, 1, 1, 0)

    def forward(self, out_features, patch_h, patch_w):
        ## we should add pooling in this section
        out = []

        index = -1
        high_level_feature = None 
        for i, x in enumerate(out_features):
            if self.use_clstoken:
                x, cls_token = x[0], x[1]
                readout = cls_token.unsqueeze(1).expand_as(x)
                x = self.readout_projects[i](torch.cat((x, readout), -1))
            else:
                x = x[0]
            
            x = x.permute(0, 2, 1).reshape((x.shape[0], x.shape[-1], patch_h, patch_w))
            
            bt, c, h, w = x.shape
            x = x.reshape((-1, self.num_frames, c, h, w))
            
            if  i == len(out_features) - 1:
                ## temporal fusion 
                short_term_frames = x[:, :, ...].permute(0, 2, 1, 3, 4)

                x = x[:, index, ...] + self.fuse_layer_1(torch.cat([affinity(x[:, index, ...], short_term_frames), affinity_pattern(x[:, index, ...], short_term_frames)], dim=1))
                high_level_feature = x       

            else :
                x = x[:, index, ...]
            
            x = self.projects[i](x)
            x = self.resize_layers[i](x)
            
            out.append(x)
        
        layer_1, layer_2, layer_3, layer_4 = out
        
        layer_1_rn = self.scratch.layer1_rn(layer_1)
        layer_2_rn = self.scratch.layer2_rn(layer_2)
        layer_3_rn = self.scratch.layer3_rn(layer_3)
        layer_4_rn = self.scratch.layer4_rn(layer_4)
        
        path_4 = self.scratch.refinenet4(layer_4_rn, size=layer_3_rn.shape[2:])
        path_3 = self.scratch.refinenet3(path_4, layer_3_rn, size=layer_2_rn.shape[2:])
        path_2 = self.scratch.refinenet2(path_3, layer_2_rn, size=layer_1_rn.shape[2:])
        path_1 = self.scratch.refinenet1(path_2, layer_1_rn)
        
        out = self.scratch.output_conv(path_1)
        out = F.interpolate(out, (int(patch_h * 14), int(patch_w * 14)), mode="bilinear", align_corners=True)
        
        return out, high_level_feature
        
        
class DPT_DINOv2(nn.Module):
    def __init__(self, encoder='vitl', features=256, out_channels=[256, 512, 1024, 1024], use_bn=False, use_clstoken=False, localhub=True):
        super(DPT_DINOv2, self).__init__()
        
        self.num_frames = 3

        assert encoder in ['vits', 'vitb', 'vitl']
        
        # in case the Internet connection is not stable, please load the DINOv2 locally
        if localhub:
            self.pretrained = torch.hub.load('torchhub/facebookresearch_dinov2_main', 'dinov2_{:}14'.format(encoder), source='local', pretrained=False)
        else:
            self.pretrained = torch.hub.load('facebookresearch/dinov2', 'dinov2_{:}14'.format(encoder))
        
        dim = self.pretrained.blocks[0].attn.qkv.in_features
        
        self.depth_head = DPTHead(1, dim, features, use_bn, out_channels=out_channels, use_clstoken=use_clstoken)
        self.depth_head_seg = DPTHead(6, dim, features, use_bn, out_channels=out_channels, use_clstoken=use_clstoken, num_frames=self.num_frames)

        for p in self.pretrained.parameters():
            p.requires_grad = False
        for p in self.depth_head.parameters():
            p.requires_grad = False
        for p in self.depth_head_seg.parameters():
            p.requires_grad = True
        
    def forward(self, x):
        
        
        B, T, C, W, H = x.shape
        x = x.reshape(B*T, C, W, H)

        x = torch.nn.functional.interpolate(x, size=(518, 518), mode="bilinear", 
                                                align_corners=True)
        h, w = x.shape[-2:]

        features = self.pretrained.get_intermediate_layers(x, 4, return_class_token=True)
        
        patch_h, patch_w = h // 14, w // 14

        depth = self.depth_head(features, patch_h, patch_w)
        depth = F.interpolate(depth, size=(h, w), mode="bilinear", align_corners=True)
        depth = F.relu(depth)

        mirror_patterns, high_features = self.depth_head_seg(features, patch_h, patch_w)
        mirror_patterns = F.interpolate(mirror_patterns, size=(h, w), mode="bilinear", align_corners=True)
        mirror_patterns = F.relu(mirror_patterns)

        depth = torch.nn.functional.interpolate(depth, size=(512, 512), mode="bilinear", 
                                                align_corners=True)
        
        mirror_patterns = torch.nn.functional.interpolate(mirror_patterns, size=(512, 512), mode="bilinear", 
                                                align_corners=True)
        
        return torch.cat([depth, mirror_patterns], dim=1), high_features


class DepthAnything(DPT_DINOv2, PyTorchModelHubMixin):
    def __init__(self, config):
        super().__init__(**config)
        

if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument(
        "--encoder",
        default="vits",
        type=str,
        choices=["vits", "vitb", "vitl"],
    )
    args = parser.parse_args()
    
    model = DepthAnything.from_pretrained("LiheYoung/depth_anything_{:}14".format(args.encoder))
    
    print(model)
    