# Copyright (c) Meta Platforms, Inc. and affiliates.
# This software may be used and distributed according to the terms of the Llama 2 Community License Agreement.

import importlib
from functools import partial
from pathlib import Path

import torch

from llama_recipes.datasets import (
    get_qa_dataset,
    get_reward_dataset_ppo,
    get_contrastive_qa_dataset,
    get_contrastive2_qa_dataset,
    get_contrastive3_qa_dataset
)


def load_module_from_py_file(py_file: str) -> object:
    """
    This method loads a module from a py file which is not in the Python path
    """
    module_name = Path(py_file).name
    loader = importlib.machinery.SourceFileLoader(module_name, py_file)
    spec = importlib.util.spec_from_loader(module_name, loader)
    module = importlib.util.module_from_spec(spec)

    loader.exec_module(module)

    return module


def get_custom_dataset(dataset_config, tokenizer, split: str):
    if ":" in dataset_config.file:
        module_path, func_name = dataset_config.file.split(":")
    else:
        module_path, func_name = dataset_config.file, "get_custom_dataset"

    if not module_path.endswith(".py"):
        raise ValueError(f"Dataset file {module_path} is not a .py file.")

    module_path = Path(module_path)
    if not module_path.is_file():
        raise FileNotFoundError(f"Dataset py file {module_path.as_posix()} does not exist or is not a file.")

    module = load_module_from_py_file(module_path.as_posix())
    try:
        return getattr(module, func_name)(dataset_config, tokenizer, split)
    except AttributeError as e:
        print(f"It seems like the given method name ({func_name}) is not present in the dataset .py file ({module_path.as_posix()}).")
        raise e


DATASET_PREPROC = {
    "Triviaqa_llama2_7b_chat_threshold_1_0": partial(get_qa_dataset),
    "Triviaqa_llama2_7b_chat_threshold_1_0_half_data": partial(get_qa_dataset),
    "Triviaqa_llama2_7b_chat_threshold_1_0_preference_data": partial(get_reward_dataset_ppo),
    "Triviaqa_llama2_7b_chat_hir": partial(get_qa_dataset),
    "Triviaqa_llama2_7b_chat_threshold_1_0_contrastive": partial(get_contrastive_qa_dataset),
    "Triviaqa_llama2_7b_chat_threshold_1_0_contrastive_debug": partial(get_contrastive_qa_dataset),
    "Triviaqa_llama2_7b_chat_threshold_1_0_debug": partial(get_qa_dataset),
    "Triviaqa_llama2_7b_chat_threshold_0_9_contrastive2_debug": partial(get_contrastive2_qa_dataset),
    "Triviaqa_llama2_7b_chat_threshold_1_0_contrastive2": partial(get_contrastive2_qa_dataset),
    "Triviaqa_llama2_7b_chat_threshold_1_0_contrastive2_debug": partial(get_contrastive2_qa_dataset),
    "Triviaqa_llama2_7b_chat_threshold_1_0_contrastive3": partial(get_contrastive3_qa_dataset),
    "Triviaqa_llama2_7b_chat_threshold_1_0_contrastive3_debug": partial(get_contrastive3_qa_dataset),
    "Triviaqa_llama2_7b_chat_threshold_1_0_contrastive_knowing_0_7": partial(get_contrastive_qa_dataset),
    "Triviaqa_llama2_7b_chat_threshold_1_0_contrastive_knowing_0_7_with_anchor": partial(get_contrastive_qa_dataset),
    "Triviaqa_llama2_7b_chat_threshold_1_0_contrastive_knowing_0_7_p2_correct": partial(get_contrastive_qa_dataset),
    "Triviaqa_llama2_7b_chat_threshold_1_0_different": partial(get_qa_dataset),
    "Triviaqa_mistral_7B_instruct_v0_1_threshold_1_0_contrastive_knowing_0_7_with_anchor": partial(get_contrastive_qa_dataset),
    "Triviaqa_mistral_7B_instruct_v0_1_threshold_1_0": partial(get_qa_dataset),
    "Triviaqa_mistral_7B_instruct_v0_1_threshold_1_0_contrastive_debug": partial(get_contrastive_qa_dataset),
    "Triviaqa_mistral_7B_instruct_v0_1_threshold_1_0_contrastive": partial(get_contrastive_qa_dataset),
    "Triviaqa_mistral_7B_instruct_v0_1_threshold_1_0_contrastive_knowing_0_5_with_anchor": partial(get_contrastive_qa_dataset),
    "Triviaqa_mistral_7B_instruct_v0_1_threshold_1_0_contrastive_knowing_0_9_with_anchor": partial(get_contrastive_qa_dataset),
    "Triviaqa_llama2_7b_chat_threshold_1_0_contrastive_knowing_0_9_with_anchor": partial(get_contrastive_qa_dataset),
    "Triviaqa_llama2_7b_chat_threshold_1_0_contrastive_knowing_0_8_with_anchor": partial(get_contrastive_qa_dataset),
    "Triviaqa_llama2_7b_chat_threshold_1_0_contrastive_knowing_0_6_with_anchor": partial(get_contrastive_qa_dataset),
    "Triviaqa_llama2_7b_chat_threshold_1_0_contrastive_knowing_0_5_with_anchor": partial(get_contrastive_qa_dataset),
    "Triviaqa_llama2_7b_chat_threshold_1_0_contrastive_knowing_0_7_ablation_1": partial(get_contrastive_qa_dataset),
    "Triviaqa_llama2_7b_chat_threshold_1_0_contrastive_knowing_0_7_ablation_2": partial(get_contrastive_qa_dataset),
    "Triviaqa_llama2_7b_chat_threshold_1_0_contrastive_knowing_0_7_ablation_3": partial(get_contrastive_qa_dataset),
}


def get_preprocessed_dataset(
    tokenizer, dataset_config, split: str = "train"
) -> torch.utils.data.Dataset:
    if not dataset_config.dataset in DATASET_PREPROC:
        raise NotImplementedError(f"{dataset_config.dataset} is not (yet) implemented")

    def get_split():
        return (
            dataset_config.train_split
            if split == "train"
            else dataset_config.test_split
        )

    return DATASET_PREPROC[dataset_config.dataset](
        dataset_config,
        tokenizer,
        get_split(),
    )
