# Copyright (c) Meta Platforms, Inc. and affiliates.from_pretrained
# This software may be used and distributed according to the terms of the Llama 2 Community License Agreement.

import os
from pkg_resources import packaging

import fire
import random
import torch
import torch.optim as optim
import datetime
from torch.utils.tensorboard import SummaryWriter
from peft import get_peft_model, prepare_model_for_int8_training
from torch.distributed.fsdp import (
    FullyShardedDataParallel as FSDP,
)
from torch.distributed.fsdp.fully_sharded_data_parallel import CPUOffload
from torch.optim.lr_scheduler import StepLR
from transformers import (
    AutoModel,
    LlamaForCausalLM,
    LlamaTokenizer,
    LlamaConfig,
    MistralForCausalLM,
    MistralConfig,
    PretrainedConfig,
    # LlamaForCausalLM_Constrative
)
from llama_recipes.ContrastiveLlama import LlamaForCausalLM_Constrative, LlamaForCausalLM_Constrative2, LlamaForCausalLM_Constrative3
from llama_recipes.ContrastiveMistral import MistralForCausalLM_Contrastive
from transformers.models.llama.modeling_llama import LlamaDecoderLayer
from transformers.models.mistral.modeling_mistral import MistralDecoderLayer

from llama_recipes.configs import fsdp_config as FSDP_CONFIG
from llama_recipes.configs import train_config as TRAIN_CONFIG
from llama_recipes.data.concatenator import ConcatDataset
from llama_recipes.policies import AnyPrecisionAdamW, apply_fsdp_checkpointing

from llama_recipes.utils import fsdp_auto_wrap_policy
from llama_recipes.utils.config_utils import (
    update_config,
    generate_peft_config,
    generate_dataset_config,
    get_dataloader_kwargs,
)
from llama_recipes.utils.dataset_utils import get_preprocessed_dataset

from llama_recipes.utils.train_utils import (
    train,
    train_ppo_reward_model,
    freeze_transformer_layers,
    setup,
    setup_environ_flags,
    clear_gpu_cache,
    print_model_size,
    get_policies,
    RewardModel
)

class CustomConfig(LlamaConfig):
    def __init__(self, tau=0.01, pooling_method='last', contrastive_loss_ratio=1000,ablation=None, **kwargs):
        super().__init__(**kwargs)
        self.tau = tau
        self.pooling_method = pooling_method
        self.contrastive_loss_ratio = contrastive_loss_ratio
        self.ablation = ablation

def main(**kwargs):
    # Update the configuration for the training and sharding process
    train_config, fsdp_config = TRAIN_CONFIG(), FSDP_CONFIG()
    update_config((train_config, fsdp_config), **kwargs)
    
    # Tensorboard writer
    current_time = datetime.datetime.now().strftime("%Y-%m-%d_%H-%M-%S")
    if train_config.debug:
        writer = SummaryWriter(log_dir=os.path.join('./tensorboard_summaries/debug', train_config.dist_checkpoint_folder, current_time))
    else:   
        writer = SummaryWriter(log_dir=os.path.join('./tensorboard_summaries', train_config.dist_checkpoint_folder, current_time))
    tau = 0.01
    # Set the seeds for reproducibility
    torch.cuda.manual_seed(train_config.seed)
    torch.manual_seed(train_config.seed)
    random.seed(train_config.seed)

    if train_config.enable_fsdp:
        # torchrun specific
        local_rank = int(os.environ["LOCAL_RANK"])
        rank = int(os.environ["RANK"])
        world_size = int(os.environ["WORLD_SIZE"])
        setup(rank, world_size)

    if torch.distributed.is_initialized():
        torch.cuda.set_device(local_rank)
        clear_gpu_cache(local_rank)
        setup_environ_flags(rank)

    # Load the tokenizer and add special tokens
    tokenizer = LlamaTokenizer.from_pretrained(train_config.model_name)
    tokenizer.pad_token_id = tokenizer.eos_token_id

    # Load the pre-trained model and setup its configuration
    use_cache = False if train_config.enable_fsdp else None
    
    # Use flash attention
    if train_config.use_flash_attention:
        attn_implementation = "flash_attention_2"
    else:
        attn_implementation = None

    def extract_threshold_value(s):
        import re
        # 使用正则表达式匹配 "threshold_" 后面的数字
        match = re.search(r'threshold_(\d+_\d+)', s)
        if match:
            return float(match.group(1).replace('_', '.'))  # 将匹配到的字符串转换为 float
        return None

    contrastive = train_config.contrastive
    if contrastive:
        assert "contrastive" in train_config.dataset
        # custom_config = CustomConfig.from_pretrained(
        #     train_config.model_name,
        #     ablation=train_config.ablation
        # )
        if "contrastive2" in train_config.dataset:
            contrastive = 2
        elif "contrastive3" in train_config.dataset:
            contrastive = 3
        else:
            contrastive = 1
        knowing_threshold = extract_threshold_value(train_config.dataset)
        

    if train_config.enable_fsdp and train_config.low_cpu_fsdp:
        """
        for FSDP, we can save cpu memory by loading pretrained model on rank0 only.
        this avoids cpu oom when loading large models like llama 70B, in which case
        model alone would consume 2+TB cpu mem (70 * 4 * 8). This will add some comms
        overhead and currently requires latest nightly.
        """
        v = packaging.version.parse(torch.__version__)
        verify_latest_nightly = v.is_devrelease and v.dev >= 20230701
        if not verify_latest_nightly:
            raise Exception("latest pytorch nightly build is required to run with low_cpu_fsdp config, "
                            "please install latest nightly.")
        if rank == 0:
            if 'mistral' in train_config.model_name.lower():
                if contrastive:
                    if contrastive == 1:
                        model = MistralForCausalLM_Contrastive.from_pretrained(
                            train_config.model_name,
                            load_in_8bit=True if train_config.quantization else None,
                            device_map="auto" if train_config.quantization else None,
                            use_cache=use_cache,
                            attn_implementation = attn_implementation
                        )
                else:
                    model = MistralForCausalLM.from_pretrained(
                        train_config.model_name,
                        load_in_8bit=True if train_config.quantization else None,
                        device_map="auto" if train_config.quantization else None,
                        use_cache=use_cache,
                        attn_implementation = attn_implementation
                    )
            else:
                if contrastive:
                    if contrastive == 1:
                        model = LlamaForCausalLM_Constrative.from_pretrain(
                            train_config.model_name,
                            # config=custom_config,
                            load_in_8bit=True if train_config.quantization else None,
                            device_map="auto" if train_config.quantization else None,
                            use_cache=use_cache,
                        )
                    elif contrastive == 2:
                        model = LlamaForCausalLM_Constrative2.from_pretrain(
                            train_config.model_name,
                            load_in_8bit=True if train_config.quantization else None,
                            device_map="auto" if train_config.quantization else None,
                            use_cache=use_cache,
                        )
                    elif contrastive == 3:
                        model = LlamaForCausalLM_Constrative3.from_pretrain(
                            train_config.model_name,
                            load_in_8bit=True if train_config.quantization else None,
                            device_map="auto" if train_config.quantization else None,
                            use_cache=use_cache,
                        )
                else:
                    model = LlamaForCausalLM.from_pretrained(
                        train_config.model_name,
                        load_in_8bit=True if train_config.quantization else None,
                        device_map="auto" if train_config.quantization else None,
                        use_cache=use_cache,
                    )
        else:
            if 'mistral' in train_config.model_name.lower():
                mistral_config = MistralConfig.from_pretrained(train_config.model_name)
                mistral_config.use_cache = use_cache
                if contrastive:
                    with torch.device("meta"):
                        if contrastive == 1:
                            model = MistralForCausalLM_Contrastive(mistral_config)
                else:
                    with torch.device("meta"):
                        model = MistralForCausalLM(mistral_config)
            else:
                llama_config = LlamaConfig.from_pretrained(train_config.model_name)
                if contrastive:
                    with torch.device("meta"):
                        if contrastive == 1:
                            model = LlamaForCausalLM_Constrative(llama_config)
                        elif contrastive == 2:
                            model = LlamaForCausalLM_Constrative2(llama_config)
                        elif contrastive == 3:
                            model = LlamaForCausalLM_Constrative3(llama_config)
                else:
                    with torch.device("meta"):
                        model = LlamaForCausalLM(llama_config)

    else:
        if train_config.train_ppo_reward_model:
            base_model = AutoModel.from_pretrained(
                    train_config.model_name,
                    load_in_8bit=True if train_config.quantization else None,
                    device_map="auto" if train_config.quantization else None,
                    use_cache=use_cache,
                )
            if train_config.use_fp16:
                model = RewardModel(base_model=base_model, tokenizer=tokenizer, num_padding_at_beginning=0, compute_fp32_loss=True, score_head_dtype=torch.float16).cuda()
            elif fsdp_config.pure_bf16:
                model = RewardModel(base_model=base_model, tokenizer=tokenizer, num_padding_at_beginning=0, compute_fp32_loss=True, score_head_dtype=torch.bfloat16).cuda()
            else:
                model = RewardModel(base_model=base_model, tokenizer=tokenizer, num_padding_at_beginning=0, compute_fp32_loss=False, score_head_dtype=torch.float32).cuda()
        else:
            if 'mistral' in train_config.model_name.lower():
                if contrastive:
                    if contrastive == 1:
                        model = MistralForCausalLM_Contrastive.from_pretrained(
                            train_config.model_name,
                            load_in_8bit=True if train_config.quantization else None,
                            device_map="auto" if train_config.quantization else None,
                            use_cache=use_cache,
                            attn_implementation = attn_implementation
                        )
                else:
                    model = MistralForCausalLM.from_pretrained(
                        train_config.model_name,
                        load_in_8bit=True if train_config.quantization else None,
                        device_map="auto" if train_config.quantization else None,
                        use_cache=use_cache,
                    )
            else:
                if contrastive:
                    if contrastive == 1:
                        model = LlamaForCausalLM_Constrative.from_pretrained(
                            train_config.model_name,
                            load_in_8bit=True if train_config.quantization else None,
                            device_map="auto" if train_config.quantization else None,
                            use_cache=use_cache,
                            attn_implementation = attn_implementation
                        )
                    elif contrastive == 2:
                        model = LlamaForCausalLM_Constrative2.from_pretrained(
                            train_config.model_name,
                            load_in_8bit=True if train_config.quantization else None,
                            device_map="auto" if train_config.quantization else None,
                            use_cache=use_cache,
                            attn_implementation = attn_implementation
                        )
                    elif contrastive == 3:
                        model = LlamaForCausalLM_Constrative3.from_pretrained(
                            train_config.model_name,
                            load_in_8bit=True if train_config.quantization else None,
                            device_map="auto" if train_config.quantization else None,
                            use_cache=use_cache,
                            attn_implementation = attn_implementation,
                            knowing_threshold = knowing_threshold
                        )
                else:
                    model = LlamaForCausalLM.from_pretrained(
                        train_config.model_name,
                        load_in_8bit=True if train_config.quantization else None,
                        device_map="auto" if train_config.quantization else None,
                        use_cache=use_cache,
                        attn_implementation = attn_implementation
                    )
    if train_config.enable_fsdp and train_config.use_fast_kernels:
        """
        For FSDP and FSDP+PEFT, setting 'use_fast_kernels' will enable
        using of Flash Attention or Xformer memory-efficient kernels
        based on the hardware being used. This would speed up fine-tuning.
        """
        try:
            from optimum.bettertransformer import BetterTransformer
            model = BetterTransformer.transform(model)
        except ImportError:
            print("Module 'optimum' not found. Please install 'optimum' it before proceeding.")

    print_model_size(model, train_config, rank if train_config.enable_fsdp else 0)

    # Prepare the model for int8 training if quantization is enabled
    if train_config.quantization:
        model = prepare_model_for_int8_training(model)

    # Convert the model to bfloat16 if fsdp and pure_bf16 is enabled
    if train_config.enable_fsdp and fsdp_config.pure_bf16:
        model.to(torch.bfloat16)

    if train_config.use_peft:
        peft_config = generate_peft_config(train_config, kwargs)
        model = get_peft_model(model, peft_config)
        model.print_trainable_parameters()

    #setting up FSDP if enable_fsdp is enabled
    if train_config.enable_fsdp:
        if not train_config.use_peft and train_config.freeze_layers:

            freeze_transformer_layers(train_config.num_freeze_layers)

        mixed_precision_policy, wrapping_policy = get_policies(fsdp_config, rank)
        if 'mistral' in train_config.model_name.lower():
            my_auto_wrapping_policy = fsdp_auto_wrap_policy(model, MistralDecoderLayer)
        else:
            my_auto_wrapping_policy = fsdp_auto_wrap_policy(model, LlamaDecoderLayer)

        model = FSDP(
            model,
            auto_wrap_policy= my_auto_wrapping_policy if train_config.use_peft else wrapping_policy,
            cpu_offload=CPUOffload(offload_params=True) if fsdp_config.fsdp_cpu_offload else None,
            mixed_precision=mixed_precision_policy if not fsdp_config.pure_bf16 else None,
            sharding_strategy=fsdp_config.sharding_strategy,
            device_id=torch.cuda.current_device(),
            limit_all_gathers=True,
            sync_module_states=train_config.low_cpu_fsdp,
            param_init_fn=lambda module: module.to_empty(device=torch.device("cuda"), recurse=False)
            if train_config.low_cpu_fsdp and rank != 0 else None,
        )
        if fsdp_config.fsdp_activation_checkpointing:
            apply_fsdp_checkpointing(model)
    elif not train_config.quantization and not train_config.enable_fsdp:
        model.to("cuda")

    dataset_config = generate_dataset_config(train_config, kwargs)

     # Load and preprocess the dataset for training and validation
    dataset_train = get_preprocessed_dataset(
        tokenizer,
        dataset_config,
        split="train",
    )

    if not train_config.enable_fsdp or rank == 0:
        print(f"--> Training Set Length = {len(dataset_train)}")

    dataset_val = get_preprocessed_dataset(
        tokenizer,
        dataset_config,
        split="test",
    )
    if not train_config.enable_fsdp or rank == 0:
        print(f"--> Validation Set Length = {len(dataset_val)}")

    if train_config.batching_strategy == "packing":
        dataset_train = ConcatDataset(dataset_train, chunk_size=train_config.context_length, contrastive=contrastive)

    train_dl_kwargs = get_dataloader_kwargs(train_config, dataset_train, tokenizer, "train", contrastive)

    # Create DataLoaders for the training and validation dataset
    train_dataloader = torch.utils.data.DataLoader(
        dataset_train,
        num_workers=train_config.num_workers_dataloader,
        pin_memory=True,
        **train_dl_kwargs,
    )

    eval_dataloader = None
    if train_config.run_validation:
        if train_config.batching_strategy == "packing":
            dataset_val = ConcatDataset(dataset_val, chunk_size=train_config.context_length,contrastive=contrastive)

        val_dl_kwargs = get_dataloader_kwargs(train_config, dataset_val, tokenizer, "val")

        eval_dataloader = torch.utils.data.DataLoader(
            dataset_val,
            num_workers=train_config.num_workers_dataloader,
            pin_memory=True,
            **val_dl_kwargs,
        )

    # Initialize the optimizer and learning rate scheduler
    if fsdp_config.pure_bf16 and fsdp_config.optimizer == "anyprecision":
        optimizer = AnyPrecisionAdamW(
            model.parameters(),
            lr=train_config.lr,
            momentum_dtype=torch.bfloat16,
            variance_dtype=torch.bfloat16,
            use_kahan_summation=False,
            weight_decay=train_config.weight_decay,
        )
    else:
        optimizer = optim.AdamW(
            model.parameters(),
            lr=train_config.lr,
            weight_decay=train_config.weight_decay,
        )
    scheduler = StepLR(optimizer, step_size=1, gamma=train_config.gamma)

    # Start the training process
    if train_config.train_ppo_reward_model:
        results = train_ppo_reward_model(
            model,
            train_dataloader,
            eval_dataloader,
            tokenizer,
            optimizer,
            scheduler,
            train_config.gradient_accumulation_steps,
            train_config,
            fsdp_config if train_config.enable_fsdp else None,
            local_rank if train_config.enable_fsdp else None,
            rank if train_config.enable_fsdp else None,
            writer,
        )
    else:
        results = train(
            model,
            train_dataloader,
            eval_dataloader,
            tokenizer,
            optimizer,
            scheduler,
            train_config.gradient_accumulation_steps,
            train_config,
            fsdp_config if train_config.enable_fsdp else None,
            local_rank if train_config.enable_fsdp else None,
            rank if train_config.enable_fsdp else None,
            writer,
        )
    writer.close()
    if not train_config.enable_fsdp or rank==0:
        [print(f'Key: {k}, Value: {v}') for k, v in results.items()]

if __name__ == "__main__":
    fire.Fire(main)
