# Copyright (c) Facebook, Inc. and its affiliates. All Rights Reserved

import argparse
import collections
import json
import os
import random
import sys
import time
import uuid

import numpy as np
import PIL
import torch
import torchvision
import torch.utils.data
import sklearn.metrics

from domainbed import datasets
from domainbed import hparams_registry
from domainbed import algorithms
from domainbed.lib import misc
from domainbed.lib.fast_data_loader import InfiniteDataLoader, FastDataLoader

import logging
logging.basicConfig(level=logging.INFO)

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description='Domain generalization')
    parser.add_argument('--model_dir', type=str, required=True)
    args = parser.parse_args()

    # If we ever want to implement checkpointing, just persist these values
    # every once in a while, and then load them from disk here.

    checkpoint = torch.load(args.model_dir+'/model.pkl')
    algorithm_dict = checkpoint['model_dict']

    from easydict import EasyDict as edict
    args = edict(checkpoint['args'])
    args.dataset = args.dataset.split('&')[0]
    start_step = args.start_step

    if args.algorithm == 'GroupDRO':
        algorithm_dict['q'] = torch.Tensor()

    print("Environment:")
    print("\tPython: {}".format(sys.version.split(" ")[0]))
    print("\tPyTorch: {}".format(torch.__version__))
    print("\tTorchvision: {}".format(torchvision.__version__))
    print("\tCUDA: {}".format(torch.version.cuda))
    print("\tCUDNN: {}".format(torch.backends.cudnn.version()))
    print("\tNumPy: {}".format(np.__version__))
    print("\tPIL: {}".format(PIL.__version__))

    print('Args:')
    for k, v in sorted(vars(args).items()):
        print('\t{}: {}'.format(k, v))

    if args.hparams_seed == 0:
        hparams = hparams_registry.default_hparams(args.algorithm, args.dataset)
    else:
        hparams = hparams_registry.random_hparams(args.algorithm, args.dataset,
            misc.seed_hash(args.hparams_seed, args.trial_seed))
    if args.hparams:
        hparams.update(json.loads(args.hparams))
    
    hparams.update(checkpoint['model_hparams'])

    print('HParams:')
    for k, v in sorted(hparams.items()):
        print('\t{}: {}'.format(k, v))

    random.seed(args.seed)
    np.random.seed(args.seed)
    torch.manual_seed(args.seed)
    torch.backends.cudnn.deterministic = True
    torch.backends.cudnn.benchmark = False

    if torch.cuda.is_available():
        device = "cuda"
    else:
        device = "cpu"
        raise Exception("NO GPU AVAILABLE")

    if args.dataset in vars(datasets):
        dataset = vars(datasets)[args.dataset](args.data_dir,
            args.test_envs, hparams)
    else:
        raise NotImplementedError

    # Split each env into an 'in-split' and an 'out-split'. We'll train on
    # each in-split except the test envs, and evaluate on all splits.

    # To allow unsupervised domain adaptation experiments, we split each test
    # env into 'in-split', 'uda-split' and 'out-split'. The 'in-split' is used
    # by collect_results.py to compute classification accuracies.  The
    # 'out-split' is used by the Oracle model selectino method. The unlabeled
    # samples in 'uda-split' are passed to the algorithm at training time if
    # args.task == "domain_adaptation". If we are interested in comparing
    # domain generalization and domain adaptation results, then domain
    # generalization algorithms should create the same 'uda-splits', which will
    # be discared at training.
    in_splits = [] # train
    out_splits = [] # train val
    uda_splits = [] # domain adaptation
    for env_i, env in enumerate(dataset):
        uda = []

        out, in_ = misc.split_dataset(env,
            int(len(env)*args.holdout_fraction),
            misc.seed_hash(args.trial_seed, env_i))

        if env_i in args.test_envs:
            uda, in_ = misc.split_dataset(in_,
                int(len(in_)*args.uda_holdout_fraction),
                misc.seed_hash(args.trial_seed, env_i))

        if hparams['class_balanced']:
            in_weights = misc.make_weights_for_balanced_classes(in_)
            out_weights = misc.make_weights_for_balanced_classes(out)
            if uda is not None:
                uda_weights = misc.make_weights_for_balanced_classes(uda)
        else:
            in_weights, out_weights, uda_weights = None, None, None
        in_splits.append((in_, in_weights))
        # out_splits.append((out, out_weights))
        if len(out) > 0:
            out_splits.append((out, out_weights))
        if len(uda):
            uda_splits.append((uda, uda_weights))
    
    print('in_splits', len(in_splits), 'out_splits', len(out_splits), 'uda_splits', len(uda_splits))

    if args.task == "domain_adaptation" and len(uda_splits) == 0:
        raise ValueError("Not enough unlabeled samples for domain adaptation.")

    if args.num_workers is not None:
        dataset.N_WORKERS = args.num_workers

    train_loaders = [FastDataLoader(
        dataset=env,
        batch_size=hparams['batch_size'],
        num_workers=dataset.N_WORKERS)
        for i, (env, env_weights) in enumerate(in_splits)
        if i not in args.test_envs]


    if hparams['study_noise']:
        logging.info("Studying noise")
        noisy_hard_loaders = [FastDataLoader(
            dataset=env,
            batch_size=hparams['test_batch_size'],
            num_workers=dataset.N_WORKERS)
            # for i, env in enumerate(dataset.noisy_datasets + dataset.hard_datasets) if i in args.test_envs]
            for i, env in enumerate(dataset.noisy_datasets + dataset.hard_datasets)]
        eval_loaders = noisy_hard_loaders
        # eval_loader_names += ['env{}_noisy'.format(i)
        #     for i in range(len(dataset.noisy_datasets)) if i in args.test_envs]
        # eval_loader_names += ['env{}_hard'.format(i)
        #     for i in range(len(dataset.hard_datasets)) if i in args.test_envs]

        eval_loader_names = ['env{}_noisy'.format(i)
            for i in range(len(dataset.noisy_datasets))]
        eval_loader_names += ['env{}_hard'.format(i)
            for i in range(len(dataset.hard_datasets))]
        
        eval_weights = [None for _ in eval_loader_names]
    
    # TODO: track worst group performance for theses datasets with subpopulation shifts
    if args.dataset in ['WILDSWaterbirds', 'WILDSCelebA']:
        logging.info("Tracking worst group performance")
        test_group_loaders = [FastDataLoader(
            dataset=env,
            batch_size=hparams['test_batch_size'],
            num_workers=dataset.N_WORKERS)
            for i, env in enumerate(dataset.grouped_test_datasets)]
        eval_loaders += test_group_loaders
        eval_loader_names += ['test{}'.format(i)
            for i in range(len(dataset.grouped_test_datasets))]
        eval_weights = [None for _ in eval_loader_names]

        val_group_loaders = [FastDataLoader(
            dataset=env,
            batch_size=hparams['test_batch_size'],
            num_workers=dataset.N_WORKERS)
            for i, env in enumerate(dataset.grouped_val_datasets)]
        eval_loaders += val_group_loaders
        eval_loader_names += ['val{}'.format(i)
            for i in range(len(dataset.grouped_val_datasets))]
        eval_weights = [None for _ in eval_loader_names]


    algorithm_class = algorithms.get_algorithm_class(args.algorithm)
    algorithm = algorithm_class(dataset.input_shape, dataset.num_classes,
        len(dataset) - len(args.test_envs), hparams)

    if algorithm_dict is not None:
        algorithm.load_state_dict(algorithm_dict)

    algorithm.to(device)

    train_minibatches_iterator = zip(*train_loaders)
    checkpoint_vals = collections.defaultdict(lambda: [])

    steps_per_epoch = min([len(env)/hparams['batch_size'] for env,_ in in_splits])

    n_steps = args.steps or dataset.N_STEPS
    checkpoint_freq = args.checkpoint_freq or dataset.CHECKPOINT_FREQ

    # NOTE: updated to include flexible noise level for different datasets
    if hparams['flip_prob'] is not None:
        args.dataset = args.dataset + '&' + str(hparams['flip_prob'])

    last_results_keys = None
    algorithm.eval()
    for step in range(start_step, n_steps):
        step_start_time = time.time()

        # minibatches_device = [(x.to(device), y.to(device))
        # for x,y in next(eval_loaders[0])]
        evals = zip(eval_loader_names, eval_loaders, eval_weights)
        for name, loader, weights in evals:
            print(name)
            acc = misc.accuracy(algorithm, loader, weights, device)
            print(acc)
            break

        for x, y in eval_loaders[0]:
            x = x.to(device)
            # all_x = torch.cat([x for x, y in minibatches_device])
            p = algorithm.predict(x)[:32]
            
            print(p)
            prob = torch.nn.functional.softmax(p, dim=1)
            print(prob)
            break

        
        acc = misc.accuracy(algorithm, train_loaders[0], None, device)
        print(acc)
        for x, y in train_loaders[0]:
            x = x.to(device)
            p = algorithm.predict(x)[:32]
            
            print(p)
            prob = torch.nn.functional.softmax(p, dim=1)
            print(prob)
            break
        
        break
