import os
import numpy as np
import pandas as pd
from tqdm import tqdm
from random import Random
from collections import defaultdict

import torch
from torch_geometric.data import Data, InMemoryDataset
from torch_geometric.data import DataLoader

from rdkit import Chem
from rdkit.Chem.BRICS import FindBRICSBonds
from rdkit.Chem.Scaffolds import MurckoScaffold
from rdkit import RDLogger

from utils import get_task_names

RDLogger.DisableLog('rdApp.*')


# -------------------------------------
# attentive_fp fashion featurization
# -------------------------------------
def onehot_encoding(x, allowable_set):
    if x not in allowable_set:
        raise Exception("input {0} not in allowable set{1}:".format(
            x, allowable_set))
    return [x == s for s in allowable_set]


def onehot_encoding_unk(x, allowable_set):
    """Maps inputs not in the allowable set to the last element."""
    if x not in allowable_set:
        x = allowable_set[-1]
    return [x == s for s in allowable_set]


def atom_attr(mol, explicit_H=False, use_chirality=False, pharmaco=False, scaffold=False):
    if pharmaco:
        mol = tag_pharmacophore(mol)
    if scaffold:
        mol = tag_scaffold(mol)

    feat = []
    for i, atom in enumerate(mol.GetAtoms()):
        results = onehot_encoding_unk(atom.GetSymbol(), ["C", "N", "O", "F", "Cl", "Br", "P", "S", "B", "I", "Unk"]) 
        feat.append(results)

    return np.array(feat)


def bond_attr(mol, use_chirality=True):
    feat = []
    index = []
    n = mol.GetNumAtoms()
    for i in range(n):
        for j in range(n):
            if i != j:
                bond = mol.GetBondBetweenAtoms(i, j)
                if bond is not None:
                    bt = bond.GetBondType()
                    bond_feats = [
                        bt == Chem.rdchem.BondType.SINGLE, bt == Chem.rdchem.BondType.DOUBLE,
                        bt == Chem.rdchem.BondType.TRIPLE, bt == Chem.rdchem.BondType.AROMATIC,
                        bond.GetIsConjugated(),
                        bond.IsInRing()
                    ]
                    if use_chirality:
                        bond_feats = bond_feats + onehot_encoding_unk(
                            str(bond.GetStereo()),
                            ["STEREONONE", "STEREOANY", "STEREOZ", "STEREOE"])
                    feat.append(bond_feats)
                    index.append([i, j])

    # Handle empty edge case - ensure we return proper 2D arrays
    if len(index) == 0:
        # Return empty arrays with correct shapes
        return np.empty((0, 2), dtype=np.int64), np.empty((0, len(bond_feats) if 'bond_feats' in locals() else 10), dtype=np.float32)
    
    return np.array(index), np.array(feat)

def bond_break(mol):
    results = np.array(sorted(list(FindBRICSBonds(mol))), dtype=np.int32)

    if results.size == 0:
        cluster_idx = []
        Chem.rdmolops.GetMolFrags(mol, asMols=True, frags=cluster_idx)
        fra_edge_index, fra_edge_attr = bond_attr(mol)

    else:
        bond_to_break = results[:, 0, :]
        bond_to_break = bond_to_break.tolist()
        with Chem.RWMol(mol) as rwmol:
            for i in bond_to_break:
                rwmol.RemoveBond(*i)
        rwmol = rwmol.GetMol()
        cluster_idx = []
        Chem.rdmolops.GetMolFrags(rwmol, asMols=True, sanitizeFrags=False, frags=cluster_idx)
        fra_edge_index, fra_edge_attr = bond_attr(rwmol)
        cluster_idx = torch.LongTensor(cluster_idx)

    return fra_edge_index, fra_edge_attr, cluster_idx


# ---------------------------------------------
# Scaffold and pharmacophore information utils
# ---------------------------------------------
# tag pharmoco features to each atom
fun_smarts = {
        'Hbond_donor': '[$([N;!H0;v3,v4&+1]),$([O,S;H1;+0]),n&H1&+0]',
        'Hbond_acceptor': '[$([O,S;H1;v2;!$(*-*=[O,N,P,S])]),$([O,S;H0;v2]),$([O,S;-]),$([N;v3;!$(N-*=[O,N,P,S])]),n&X2&H0&+0,$([o,s;+0;!$([o,s]:n);!$([o,s]:c:n)])]',
        'Basic': '[#7;+,$([N;H2&+0][$([C,a]);!$([C,a](=O))]),$([N;H1&+0]([$([C,a]);!$([C,a](=O))])[$([C,a]);!$([C,a](=O))]),$([N;H0&+0]([C;!$(C(=O))])([C;!$(C(=O))])[C;!$(C(=O))]),$([n;X2;+0;-0])]',
        'Acid': '[C,S](=[O,S,P])-[O;H1,-1]',
        'Halogen': '[F,Cl,Br,I]'
        }
FunQuery = dict([(pharmaco, Chem.MolFromSmarts(s)) for (pharmaco, s) in fun_smarts.items()])


def tag_pharmacophore(mol):
    for fungrp, qmol in FunQuery.items():
        matches = mol.GetSubstructMatches(qmol)
        match_idxes = []
        for mat in matches:
            match_idxes.extend(mat)
        for i, atom in enumerate(mol.GetAtoms()):
            tag = '1' if i in match_idxes else '0'
            atom.SetProp(fungrp, tag)
    return mol


# tag scaffold information to each atom
def tag_scaffold(mol):
    core = MurckoScaffold.GetScaffoldForMol(mol)
    match_idxes = mol.GetSubstructMatch(core)
    for i, atom in enumerate(mol.GetAtoms()):
        tag = '1' if i in match_idxes else '0'
        atom.SetProp('Scaffold', tag)
    return mol


# ---------------------------------
# data and dataset
# ---------------------------------
class MolData(Data):
    def __init__(self, fra_edge_index=None, fra_edge_attr=None, cluster_index=None, **kwargs):
        super(MolData, self).__init__(**kwargs)
        self.cluster_index = cluster_index
        self.fra_edge_index = fra_edge_index
        self.fra_edge_attr = fra_edge_attr

    def __inc__(self, key, value, *args, **kwargs):
        if key == 'cluster_index':
            return int(self.cluster_index.max()) + 1
        else:
            return super().__inc__(key, value, *args, **kwargs)


class MolDataset(InMemoryDataset):

    def __init__(self, root, dataset, task_type, tasks, logger=None,
                 transform=None, pre_transform=None, pre_filter=None):

        self.tasks = tasks
        self.dataset = dataset
        self.task_type = task_type
        self.logger = logger

        super(MolDataset, self).__init__(root, transform, pre_transform, pre_filter)
        self.data, self.slices = torch.load(self.processed_paths[0])

    @property
    def raw_file_names(self):
        return ['{}.csv'.format(self.dataset)]

    @property
    def processed_file_names(self):
        return ['{}.pt'.format(self.dataset)]

    def download(self):
        pass

    def process(self):
        df = pd.read_csv("/path to data/")
        smilesList = df.smiles.values
        self.logger.info(f'number of all smiles: {len(smilesList)}')
        remained_smiles = []
        canonical_smiles_list = []
        for smiles in smilesList:
            try:
                canonical_smiles_list.append(Chem.MolToSmiles(Chem.MolFromSmiles(smiles), isomericSmiles=False))
                remained_smiles.append(smiles)
            except:
                self.logger.info(f'not successfully processed smiles: {smiles}')
                pass
        self.logger.info(f'number of successfully processed smiles: {len(remained_smiles)}')

        df = df[df["smiles"].isin(remained_smiles)].reset_index()
        target = df[self.tasks].values
        smilesList = df.smiles.values
        data_list = []

        for i, smi in enumerate(tqdm(smilesList)):

            mol = Chem.MolFromSmiles(smi)
            data = self.mol2graph(mol)

            if data is not None:
                label = target[i]
                label[np.isnan(label)] = 666
                data.y = torch.LongTensor([label])
                if self.task_type == 'regression':
                    data.y = torch.FloatTensor([label])
                data_list.append(data)

        if self.pre_filter is not None:
            data_list = [data for data in data_list if self.pre_filter(data)]
        if self.pre_transform is not None:
            data_list = [self.pre_transform(data) for data in data_list]

        data, slices = self.collate(data_list)
        torch.save((data, slices), self.processed_paths[0])

    def mol2graph(self, mol):
        smiles = Chem.MolToSmiles(mol)
        if mol is None: return None
        node_attr = atom_attr(mol)
        edge_index, edge_attr = bond_attr(mol)
        fra_edge_index, fra_edge_attr, cluster_index = bond_break(mol)
        data = MolData(
            x=torch.FloatTensor(node_attr),
            edge_index=torch.LongTensor(edge_index).t(),
            edge_attr=torch.FloatTensor(edge_attr),
            fra_edge_index=torch.LongTensor(fra_edge_index).t(),
            fra_edge_attr=torch.FloatTensor(fra_edge_attr),
            cluster_index=torch.LongTensor(cluster_index),
            y=None,
            smiles=smiles,
        )
        return data

# ---------------------------------
# load dataset
# ---------------------------------
SYNTHETIC_DATASET = ["X", "P", "B", "rings-count", "rings-max", "indole", "PAINS"]
REAL_DATASET = ["cyp", "herg_K", "sol"]


def build_dataset_(kwargs:dict) -> tuple:
    """
    Build the dataset based on the provided arguments.

    Args:
        kwargs (dict): Arguments to configure the dataset.
    Returns:
        tuple: Training, validation, and test splits of the dataset.
    """
    from tdc.single_pred import HTS, ADME, QM
    from tdc.single_pred import Tox

    if kwargs["data_set"] == "sol":
        data = ADME(name='Solubility_AqSolDB',path="//path to data//")
        split = data.get_split(seed=kwargs['split'])
    elif kwargs["data_set"] == "cyp":
        data = ADME(name='CYP2C9_Veith',path="//path to data//")
        split = data.get_split(seed=kwargs['split'],)
    elif kwargs["data_set"] == "herg":
        data= Tox(name = 'hERG',path="/path to data//")
        split = data.get_split(seed=kwargs['split'])
    elif kwargs["data_set"] == "herg_K":
        data= Tox(name = 'hERG_Karim',path="/path to data/")
        split = data.get_split(seed=kwargs['split'])
    elif kwargs["data_set"] in [
        "rings-count",
        "rings-max",
        "X",
        "P",
        "B",
        "indole",
        "PAINS",
    ]:
        data = pd.read_csv("/path to data.csv")
        print(data.head())
        data=data.rename(columns={"smiles": "Drug", kwargs['data_set']: "Y"})
        
        train = data[data[f"split_{kwargs['split']}"] == 'train']
        valid = data[data[f"split_{kwargs['split']}"] == 'valid']
        test = data[data[f"split_{kwargs['split']}"] == 'test']

        split = {}
        split['train'] = train
        split['valid'] = valid
        split['test'] = test
    else:
        raise ValueError("Invalid dataset name")
    print("Dataset:", kwargs["data_set"])
    return split['train'], split['valid'], split['test']

class CustomDataset(torch.utils.data.Dataset):
    def __init__(self, data_list):
        super().__init__()
        self.data_list = data_list
        self.weights = None
        
    def len(self):
        return len(self.data_list)
    
    # Add proper __len__ method for PyTorch DataLoader
    def __len__(self):
        return len(self.data_list)
        
    def get(self, idx):
        return self.data_list[idx]
    
    def __getitem__(self, idx):
        return self.data_list[idx]

def load_dataset_random(path, dataset, seed, task_type, tasks=None, logger=None, split=0):
    save_path = f"/dataset_path" + 'processed/train_valid_test_{}_seed_{}_split{}.ckpt'.format(dataset, seed, split)
    if os.path.isfile(save_path):
        trn, val, test = torch.load(save_path)
        return trn, val, test
    
    # Use the new build_dataset function to get data splits
    kwargs = {
        "data_set": dataset,
        "split": split if split else 0
    }
    
    train_df, valid_df, test_df = build_dataset_(kwargs)
    
    if logger:
        logger.info(f'Using build_dataset function with splits. Total rows: '
                    f'train={len(train_df)}, valid={len(valid_df)}, test={len(test_df)}')
    
    # Process datasets to PyG format
    train_data_list, valid_data_list, test_data_list = [], [], []
    
    # Helper function to process dataframes to PyG data
    def process_dataframe(df, data_list):
        for _, row in df.iterrows():
            smi = row['Drug'] if 'Drug' in df.columns else row['smiles']
            mol = Chem.MolFromSmiles(smi)
            if mol is not None:
                # Convert molecule to PyG data format
                node_attr = atom_attr(mol)
                edge_index, edge_attr = bond_attr(mol)
                fra_edge_index, fra_edge_attr, cluster_index = bond_break(mol)
                
                # Get label data
                if task_type == 'classification':
                    y = torch.LongTensor([[int(row['Y'])]])
                else:
                    y = torch.FloatTensor([[(float(row['Y'])+ 2.86)/2.38]])
                
                data = MolData(
                    x=torch.FloatTensor(node_attr),
                    edge_index=torch.LongTensor(edge_index).t(),
                    edge_attr=torch.FloatTensor(edge_attr),
                    fra_edge_index=torch.LongTensor(fra_edge_index).t(),
                    fra_edge_attr=torch.FloatTensor(fra_edge_attr),
                    cluster_index=torch.LongTensor(cluster_index),
                    y=y,
                    smiles=Chem.MolToSmiles(mol)
                )
                
                data_list.append(data)
    
    # Process each dataset
    process_dataframe(train_df, train_data_list)
    process_dataframe(valid_df, valid_data_list)
    process_dataframe(test_df, test_data_list)
    
    if logger:
        logger.info(f'Total processed molecules = '
                    f'{len(train_data_list) + len(valid_data_list) + len(test_data_list):,} | '
                    f'train molecules = {len(train_data_list):,} | '
                    f'val molecules = {len(valid_data_list):,} | '
                    f'test molecules = {len(test_data_list):,}')
    
    # Create final datasets using the top-level CustomDataset class
    trn = CustomDataset(train_data_list)
    val = CustomDataset(valid_data_list)
    test = CustomDataset(test_data_list)
    
    # Calculate weights for classification tasks
    if task_type == 'classification' and train_data_list:
        weights = []
        # Single classification task
        all_y = torch.cat([data.y.view(-1) for data in train_data_list])
        validId = torch.where((all_y == 0) | (all_y == 1))[0]
        if len(validId) > 0:
            pos_len = all_y[validId].sum().item()
            neg_len = len(validId) - pos_len
            if neg_len > 0 and pos_len > 0:
                weights.append([(neg_len + pos_len) / neg_len, (neg_len + pos_len) / pos_len])
            else:
                weights.append([1.0, 1.0])
        else:
            weights.append([1.0, 1.0])
    
        trn.weights = weights
    else:
        trn.weights = None

    # Save the datasets
    try:
        # Ensure directory exists
        os.makedirs(os.path.dirname(save_path), exist_ok=True)
        torch.save([trn, val, test], save_path)
    except Exception as e:
        if logger:
            logger.warning(f"Failed to save datasets: {str(e)}")
    
    return trn, val, test


# ---------------------------------------------
# Scaffold utils, copy from chemprop.
# ---------------------------------------------
def generate_scaffold(mol, include_chirality=False):
    """
    Computes the Bemis-Murcko scaffold for a SMILES string.
    :param mol: A SMILES or an RDKit molecule.
    :param include_chirality: Whether to include chirality in the computed scaffold..
    :return: The Bemis-Murcko scaffold for the molecule.
    """
    mol = Chem.MolFromSmiles(mol) if type(mol) == str else mol
    scaffold = MurckoScaffold.MurckoScaffoldSmiles(mol=mol, includeChirality=include_chirality)

    return scaffold


def scaffold_to_smiles(smiles, use_indices=False):
    """
    Computes the scaffold for each SMILES and returns a mapping from scaffolds to sets of smiles (or indices).
    :param smiles: A list of SMILES or RDKit molecules.
    :param use_indices: Whether to map to the SMILES's index in :code:`mols` rather than
                        mapping to the smiles string itself. This is necessary if there are duplicate smiles.
    :return: A dictionary mapping each unique scaffold to all SMILES (or indices) which have that scaffold.
    """
    scaffolds = defaultdict(set)
    for i, smi in enumerate(smiles):
        scaffold = generate_scaffold(smi)
        if use_indices:
            scaffolds[scaffold].add(i)
        else:
            scaffolds[scaffold].add(smi)

    return scaffolds


def scaffold_split(pyg_dataset, task_type, tasks, sizes=(0.8, 0.1, 0.1), balanced=True, seed=1, logger=None):

    assert sum(sizes) == 1

    # Split
    logger.info('generating scaffold......')
    num = len(pyg_dataset)
    train_size, val_size, test_size = sizes[0] * num, sizes[1] * num, sizes[2] * num
    train_ids, val_ids, test_ids = [], [], []
    train_scaffold_count, val_scaffold_count, test_scaffold_count = 0, 0, 0

    # Map from scaffold to index in the data
    scaffold_to_indices = scaffold_to_smiles(pyg_dataset.data.smiles, use_indices=True)

    # Seed randomness
    random = Random(seed)

    if balanced:  # Put stuff that's bigger than half the val/test size into train, rest just order randomly
        index_sets = list(scaffold_to_indices.values())
        big_index_sets = []
        small_index_sets = []
        for index_set in index_sets:
            if len(index_set) > val_size / 2 or len(index_set) > test_size / 2:
                big_index_sets.append(index_set)
            else:
                small_index_sets.append(index_set)
        random.seed(seed)
        random.shuffle(big_index_sets)
        random.shuffle(small_index_sets)
        index_sets = big_index_sets + small_index_sets
    else:  # Sort from largest to smallest scaffold sets
        index_sets = sorted(list(scaffold_to_indices.values()),
                            key=lambda index_set: len(index_set),
                            reverse=True)

    for index_set in index_sets:
        if len(train_ids) + len(index_set) <= train_size:
            train_ids += index_set
            train_scaffold_count += 1
        elif len(val_ids) + len(index_set) <= val_size:
            val_ids += index_set
            val_scaffold_count += 1
        else:
            test_ids += index_set
            test_scaffold_count += 1

    logger.info(f'Total scaffolds = {len(scaffold_to_indices):,} | '
                 f'train scaffolds = {train_scaffold_count:,} | '
                 f'val scaffolds = {val_scaffold_count:,} | '
                 f'test scaffolds = {test_scaffold_count:,}')

    logger.info(f'Total smiles = {num:,} | '
                 f'train smiles = {len(train_ids):,} | '
                 f'val smiles = {len(val_ids):,} | '
                 f'test smiles = {len(test_ids):,}')

    assert len(train_ids) + len(val_ids) + len(test_ids) == len(pyg_dataset)

    # Compute weights
    assert task_type == 'classification' or 'regression'
    if task_type == 'classification':
        weights = []
        for i in range(len(tasks)):
            pos_len = (pyg_dataset.data.y[:, i].sum()).item()
            neg_len = len(pyg_dataset) - pos_len
            weights.append([(neg_len + pos_len) / neg_len, (neg_len + pos_len) / pos_len])

    else:
        weights = None

    return train_ids, val_ids, test_ids, weights


# ---------------------------------
# build dataset and dataloader
# ---------------------------------
def build_dataset(cfg, logger):

    cfg.defrost()
    if cfg.DATA.DATASET in SYNTHETIC_DATASET:
        cfg.DATA.TASK_TYPE = 'classification'
        cfg.DATA.DATA_PATH = './data/data.csv/'
    task_name = "Y"
    if cfg.DATA.TASK_TYPE == 'classification':
        out_dim = 2
    elif cfg.DATA.TASK_TYPE == 'regression':
        out_dim =1
    else:
        raise Exception('Unknown task type')
    opts = ['DATA.TASK_NAME', cfg.DATA.DATASET, 'MODEL.OUT_DIM', out_dim]
    cfg.defrost()
    cfg.merge_from_list(opts)
    cfg.freeze()

    train_dataset, valid_dataset, test_dataset = load_dataset_random(cfg.DATA.DATA_PATH,
                                                                        cfg.DATA.DATASET,
                                                                        cfg.SEED,
                                                                        cfg.DATA.TASK_TYPE,
                                                                        cfg.DATA.TASK_NAME,
                                                                        logger,
                                                                        split=cfg.DATA.SPLIT
                                                                        )


    return train_dataset, valid_dataset, test_dataset


def build_loader(cfg, logger):
    train_dataset, valid_dataset, test_dataset = build_dataset(cfg, logger)
    train_dataloader = DataLoader(train_dataset, batch_size=cfg.DATA.BATCH_SIZE, shuffle=True)
    valid_dataloader = DataLoader(valid_dataset, batch_size=cfg.DATA.BATCH_SIZE)
    test_dataloader = DataLoader(test_dataset, batch_size=cfg.DATA.BATCH_SIZE)
    weights = train_dataset.weights

    return train_dataloader, valid_dataloader, test_dataloader, weights

