'''ResNet in PyTorch.
For Pre-activation ResNet, see 'preact_resnet.py'.
Reference:
[1] Kaiming He, Xiangyu Zhang, Shaoqing Ren, Jian Sun
    Deep Residual Learning for Image Recognition. arXiv:1512.03385
'''
from functools import partial
import torch
import torch.nn as nn
import torch.nn.functional as F
from acat.utils import print_once


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, in_planes, planes, stride=1, use_activ=True,batch_norm=True,celu=True, **kwargs):
        super(BasicBlock, self).__init__()
        self.use_activ = use_activ
        act_func=nn.CELU if celu else nn.ReLU
        self.activ = partial(act_func(inplace=True)) if self.use_activ else lambda x: x
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        groups=min(8,max(planes//4,1))
        self.bn1 = nn.BatchNorm2d(planes) if batch_norm else nn.GroupNorm(groups,planes)
        print_once(f"Type norm:{type(self.bn1)}",key="basic_1")
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes) if batch_norm else nn.GroupNorm(groups,planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            groups = min(8, max(planes*self.expansion // 4, 1))
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes, kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion*planes) if batch_norm else nn.GroupNorm(groups,planes*self.expansion)
            )

    def forward(self, x):
        out = self.activ(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out += self.shortcut(x)
        out = self.activ(out)
        return out


class Bottleneck(nn.Module):
    expansion = 4

    def __init__(self, in_planes, planes, stride=1, use_activ=True,batch_norm=True,celu=True):
        super(Bottleneck, self).__init__()
        self.use_activ = use_activ
        act_func=nn.CELU if celu else nn.ReLU
        self.activ = partial(act_func(inplace=True)) if self.use_activ else lambda x: x
        groups=min(8,max(planes//4,1))
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes) if batch_norm else nn.GroupNorm(groups,planes)
        print_once(f"Type norm:{type(self.bn1)}",key="bottle_1")
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)if batch_norm else nn.GroupNorm(groups,planes)
        self.conv3 = nn.Conv2d(planes, self.expansion*planes, kernel_size=1, bias=False)
        groups=min(8,max(planes*self.expansion//4,1))
        self.bn3 = nn.BatchNorm2d(self.expansion*planes)if batch_norm else nn.GroupNorm(groups,self.expansion*planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion*planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion*planes, kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion*planes) if batch_norm else nn.GroupNorm(groups,self.expansion*planes)
            )

    def forward(self, x):
        out = self.activ(self.bn1(self.conv1(x)))
        out = self.activ(self.bn2(self.conv2(out)))
        out = self.bn3(self.conv3(out))
        out += self.shortcut(x)
        out = self.activ(out)
        return out


class ResNet(nn.Module):
    def __init__(self, block, num_blocks, num_classes=10, ch_in=3, pool_adapt=False,batch_norm=True, **kwargs):
        super(ResNet, self).__init__()
        self.in_planes = 64
        self.pool_adapt = pool_adapt
        if pool_adapt:
            self.avg_pool = nn.AdaptiveAvgPool2d((1, 1))
        else:
            self.avg_pool = partial(F.avg_pool2d, kernel_size=4)

        self.conv1 = nn.Conv2d(ch_in, 64, kernel_size=3, stride=1, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(64) if batch_norm else nn.GroupNorm(8,64)
        self.layer1 = self._make_layer(block, 64, num_blocks[0], stride=1,batch_norm=batch_norm, **kwargs)
        self.layer2 = self._make_layer(block, 128, num_blocks[1], stride=2,batch_norm=batch_norm, **kwargs)
        self.layer3 = self._make_layer(block, 256, num_blocks[2], stride=2,batch_norm=batch_norm, **kwargs)
        self.layer4 = self._make_layer(block, 512, num_blocks[3], stride=2,batch_norm=batch_norm, **kwargs)
        self.linear = nn.Linear(512*block.expansion, num_classes)

    def _make_layer(self, block, planes, num_blocks, stride, **kwargs):
        strides = [stride] + [1]*(num_blocks-1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, stride, **kwargs))
            self.in_planes = planes * block.expansion
        # # cheeky way to get the activation from the layer1, e.g. in no activation case.
        self.activ = layers[0].activ
        return nn.Sequential(*layers)

    def forward(self, x):
        out = self.activ(self.bn1(self.conv1(x)))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = self.avg_pool(out)
        out = out.view(out.size(0), -1)
        out = self.linear(out)
        return out

def ResNetnew(num_blocks=None, **kwargs):
    if num_blocks is None:
        num_blocks = [1, 1, 1, 1]
    return ResNet(BasicBlock, num_blocks, **kwargs)

def ResNet18(**kwargs):
    return ResNet(BasicBlock, [2,2,2,2], **kwargs)

def ResNet34(**kwargs):
    return ResNet(BasicBlock, [3,4,6,3], **kwargs)

def ResNet50(**kwargs):
    return ResNet(Bottleneck, [3,4,6,3], **kwargs)

def ResNet101(**kwargs):
    return ResNet(Bottleneck, [3,4,23,3], **kwargs)

def ResNet152(**kwargs):
    return ResNet(Bottleneck, [3,8,36,3], **kwargs)


def test():
    net = ResNet18()
    y = net(torch.randn(1,3,32,32))
    print(y.size())

# test()
