import numpy as np
import torch
import torch.nn as nn
import torch.nn.functional as F

import utils


class RandomShiftsAug(nn.Module):
    def __init__(self, pad):
        super().__init__()
        self.pad = pad

    def forward(self, x):
        n, c, h, w = x.size()
        assert h == w
        padding = tuple([self.pad] * 4)
        x = F.pad(x, padding, 'replicate')
        eps = 1.0 / (h + 2 * self.pad)
        arange = torch.linspace(-1.0 + eps,
                                1.0 - eps,
                                h + 2 * self.pad,
                                device=x.device,
                                dtype=x.dtype)[:h]
        arange = arange.unsqueeze(0).repeat(h, 1).unsqueeze(2)
        base_grid = torch.cat([arange, arange.transpose(1, 0)], dim=2)
        base_grid = base_grid.unsqueeze(0).repeat(n, 1, 1, 1)

        shift = torch.randint(0,
                              2 * self.pad + 1,
                              size=(n, 1, 1, 2),
                              device=x.device,
                              dtype=x.dtype)
        shift *= 2.0 / (h + 2 * self.pad)

        grid = base_grid + shift
        return F.grid_sample(x,
                             grid,
                             padding_mode='zeros',
                             align_corners=False)


class Encoder(nn.Module):
    def __init__(self, obs_shape):
        super().__init__()

        assert len(obs_shape) == 3
        self.repr_dim = 32 * 35 * 35

        if obs_shape[1] == 168:
            kernel_size, stride = 6, 4
        else:
            kernel_size, stride = 3, 2

        self.convnet = nn.Sequential(nn.Conv2d(obs_shape[0], 32, kernel_size, stride=stride),
                                     nn.ReLU(), nn.Conv2d(32, 32, 3, stride=1),
                                     nn.ReLU(), nn.Conv2d(32, 32, 3, stride=1),
                                     nn.ReLU(), nn.Conv2d(32, 32, 3, stride=1),
                                     nn.ReLU())

        self.apply(utils.weight_init)
        tmp_x = torch.zeros(1, obs_shape[0], obs_shape[1], obs_shape[2])
        tmp_x = self.convnet(tmp_x)
        tmp_x = tmp_x.view(tmp_x.shape[0], -1)
        # assert tmp_x.shape[1] == self.repr_dim, f"Expected {self.repr_dim}, but got {tmp_x.shape[1]}"
        self.repr_dim = tmp_x.shape[1]
        del tmp_x

    def forward(self, obs):
        obs = obs / 255.0 - 0.5
        h = self.convnet(obs)
        h = h.view(h.shape[0], -1)
        return h


class Actor(nn.Module):
    def __init__(self, repr_dim, action_shape, feature_dim, hidden_dim):
        super().__init__()

        self.trunk = nn.Sequential(nn.Linear(repr_dim, feature_dim),
                                   nn.LayerNorm(feature_dim), nn.Tanh())

        self.policy = nn.Sequential(nn.Linear(feature_dim, hidden_dim),
                                    nn.ReLU(inplace=True),
                                    nn.Linear(hidden_dim, hidden_dim),
                                    nn.ReLU(inplace=True),
                                    nn.Linear(hidden_dim, action_shape[0]))

        self.apply(utils.weight_init)

    def forward(self, obs, std):
        h = self.trunk(obs)

        mu = self.policy(h)
        mu = torch.tanh(mu)
        std = torch.ones_like(mu) * std

        dist = utils.TruncatedNormal(mu, std)
        return dist


class Critic(nn.Module):
    def __init__(self, repr_dim, action_shape, feature_dim, hidden_dim):
        super().__init__()

        self.trunk = nn.Sequential(nn.Linear(repr_dim, feature_dim),
                                   nn.LayerNorm(feature_dim), nn.Tanh())

        self.Q1 = nn.Sequential(
            nn.Linear(feature_dim + action_shape[0], hidden_dim),
            nn.ReLU(inplace=True), nn.Linear(hidden_dim, hidden_dim),
            nn.ReLU(inplace=True), nn.Linear(hidden_dim, 1))

        self.Q2 = nn.Sequential(
            nn.Linear(feature_dim + action_shape[0], hidden_dim),
            nn.ReLU(inplace=True), nn.Linear(hidden_dim, hidden_dim),
            nn.ReLU(inplace=True), nn.Linear(hidden_dim, 1))

        self.apply(utils.weight_init)

    def forward(self, obs, action):
        h = self.trunk(obs)
        h_action = torch.cat([h, action], dim=-1)
        q1 = self.Q1(h_action)
        q2 = self.Q2(h_action)

        return q1, q2


class DrQV2Agent:
    def __init__(self, obs_shape, action_shape, args):
        self.critic_tau = args.critic_tau
        self.num_expl_steps = args.num_expl_steps
        self.stddev_schedule = args.stddev_schedule
        self.stddev_clip = args.stddev_clip
        self.n_step = args.n_step
        self.discount = args.discount
        # models
        self.encoder = Encoder(obs_shape).cuda()
        self.actor = Actor(self.encoder.repr_dim, action_shape, args.projection_dim,
                           args.hidden_dim).cuda()

        self.critic = Critic(self.encoder.repr_dim, action_shape, args.projection_dim,
                             args.hidden_dim).cuda()
        self.critic_target = Critic(self.encoder.repr_dim, action_shape,
                                    args.projection_dim, args.hidden_dim).cuda()
        self.critic_target.load_state_dict(self.critic.state_dict())

        # optimizers
        self.encoder_opt = torch.optim.Adam(self.encoder.parameters(), lr=args.selector_lr)
        self.actor_opt = torch.optim.Adam(self.actor.parameters(), lr=args.actor_lr)
        self.critic_opt = torch.optim.Adam(self.critic.parameters(), lr=args.critic_lr)

        # data augmentation
        self.aug = RandomShiftsAug(pad=4)

        self.train()
        self.critic_target.train()

    def train(self, training=True):
        self.training = training
        self.encoder.train(training)
        self.actor.train(training)
        self.critic.train(training)

    def _obs_to_input(self, obs):
        if isinstance(obs, utils.LazyFrames):
            _obs = np.array(obs)
        else:
            _obs = obs
        _obs = torch.FloatTensor(_obs).cuda()
        if len(_obs.shape) == 3:
            _obs = _obs.unsqueeze(0)
        return _obs

    def select_action(self, obs, step=0, eval_mode=False):
        # obs = torch.as_tensor(obs, device=self.device)
        obs = self._obs_to_input(obs)
        obs = self.encoder(obs)
        stddev = utils.schedule(self.stddev_schedule, step)
        dist = self.actor(obs, stddev)
        if eval_mode:
            action = dist.mean
        else:
            action = dist.sample(clip=None)
            if step < self.num_expl_steps:
                action.uniform_(-1.0, 1.0)
        return action.cpu().data.numpy()[0]

    def update_critic(self, obs, action, reward, discount, next_obs, L=None, step=None):
        metrics = dict()

        with torch.no_grad():
            stddev = utils.schedule(self.stddev_schedule, step)
            dist = self.actor(next_obs, stddev)
            next_action = dist.sample(clip=self.stddev_clip)
            target_Q1, target_Q2 = self.critic_target(next_obs, next_action)
            target_V = torch.min(target_Q1, target_Q2)
            target_Q = reward + (discount * target_V)

        Q1, Q2 = self.critic(obs, action)
        critic_loss = F.mse_loss(Q1, target_Q) + F.mse_loss(Q2, target_Q)

        # if self.use_tb:
        #     metrics['critic_target_q'] = target_Q.mean().item()
        #     metrics['critic_q1'] = Q1.mean().item()
        #     metrics['critic_q2'] = Q2.mean().item()
        #     metrics['critic_loss'] = critic_loss.item()
        if L is not None:
            L.log('train_critic/loss', critic_loss, step)
        # optimize encoder and critic
        self.encoder_opt.zero_grad(set_to_none=True)
        self.critic_opt.zero_grad(set_to_none=True)
        critic_loss.backward()
        self.critic_opt.step()
        self.encoder_opt.step()

    def update_actor(self, obs, L=None, step=None):
        # metrics = dict()

        stddev = utils.schedule(self.stddev_schedule, step)
        dist = self.actor(obs, stddev)
        action = dist.sample(clip=self.stddev_clip)
        log_prob = dist.log_prob(action).sum(-1, keepdim=True)
        Q1, Q2 = self.critic(obs, action)
        Q = torch.min(Q1, Q2)

        actor_loss = -Q.mean()

        # optimize actor
        self.actor_opt.zero_grad(set_to_none=True)
        actor_loss.backward()
        self.actor_opt.step()

        # if self.use_tb:
        #     metrics['actor_loss'] = actor_loss.item()
        #     metrics['actor_logprob'] = log_prob.mean().item()
        #     metrics['actor_ent'] = dist.entropy().sum(dim=-1).mean().item()
        if L is not None:
            L.log('train_actor/loss', actor_loss, step)

    def update(self, replay_buffer, L, step):
        obs, action, reward, discount, next_obs = replay_buffer.sample_drqv2(n_step=self.n_step, discount=self.discount)

        # batch = next(replay_iter)

        # obs, action, reward, discount, next_obs = utils.to_torch(
        #     batch, self.device)

        # augment
        obs = self.aug(obs.float())
        next_obs = self.aug(next_obs.float())
        # encode
        obs = self.encoder(obs)
        with torch.no_grad():
            next_obs = self.encoder(next_obs)

        # if self.use_tb:
        #     metrics['batch_reward'] = reward.mean().item()

        # update

        # update critic

        self.update_critic(obs, action, reward, discount, next_obs, L, step)

        # update actor
        self.update_actor(obs.detach(), L, step)

        # update critic target
        utils.soft_update_params(self.critic, self.critic_target,
                                 self.critic_tau)
