from functools import partial

import numpy as np

from ...utils import common_utils
from . import augmentor_utils, database_sampler


class DataAugmentor(object):
    def __init__(self, root_path, augmentor_configs, class_names, logger=None):
        self.root_path = root_path
        self.class_names = class_names
        self.logger = logger

        self.data_augmentor_queue = []
        aug_config_list = (
            augmentor_configs
            if isinstance(augmentor_configs, list)
            else augmentor_configs.AUG_CONFIG_LIST
        )

        for cur_cfg in aug_config_list:
            if not isinstance(augmentor_configs, list):
                if cur_cfg.NAME in augmentor_configs.DISABLE_AUG_LIST:
                    continue
            cur_augmentor = getattr(self, cur_cfg.NAME)(config=cur_cfg)
            self.data_augmentor_queue.append(cur_augmentor)

    def only_gt_sampling(self, config=None):
        db_sampler = database_sampler.DataBaseSamplerGt(
            root_path=self.root_path,
            sampler_cfg=config,
            class_names=self.class_names,
            logger=self.logger,
        )
        return db_sampler

    def gt_sampling(self, config=None):
        db_sampler = database_sampler.DataBaseSampler(
            root_path=self.root_path,
            sampler_cfg=config,
            class_names=self.class_names,
            logger=self.logger,
        )
        return db_sampler

    def __getstate__(self):
        d = dict(self.__dict__)
        del d["logger"]
        return d

    def __setstate__(self, d):
        self.__dict__.update(d)

    def random_world_flip(self, data_dict=None, config=None):
        if data_dict is None:
            return partial(self.random_world_flip, config=config)
        gt_boxes, points = data_dict["gt_boxes"], data_dict["points"]
        for cur_axis in config["ALONG_AXIS_LIST"]:
            assert cur_axis in ["x", "y"]
            gt_boxes, points = getattr(
                augmentor_utils, "random_flip_along_%s" % cur_axis
            )(
                gt_boxes,
                points,
            )

        data_dict["gt_boxes"] = gt_boxes
        data_dict["points"] = points
        return data_dict

    def random_world_rotation(self, data_dict=None, config=None):
        if data_dict is None:
            return partial(self.random_world_rotation, config=config)
        rot_range = config["WORLD_ROT_ANGLE"]
        if not isinstance(rot_range, list):
            rot_range = [-rot_range, rot_range]
        gt_boxes, points = augmentor_utils.global_rotation(
            data_dict["gt_boxes"], data_dict["points"], rot_range=rot_range
        )

        data_dict["gt_boxes"] = gt_boxes
        data_dict["points"] = points
        return data_dict

    def random_world_scaling(self, data_dict=None, config=None):
        if data_dict is None:
            return partial(self.random_world_scaling, config=config)
        gt_boxes, points = augmentor_utils.global_scaling(
            data_dict["gt_boxes"], data_dict["points"], config["WORLD_SCALE_RANGE"]
        )
        data_dict["gt_boxes"] = gt_boxes
        data_dict["points"] = points
        return data_dict

    def normalize_object_size(self, data_dict=None, config=None):
        if data_dict is None:
            return partial(self.normalize_object_size, config=config)
        points, gt_boxes = augmentor_utils.normalize_object_size(
            data_dict["gt_boxes"],
            data_dict["points"],
            data_dict["gt_boxes_mask"],
            config["SIZE_RES"],
        )
        data_dict["gt_boxes"] = gt_boxes
        data_dict["points"] = points
        return data_dict

    def random_image_flip(self, data_dict=None, config=None):
        if data_dict is None:
            return partial(self.random_image_flip, config=config)
        images = data_dict["images"]
        depth_maps = data_dict["depth_maps"]
        gt_boxes = data_dict["gt_boxes"]
        gt_boxes2d = data_dict["gt_boxes2d"]
        calib = data_dict["calib"]
        for cur_axis in config["ALONG_AXIS_LIST"]:
            assert cur_axis in ["horizontal"]
            images, depth_maps, gt_boxes = getattr(
                augmentor_utils, "random_image_flip_%s" % cur_axis
            )(
                images,
                depth_maps,
                gt_boxes,
                calib,
            )

        data_dict["images"] = images
        data_dict["depth_maps"] = depth_maps
        data_dict["gt_boxes"] = gt_boxes
        return data_dict

    def random_world_translation(self, data_dict=None, config=None):
        """
        Please check the correctness of it before using.
        """
        if data_dict is None:
            return partial(self.random_world_translation, config=config)
        offset_range = config["WORLD_TRANSLATION_RANGE"]
        gt_boxes, points = data_dict["gt_boxes"], data_dict["points"]
        for cur_axis in config["ALONG_AXIS_LIST"]:
            assert cur_axis in ["x", "y", "z"]
            gt_boxes, points = getattr(
                augmentor_utils, "random_translation_along_%s" % cur_axis
            )(
                gt_boxes,
                points,
                offset_range,
            )

        data_dict["gt_boxes"] = gt_boxes
        data_dict["points"] = points
        return data_dict

    def random_local_translation(self, data_dict=None, config=None):
        """
        Please check the correctness of it before using.
        """
        if data_dict is None:
            return partial(self.random_local_translation, config=config)
        offset_range = config["LOCAL_TRANSLATION_RANGE"]
        gt_boxes, points = data_dict["gt_boxes"], data_dict["points"]
        for cur_axis in config["ALONG_AXIS_LIST"]:
            assert cur_axis in ["x", "y", "z"]
            gt_boxes, points = getattr(
                augmentor_utils, "random_local_translation_along_%s" % cur_axis
            )(
                gt_boxes,
                points,
                offset_range,
            )

        data_dict["gt_boxes"] = gt_boxes
        data_dict["points"] = points
        return data_dict

    def random_local_rotation(self, data_dict=None, config=None):
        """
        Please check the correctness of it before using.
        """
        if data_dict is None:
            return partial(self.random_local_rotation, config=config)
        rot_range = config["LOCAL_ROT_ANGLE"]
        if not isinstance(rot_range, list):
            rot_range = [-rot_range, rot_range]
        gt_boxes, points = augmentor_utils.local_rotation(
            data_dict["gt_boxes"], data_dict["points"], rot_range=rot_range
        )

        data_dict["gt_boxes"] = gt_boxes
        data_dict["points"] = points
        return data_dict

    def random_local_scaling(self, data_dict=None, config=None):
        """
        Please check the correctness of it before using.
        """
        if data_dict is None:
            return partial(self.random_local_scaling, config=config)
        gt_boxes, points = augmentor_utils.local_scaling(
            data_dict["gt_boxes"], data_dict["points"], config["LOCAL_SCALE_RANGE"]
        )

        data_dict["gt_boxes"] = gt_boxes
        data_dict["points"] = points
        return data_dict

    def random_world_frustum_dropout(self, data_dict=None, config=None):
        """
        Please check the correctness of it before using.
        """
        if data_dict is None:
            return partial(self.random_world_frustum_dropout, config=config)

        intensity_range = config["INTENSITY_RANGE"]
        gt_boxes, points = data_dict["gt_boxes"], data_dict["points"]
        for direction in config["DIRECTION"]:
            assert direction in ["top", "bottom", "left", "right"]
            gt_boxes, points = getattr(
                augmentor_utils, "global_frustum_dropout_%s" % direction
            )(
                gt_boxes,
                points,
                intensity_range,
            )

        data_dict["gt_boxes"] = gt_boxes
        data_dict["points"] = points
        return data_dict

    def random_local_frustum_dropout(self, data_dict=None, config=None):
        """
        Please check the correctness of it before using.
        """
        if data_dict is None:
            return partial(self.random_local_frustum_dropout, config=config)

        intensity_range = config["INTENSITY_RANGE"]
        gt_boxes, points = data_dict["gt_boxes"], data_dict["points"]
        for direction in config["DIRECTION"]:
            assert direction in ["top", "bottom", "left", "right"]
            gt_boxes, points = getattr(
                augmentor_utils, "local_frustum_dropout_%s" % direction
            )(
                gt_boxes,
                points,
                intensity_range,
            )

        data_dict["gt_boxes"] = gt_boxes
        data_dict["points"] = points
        return data_dict

    def forward(self, data_dict):
        """
        Args:
            data_dict:
                points: (N, 3 + C_in)
                gt_boxes: optional, (N, 7) [x, y, z, dx, dy, dz, heading]
                gt_names: optional, (N), string
                ...

        Returns:
        """
        for cur_augmentor in self.data_augmentor_queue:
            data_dict = cur_augmentor(data_dict=data_dict)

        data_dict["gt_boxes"][:, 6] = common_utils.limit_period(
            data_dict["gt_boxes"][:, 6], offset=0.5, period=2 * np.pi
        )
        if "calib" in data_dict:
            data_dict.pop("calib")
        if "road_plane" in data_dict:
            data_dict.pop("road_plane")
        if "gt_boxes_mask" in data_dict:
            gt_boxes_mask = data_dict["gt_boxes_mask"]
            data_dict["gt_boxes"] = data_dict["gt_boxes"][gt_boxes_mask]
            data_dict["gt_names"] = data_dict["gt_names"][gt_boxes_mask]
            if "gt_boxes2d" in data_dict:
                data_dict["gt_boxes2d"] = data_dict["gt_boxes2d"][gt_boxes_mask]

            data_dict.pop("gt_boxes_mask")
        return data_dict

    def random_object_rotation(self, data_dict=None, config=None):
        if data_dict is None:
            return partial(self.random_object_rotation, config=config)

        gt_boxes, points = augmentor_utils.rotate_objects(
            data_dict["gt_boxes"],
            data_dict["points"],
            # data_dict["gt_boxes_mask"],
            rotation_perturb=config["ROT_UNIFORM_NOISE"],
            prob=config["ROT_PROB"],
            num_try=50,
        )

        data_dict["gt_boxes"] = gt_boxes
        data_dict["points"] = points
        return data_dict

    def random_object_scaling(self, data_dict=None, config=None):
        if data_dict is None:
            return partial(self.random_object_scaling, config=config)
        points, gt_boxes = augmentor_utils.scale_pre_object(
            data_dict["gt_boxes"],
            data_dict["points"],
            # gt_boxes_mask=data_dict["gt_boxes_mask"],
            scale_perturb=config["SCALE_UNIFORM_NOISE"],
        )

        data_dict["gt_boxes"] = gt_boxes
        data_dict["points"] = points
        return data_dict

    def adjust_augment_intensity(self, config, intensity):
        adjust_map = {
            "random_object_scaling": "SCALE_UNIFORM_NOISE",
            "random_object_rotation": "ROT_UNIFORM_NOISE",
            "random_world_rotation": "WORLD_ROT_ANGLE",
            "random_world_scaling": "WORLD_SCALE_RANGE",
        }

        def cal_new_intensity(config, flag):
            origin_intensity_list = config.get(adjust_map[config.NAME])
            assert len(origin_intensity_list) == 2
            assert np.isclose(
                flag - origin_intensity_list[0], origin_intensity_list[1] - flag
            )

            noise = origin_intensity_list[1] - flag
            new_noise = noise * intensity
            new_intensity_list = [flag - new_noise, new_noise + flag]
            return new_intensity_list

        if config.NAME not in adjust_map:
            return config

        if config.NAME in ["random_object_scaling", "random_world_scaling"]:
            new_intensity_list = cal_new_intensity(config, flag=1)
            setattr(config, adjust_map[config.NAME], new_intensity_list)
            return config
        elif config.NAME in ["random_object_rotation", "random_world_rotation"]:
            new_intensity_list = cal_new_intensity(config, flag=0)
            setattr(config, adjust_map[config.NAME], new_intensity_list)
            return config
        else:
            raise NotImplementedError
