from functools import partial
import torch.nn as nn

from ...utils.spconv_utils import replace_feature, spconv


def coords_flip(coords, grid_size):
    coords[:, 2] = grid_size - 1 - coords[:, 2]
    return coords


def post_act_block(
    in_channels,
    out_channels,
    kernel_size,
    indice_key=None,
    stride=1,
    padding=0,
    conv_type="subm",
    norm_fn=None,
):

    if conv_type == "subm":
        conv = spconv.SubMConv3d(
            in_channels, out_channels, kernel_size, bias=False, indice_key=indice_key
        )
    elif conv_type == "spconv":
        conv = spconv.SparseConv3d(
            in_channels,
            out_channels,
            kernel_size,
            stride=stride,
            padding=padding,
            bias=False,
            indice_key=indice_key,
        )
    elif conv_type == "inverseconv":
        conv = spconv.SparseInverseConv3d(
            in_channels, out_channels, kernel_size, indice_key=indice_key, bias=False
        )
    else:
        raise NotImplementedError

    m = spconv.SparseSequential(
        conv,
        norm_fn(out_channels),
        nn.ReLU(),
    )

    return m


class SparseBasicBlock(spconv.SparseModule):
    expansion = 1

    def __init__(
        self, inplanes, planes, stride=1, norm_fn=None, downsample=None, indice_key=None
    ):
        super(SparseBasicBlock, self).__init__()

        assert norm_fn is not None
        bias = norm_fn is not None
        self.conv1 = spconv.SubMConv3d(
            inplanes,
            planes,
            kernel_size=3,
            stride=stride,
            padding=1,
            bias=bias,
            indice_key=indice_key,
        )
        self.bn1 = norm_fn(planes)
        self.relu = nn.ReLU()
        self.conv2 = spconv.SubMConv3d(
            planes,
            planes,
            kernel_size=3,
            stride=stride,
            padding=1,
            bias=bias,
            indice_key=indice_key,
        )
        self.bn2 = norm_fn(planes)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        identity = x

        out = self.conv1(x)
        out = replace_feature(out, self.bn1(out.features))
        out = replace_feature(out, self.relu(out.features))

        out = self.conv2(out)
        out = replace_feature(out, self.bn2(out.features))

        if self.downsample is not None:
            identity = self.downsample(x)

        out = replace_feature(out, out.features + identity.features)
        out = replace_feature(out, self.relu(out.features))

        return out




class VoxelBackBone8x(nn.Module):
    def __init__(self, model_cfg, input_channels, grid_size, **kwargs):
        super().__init__()
        self.model_cfg = model_cfg
        norm_fn = partial(nn.BatchNorm1d, eps=1e-3, momentum=0.01)

        self.sparse_shape = grid_size[::-1] + [1, 0, 0]

        self.conv_input = spconv.SparseSequential(
            spconv.SubMConv3d(
                input_channels, 16, 3, padding=1, bias=False, indice_key="subm1"
            ),
            norm_fn(16),
            nn.ReLU(),
        )
        block = post_act_block

        self.conv1 = spconv.SparseSequential(
            block(16, 16, 3, norm_fn=norm_fn, padding=1, indice_key="subm1"),
        )

        self.conv2 = spconv.SparseSequential(
            block(
                16,
                32,
                3,
                norm_fn=norm_fn,
                stride=2,
                padding=1,
                indice_key="spconv2",
                conv_type="spconv",
            ),
            block(32, 32, 3, norm_fn=norm_fn, padding=1, indice_key="subm2"),
            block(32, 32, 3, norm_fn=norm_fn, padding=1, indice_key="subm2"),
        )

        self.conv3 = spconv.SparseSequential(
            block(
                32,
                64,
                3,
                norm_fn=norm_fn,
                stride=2,
                padding=1,
                indice_key="spconv3",
                conv_type="spconv",
            ),
            block(64, 64, 3, norm_fn=norm_fn, padding=1, indice_key="subm3"),
            block(64, 64, 3, norm_fn=norm_fn, padding=1, indice_key="subm3"),
        )

        self.conv4 = spconv.SparseSequential(
            block(
                64,
                64,
                3,
                norm_fn=norm_fn,
                stride=2,
                padding=(0, 1, 1),
                indice_key="spconv4",
                conv_type="spconv",
            ),
            block(64, 64, 3, norm_fn=norm_fn, padding=1, indice_key="subm4"),
            block(64, 64, 3, norm_fn=norm_fn, padding=1, indice_key="subm4"),
        )

        last_pad = 0
        last_pad = self.model_cfg.get("last_pad", last_pad)
        self.conv_out = spconv.SparseSequential(
            spconv.SparseConv3d(
                64,
                128,
                (3, 1, 1),
                stride=(2, 1, 1),
                padding=last_pad,
                bias=False,
                indice_key="spconv_down2",
            ),
            norm_fn(128),
            nn.ReLU(),
        )
        self.num_point_features = 128
        self.backbone_channels = {
            "x_conv1": 16,
            "x_conv2": 32,
            "x_conv3": 64,
            "x_conv4": 64,
        }

    def forward(self, batch_dict, **kwargs):
        """
        Args:
            batch_dict:
                batch_size: int
                vfe_features: (num_voxels, C)
                voxel_coords: (num_voxels, 4), [batch_idx, z_idx, y_idx, x_idx]
        Returns:
            batch_dict:
                encoded_spconv_tensor: sparse tensor
        """

        voxel_features, voxel_coords = (
            batch_dict["voxel_features"],
            batch_dict["voxel_coords"],
        )
        batch_size = batch_dict["batch_size"]
        input_sp_tensor = spconv.SparseConvTensor(
            features=voxel_features,
            indices=voxel_coords.int(),
            spatial_shape=self.sparse_shape,
            batch_size=batch_size,
        )

        x = self.conv_input(input_sp_tensor)

        x_conv1 = self.conv1(x)
        x_conv2 = self.conv2(x_conv1)
        x_conv3 = self.conv3(x_conv2)
        x_conv4 = self.conv4(x_conv3)

        out = self.conv_out(x_conv4)

        batch_dict.update(
            {"encoded_spconv_tensor": out, "encoded_spconv_tensor_stride": 8}
        )
        batch_dict.update(
            {
                "multi_scale_3d_features": {
                    "x_conv1": x_conv1,
                    "x_conv2": x_conv2,
                    "x_conv3": x_conv3,
                    "x_conv4": x_conv4,
                    "x_conv_input": x,
                }
            }
        )
        batch_dict.update(
            {
                "multi_scale_3d_strides": {
                    "x_conv1": 1,
                    "x_conv2": 2,
                    "x_conv3": 4,
                    "x_conv4": 8,
                }
            }
        )

        return batch_dict

class VoxelBackBone8x_layer_wise(nn.Module):
    def __init__(self, model_cfg, input_channels, grid_size, **kwargs):
        super().__init__()
        self.model_cfg = model_cfg
        norm_fn = partial(nn.BatchNorm1d, eps=1e-3, momentum=0.01)

        self.sparse_shape = grid_size[::-1] + [1, 0, 0]

        self.conv_input = spconv.SparseSequential(
            spconv.SubMConv3d(
                input_channels, 16, 3, padding=1, bias=False, indice_key="subm1"
            ),
            norm_fn(16),
            nn.ReLU(),
        )
        block = post_act_block

        self.conv1 = spconv.SparseSequential(
            block(16, 16, 3, norm_fn=norm_fn, padding=1, indice_key="subm1"),
        )

        self.conv2_1 = block(
                16,
                32,
                3,
                norm_fn=norm_fn,
                stride=2,
                padding=1,
                indice_key="spconv2",
                conv_type="spconv",
            )
        self.conv2_2 = block(32, 32, 3, norm_fn=norm_fn, padding=1, indice_key="subm2")
        self.conv2_3 = block(32, 32, 3, norm_fn=norm_fn, padding=1, indice_key="subm2")

        self.conv3_1 = block(
                32,
                64,
                3,
                norm_fn=norm_fn,
                stride=2,
                padding=1,
                indice_key="spconv3",
                conv_type="spconv",
            )
        self.conv3_2 = block(64, 64, 3, norm_fn=norm_fn, padding=1, indice_key="subm3")
        self.conv3_3 = block(64, 64, 3, norm_fn=norm_fn, padding=1, indice_key="subm3")

        self.conv4_1 = block(
                64,
                64,
                3,
                norm_fn=norm_fn,
                stride=2,
                padding=(0, 1, 1),
                indice_key="spconv4",
                conv_type="spconv",
            )
        self.conv4_2 = block(64, 64, 3, norm_fn=norm_fn, padding=1, indice_key="subm4")
        self.conv4_3 = block(64, 64, 3, norm_fn=norm_fn, padding=1, indice_key="subm4")

        last_pad = 0
        last_pad = self.model_cfg.get("last_pad", last_pad)
        self.conv_out = spconv.SparseSequential(
            spconv.SparseConv3d(
                64,
                128,
                (3, 1, 1),
                stride=(2, 1, 1),
                padding=last_pad,
                bias=False,
                indice_key="spconv_down2",
            ),
            norm_fn(128),
            nn.ReLU(),
        )
        self.num_point_features = 128
        self.backbone_channels = {
            "x_conv1": 16,
            "x_conv2": 32,
            "x_conv3": 64,
            "x_conv4": 64,
        }

    def forward(self, batch_dict, **kwargs):
        """
        Args:
            batch_dict:
                batch_size: int
                vfe_features: (num_voxels, C)
                voxel_coords: (num_voxels, 4), [batch_idx, z_idx, y_idx, x_idx]
        Returns:
            batch_dict:
                encoded_spconv_tensor: sparse tensor
        """

        voxel_features, voxel_coords = (
            batch_dict["voxel_features"],
            batch_dict["voxel_coords"],
        )
        batch_size = batch_dict["batch_size"]
        input_sp_tensor = spconv.SparseConvTensor(
            features=voxel_features,
            indices=voxel_coords.int(),
            spatial_shape=self.sparse_shape,
            batch_size=batch_size,
        )

        x = self.conv_input(input_sp_tensor)

        x_conv1 = self.conv1(x)
        x_conv2_1 = self.conv2_1(x_conv1)
        x_conv2_2 = self.conv2_2(x_conv2_1) 
        x_conv2_3 = self.conv2_3(x_conv2_2)
        
        x_conv3_1 = self.conv3_1(x_conv2_3)
        x_conv3_2 = self.conv3_2(x_conv3_1) 
        x_conv3_3 = self.conv3_3(x_conv3_2)
        
        x_conv4_1 = self.conv4_1(x_conv3_3)
        x_conv4_2 = self.conv4_2(x_conv4_1) 
        x_conv4_3 = self.conv4_3(x_conv4_2)

        out = self.conv_out(x_conv4_3)

        batch_dict.update(
            {"encoded_spconv_tensor": out, "encoded_spconv_tensor_stride": 8}
        )
        batch_dict.update(
            {
                "multi_scale_3d_features": {
                    "x_conv1": x_conv1,
                    "x_conv2_1": x_conv2_1,
                    "x_conv2_2": x_conv2_2,
                    "x_conv2_3": x_conv2_3,
                    "x_conv3_1": x_conv3_1,
                    "x_conv3_2": x_conv3_2,
                    "x_conv3_3": x_conv3_3,
                    "x_conv4_1": x_conv4_1,
                    "x_conv4_2": x_conv4_2,
                    "x_conv4_3": x_conv4_3,
                    "x_conv_input": x,
                }
            }
        )
        batch_dict.update(
            {
                "multi_scale_3d_strides": {
                    "x_conv1": 1,
                    "x_conv2_1": 2,
                    "x_conv2_2": 2,
                    "x_conv2_3": 2,
                    "x_conv3_1": 4,
                    "x_conv3_2": 4,
                    "x_conv3_3": 4,
                    "x_conv4_1": 8,
                    "x_conv4_2": 8,
                    "x_conv4_3": 8,
                }
            }
        )

        return batch_dict


class VoxelResBackBone8x(nn.Module):
    def __init__(self, model_cfg, input_channels, grid_size, **kwargs):
        super().__init__()
        self.model_cfg = model_cfg
        norm_fn = partial(nn.BatchNorm1d, eps=1e-3, momentum=0.01)

        self.sparse_shape = grid_size[::-1] + [1, 0, 0]

        self.conv_input = spconv.SparseSequential(
            spconv.SubMConv3d(
                input_channels, 16, 3, padding=1, bias=False, indice_key="subm1"
            ),
            norm_fn(16),
            nn.ReLU(),
        )
        block = post_act_block

        self.conv1 = spconv.SparseSequential(
            SparseBasicBlock(16, 16, norm_fn=norm_fn, indice_key="res1"),
            SparseBasicBlock(16, 16, norm_fn=norm_fn, indice_key="res1"),
        )

        self.conv2 = spconv.SparseSequential(
            block(
                16,
                32,
                3,
                norm_fn=norm_fn,
                stride=2,
                padding=1,
                indice_key="spconv2",
                conv_type="spconv",
            ),
            SparseBasicBlock(32, 32, norm_fn=norm_fn, indice_key="res2"),
            SparseBasicBlock(32, 32, norm_fn=norm_fn, indice_key="res2"),
        )

        self.conv3 = spconv.SparseSequential(
            block(
                32,
                64,
                3,
                norm_fn=norm_fn,
                stride=2,
                padding=1,
                indice_key="spconv3",
                conv_type="spconv",
            ),
            SparseBasicBlock(64, 64, norm_fn=norm_fn, indice_key="res3"),
            SparseBasicBlock(64, 64, norm_fn=norm_fn, indice_key="res3"),
        )

        self.conv4 = spconv.SparseSequential(
            block(
                64,
                128,
                3,
                norm_fn=norm_fn,
                stride=2,
                padding=(0, 1, 1),
                indice_key="spconv4",
                conv_type="spconv",
            ),
            SparseBasicBlock(128, 128, norm_fn=norm_fn, indice_key="res4"),
            SparseBasicBlock(128, 128, norm_fn=norm_fn, indice_key="res4"),
        )

        last_pad = 0
        last_pad = self.model_cfg.get("last_pad", last_pad)
        self.conv_out = spconv.SparseSequential(
            spconv.SparseConv3d(
                128,
                128,
                (3, 1, 1),
                stride=(2, 1, 1),
                padding=last_pad,
                bias=False,
                indice_key="spconv_down2",
            ),
            norm_fn(128),
            nn.ReLU(),
        )
        self.num_point_features = 128
        self.backbone_channels = {
            "x_conv1": 16,
            "x_conv2": 32,
            "x_conv3": 64,
            "x_conv4": 128,
        }

    def forward(self, batch_dict):
        """
        Args:
            batch_dict:
                batch_size: int
                vfe_features: (num_voxels, C)
                voxel_coords: (num_voxels, 4), [batch_idx, z_idx, y_idx, x_idx]
        Returns:
            batch_dict:
                encoded_spconv_tensor: sparse tensor
        """
        voxel_features, voxel_coords = (
            batch_dict["voxel_features"],
            batch_dict["voxel_coords"],
        )
        batch_size = batch_dict["batch_size"]
        input_sp_tensor = spconv.SparseConvTensor(
            features=voxel_features,
            indices=voxel_coords.int(),
            spatial_shape=self.sparse_shape,
            batch_size=batch_size,
        )
        x = self.conv_input(input_sp_tensor)

        x_conv1 = self.conv1(x)
        x_conv2 = self.conv2(x_conv1)
        x_conv3 = self.conv3(x_conv2)
        x_conv4 = self.conv4(x_conv3)

        out = self.conv_out(x_conv4)

        batch_dict.update(
            {"encoded_spconv_tensor": out, "encoded_spconv_tensor_stride": 8}
        )
        batch_dict.update(
            {
                "multi_scale_3d_features": {
                    "x_conv1": x_conv1,
                    "x_conv2": x_conv2,
                    "x_conv3": x_conv3,
                    "x_conv4": x_conv4,
                }
            }
        )

        return batch_dict


class VoxelResBackBone8x_layer_wise(nn.Module):
    def __init__(self, model_cfg, input_channels, grid_size, **kwargs):
        super().__init__()
        self.model_cfg = model_cfg
        norm_fn = partial(nn.BatchNorm1d, eps=1e-3, momentum=0.01)

        self.sparse_shape = grid_size[::-1] + [1, 0, 0]

        self.conv_input = spconv.SparseSequential(
            spconv.SubMConv3d(
                input_channels, 16, 3, padding=1, bias=False, indice_key="subm1"
            ),
            norm_fn(16),
            nn.ReLU(),
        )
        block = post_act_block

        self.conv1_1 = SparseBasicBlock(16, 16, norm_fn=norm_fn, indice_key="res1")
        self.conv1_2 = SparseBasicBlock(16, 16, norm_fn=norm_fn, indice_key="res1")

        self.conv2_1 = block(
                16,
                32,
                3,
                norm_fn=norm_fn,
                stride=2,
                padding=1,
                indice_key="spconv2",
                conv_type="spconv",
            )
        self.conv2_2 = SparseBasicBlock(32, 32, norm_fn=norm_fn, indice_key="res2")
        self.conv2_3 = SparseBasicBlock(32, 32, norm_fn=norm_fn, indice_key="res2")

        self.conv3_1 = block(
                32,
                64,
                3,
                norm_fn=norm_fn,
                stride=2,
                padding=1,
                indice_key="spconv3",
                conv_type="spconv",
            )
        self.conv3_2 = SparseBasicBlock(64, 64, norm_fn=norm_fn, indice_key="res3")
        self.conv3_3 = SparseBasicBlock(64, 64, norm_fn=norm_fn,  indice_key="res3")

        self.conv4_1 = block(
                64,
                128,
                3,
                norm_fn=norm_fn,
                stride=2,
                padding=(0, 1, 1),
                indice_key="spconv4",
                conv_type="spconv",
            )
        self.conv4_2 = SparseBasicBlock(128, 128, norm_fn=norm_fn, indice_key="res4")
        self.conv4_3 = SparseBasicBlock(128, 128, norm_fn=norm_fn, indice_key="res4")


        last_pad = 0
        last_pad = self.model_cfg.get("last_pad", last_pad)
        self.conv_out = spconv.SparseSequential(
            spconv.SparseConv3d(
                128,
                128,
                (3, 1, 1),
                stride=(2, 1, 1),
                padding=last_pad,
                bias=False,
                indice_key="spconv_down2",
            ),
            norm_fn(128),
            nn.ReLU(),
        )
        self.num_point_features = 128
        self.backbone_channels = {
            "x_conv1": 16,
            "x_conv2": 32,
            "x_conv3": 64,
            "x_conv4": 128,
        }

    def forward(self, batch_dict):
        """
        Args:
            batch_dict:
                batch_size: int
                vfe_features: (num_voxels, C)
                voxel_coords: (num_voxels, 4), [batch_idx, z_idx, y_idx, x_idx]
        Returns:
            batch_dict:
                encoded_spconv_tensor: sparse tensor
        """
        voxel_features, voxel_coords = (
            batch_dict["voxel_features"],
            batch_dict["voxel_coords"],
        )
        batch_size = batch_dict["batch_size"]
        input_sp_tensor = spconv.SparseConvTensor(
            features=voxel_features,
            indices=voxel_coords.int(),
            spatial_shape=self.sparse_shape,
            batch_size=batch_size,
        )

        x = self.conv_input(input_sp_tensor)

        x_conv1_1 = self.conv1_1(x)
        x_conv1_2 = self.conv1_2(x_conv1_1)
        x_conv2_1 = self.conv2_1(x_conv1_2)
        x_conv2_2 = self.conv2_2(x_conv2_1) 
        x_conv2_3 = self.conv2_3(x_conv2_2)
        
        x_conv3_1 = self.conv3_1(x_conv2_3)
        x_conv3_2 = self.conv3_2(x_conv3_1) 
        x_conv3_3 = self.conv3_3(x_conv3_2)
        
        x_conv4_1 = self.conv4_1(x_conv3_3)
        x_conv4_2 = self.conv4_2(x_conv4_1) 
        x_conv4_3 = self.conv4_3(x_conv4_2)

        out = self.conv_out(x_conv4_3)

        batch_dict.update(
            {"encoded_spconv_tensor": out, "encoded_spconv_tensor_stride": 8}
        )
        batch_dict.update(
            {
                "multi_scale_3d_features": {
                    "x_conv1_1": x_conv1_1,
                    "x_conv1_2": x_conv1_2,
                    "x_conv2_1": x_conv2_1,
                    "x_conv2_2": x_conv2_2,
                    "x_conv2_3": x_conv2_3,
                    "x_conv3_1": x_conv3_1,
                    "x_conv3_2": x_conv3_2,
                    "x_conv3_3": x_conv3_3,
                    "x_conv4_1": x_conv4_1,
                    "x_conv4_2": x_conv4_2,
                    "x_conv4_3": x_conv4_3,
                    "x_conv_input": x,
                }
            }
        )
        batch_dict.update(
            {
                "multi_scale_3d_strides": {
                    "x_conv1_1": 1,
                    "x_conv1_2": 1,
                    "x_conv2_1": 2,
                    "x_conv2_2": 2,
                    "x_conv2_3": 2,
                    "x_conv3_1": 4,
                    "x_conv3_2": 4,
                    "x_conv3_3": 4,
                    "x_conv4_1": 8,
                    "x_conv4_2": 8,
                    "x_conv4_3": 8,
                }
            }
        )

        return batch_dict
