'''Train CIFAR10 with PyTorch.'''
import torch
import torch.nn as nn
import torch.optim as optim
import torch.nn.functional as F
import torch.backends.cudnn as cudnn

import torchvision
import torchvision.transforms as transforms

import os
import argparse
from datetime import datetime

from models import *
from utils import progress_bar

import agm_optimizer_cleaned as agm
# import agm_optimizer as agm
import numpy as np

from lion_pytorch import Lion
import hashlib
import wandb

from randomaug import RandAugment
from schedulers.CustomCosineAnnealing import CustomCosineAnnealingLR
from schedulers.LinearIncreasing import LinearIncreasingLRScheduler


parser = argparse.ArgumentParser(description='PyTorch CIFAR10 Training')
parser.add_argument('--lr', default=0.1, type=float, help='learning rate')
parser.add_argument('--resume', '-r', action='store_true', help='resume from checkpoint')
parser.add_argument('--lr_tau', default=0.001, type=float, help='learning rate tau')
parser.add_argument('--lr_eta', default=0.02, type=float, help='learning rate eta')
parser.add_argument('--lr_alpha', default=0.01, type=float, help='learning rate alpha')
parser.add_argument('--lr_beta1', default=0.9, type=float, help='learning rate momentum beta1')
parser.add_argument('--lr_beta2', default=0.99, type=float, help='learning rate momentum beta2')
parser.add_argument('--weight_decay', default=1e-2, type=float, help='weight decay')
parser.add_argument('--dampening', default=0.0, type=float, help='dampening')
parser.add_argument('--q_norm', default=2.4, type=float, help='q norm')
parser.add_argument('--debug', default=False, type=bool, help='parameters tracking')
parser.add_argument('--optimizer', default='AGM_NE', type=str, help='parameters tracking')
parser.add_argument('--network', default='R18', type=str, help='neural network tracking')
parser.add_argument('--eps', default=1e-8, type=float, help='eps for denominator')
parser.add_argument('--augmentation', default='No_Aug', type=str, help='randomize data augmentation')
parser.add_argument('--scheduler', default='None_scheduler', type=str, help='scheduler')
args = parser.parse_args()

# create folder for results
def create_hash(params_string: str) -> str:
    R"""
    Create hash from command line argument string. This is mainly for logging purposes.
    """
    hasher = hashlib.md5()
    hasher.update(params_string.encode('utf-8'))
    raw_hash =  hasher.hexdigest()
    hash_str = "{}".format(raw_hash)[:8]
    return hash_str

# Create a hash using the args
run_config = vars(args)
run_hash = create_hash(str(run_config))
run_config['run_hash'] = run_hash

# Create a folder for the results using run hash
results_folder = os.path.join('results', run_hash + '-' + datetime.now().strftime("%Y-%m-%d_%H-%M-%S"))
# put date and time after the run_hash
# results_folder = os.path.join(results_folder, datetime.now().strftime("%Y-%m-%d_%H-%M-%S"))
os.makedirs(results_folder, exist_ok=True)

# Create a txt file to save all args parameters
with open(os.path.join(results_folder, 'args.txt'), 'w') as f:
    # write the date and time in the file
    f.write(f'{datetime.now()}\n')

    for key, value in vars(args).items():
        f.write(f'{key}: {value}\n')

# Create a wandb project
# wandb.init(project='AGM', config=run_config)
wandb.init(
    project='None_Euclidean_Optimize',
    name=f"{args.optimizer}_{args.network}_{args.augmentation}_{args.scheduler}_{run_hash}",
    config=run_config
)

device = 'cuda' if torch.cuda.is_available() else 'cpu'
best_acc = 0  # best test accuracy
start_epoch = 0  # start from epoch 0 or last checkpoint epoch

# Data
print('==> Preparing data..')
transform_train = transforms.Compose([
    transforms.RandomCrop(32, padding=4),
    transforms.RandomHorizontalFlip(),
    transforms.ToTensor(),
    transforms.Normalize((0.4914, 0.4822, 0.4465), (0.2023, 0.1994, 0.2010)),
])

transform_test = transforms.Compose([
    transforms.ToTensor(),
    transforms.Normalize((0.4914, 0.4822, 0.4465), (0.2023, 0.1994, 0.2010)),
])

# Add RandAugment with N, M(hyperparameter)
if args.augmentation == 'Aug':  
    N = 2; M = 14;
    transform_train.transforms.insert(0, RandAugment(N, M))

trainset = torchvision.datasets.CIFAR10(
    root='./data', train=True, download=True, transform=transform_train)
trainloader = torch.utils.data.DataLoader(
    trainset, batch_size=128, shuffle=True, num_workers=2)

testset = torchvision.datasets.CIFAR10(
    root='./data', train=False, download=True, transform=transform_test)
testloader = torch.utils.data.DataLoader(
    testset, batch_size=100, shuffle=False, num_workers=2)

classes = ('plane', 'car', 'bird', 'cat', 'deer',
           'dog', 'frog', 'horse', 'ship', 'truck')

# Model
print('==> Building model..')
# net = VGG('VGG19')
if args.network == 'R18':
    net = ResNet18()
elif args.network == 'R50':
    net = ResNet50()
elif args.network == 'VGG19':
    net = VGG('VGG19')
elif args.network == 'VGG16':
    net = VGG('VGG16')
# net = PreActResNet18()
# net = GoogLeNet()
# net = DenseNet121()
# net = ResNeXt29_2x64d()
# net = MobileNet()
# net = MobileNetV2()
# net = DPN92()
# net = ShuffleNetG2()
# net = SENet18()
# net = ShuffleNetV2(1)
# net = EfficientNetB0()
# net = RegNetX_200MF()
# net = SimpleDLA()
net = net.to(device)
if device == 'cuda':
    net = torch.nn.DataParallel(net)
    cudnn.benchmark = True

if args.resume:
    # Load checkpoint.
    print('==> Resuming from checkpoint..')
    assert os.path.isdir('checkpoint'), 'Error: no checkpoint directory found!'
    checkpoint = torch.load('./checkpoint/ckpt.pth')
    net.load_state_dict(checkpoint['net'])
    best_acc = checkpoint['acc']
    start_epoch = checkpoint['epoch']

criterion = nn.CrossEntropyLoss()

if args.optimizer == 'AGM':
    optimizer = agm.AGM(net.parameters(), lr_tau=args.lr_tau, lr_eta=args.lr_eta, 
                        lr_alpha=args.lr_alpha, lr_beta1=args.lr_beta1, 
                        lr_beta2=args.lr_beta2,weight_decay=args.weight_decay, 
                        dampening=args.dampening, q=args.q_norm, debug=args.debug)

elif args.optimizer == 'AGM_Auto':
    optimizer = agm.AGM_Auto(net.parameters(), lr_tau=args.lr_tau, lr_eta=args.lr_eta, 
                                 lr_alpha=args.lr_alpha, lr_beta1=args.lr_beta1, 
                                 lr_beta2=args.lr_beta2,weight_decay=args.weight_decay, 
                                 dampening=args.dampening, q=args.q_norm, debug=args.debug)

elif args.optimizer == 'Lion':
    optimizer = Lion(net.parameters(), lr=args.lr, weight_decay=args.weight_decay)

elif args.optimizer == 'Adam':
    optimizer = optim.Adam(net.parameters(), lr=args.lr)

elif args.optimizer == 'SGD':
    optimizer = optim.SGD(net.parameters(), lr=args.lr, momentum=args.lr_beta1, weight_decay=args.weight_decay)

elif args.optimizer == 'AGM_NE':
    optimizer = agm.AGM_NE(net.parameters(), lr_tau=args.lr_tau, lr_eta=args.lr_eta, 
                        lr_alpha=args.lr_alpha, lr_beta1=args.lr_beta1, 
                        lr_beta2=args.lr_beta2,weight_decay=args.weight_decay, 
                        dampening=args.dampening, q=args.q_norm, eps=args.eps,
                        debug=args.debug)

elif args.optimizer == 'AGM_HASP':
    optimizer = agm.AGM_HASP(net.parameters(), lr_tau=args.lr_tau, lr_eta=args.lr_eta, 
                        lr_alpha=args.lr_alpha, lr_beta1=args.lr_beta1, 
                        lr_beta2=args.lr_beta2,weight_decay=args.weight_decay, 
                        dampening=args.dampening, q=args.q_norm, eps=args.eps,
                        debug=args.debug)

elif args.optimizer == 'AdamW':
    optimizer = optim.AdamW(net.parameters(), lr=args.lr, weight_decay=args.weight_decay)

elif args.optimizer == 'SGD_Nesterov':
    optimizer = optim.SGD(net.parameters(), lr=args.lr, momentum=args.lr_beta1, weight_decay=args.weight_decay, nesterov=True)

else:
    raise ValueError('Optimizer not supported')

# optimizer = Lion(net.parameters(), lr=1e-4, weight_decay=1e-2)
# optimizer = optim.Adam(net.parameters(), lr=0.1)
# optimizer = optim.SGD(net.parameters(), lr=0.01, momentum=0, weight_decay=0)
# optimizer = agm.AGM(net.parameters(), lr_tau=args.lr_tau, lr_eta=args.lr_eta, 
#                     lr_alpha=args.lr_alpha, lr_beta1=args.lr_beta1, 
#                     lr_beta2=args.lr_beta2,weight_decay=args.weight_decay, 
#                     dampening=args.dampening, q=args.q_norm, debug=args.debug)

if args.scheduler == 'Cosine':
    scheduler1 = CustomCosineAnnealingLR(optimizer, T_max=200, lr_name='lr_eta')
    # scheduler2 = CustomCosineAnnealingLR(optimizer, T_max=200, lr_name='lr_alpha')

elif args.scheduler == 'CosineLinearIncrease':
    scheduler1 = CustomCosineAnnealingLR(optimizer, T_max=200, lr_name='lr_eta')
    scheduler2 = LinearIncreasingLRScheduler(optimizer, lr_name='lr_alpha', 
                                             initial_lr=args.lr_alpha, final_lr=args.lr_alpha * 20.0)


#results 
global train_acc_list
global train_loss_list
train_acc_list = []
train_loss_list = []

global val_acc_list
global val_loss_list
val_acc_list = []
val_loss_list = []

global param_grad
global param_pdata
global param_z
global param_y
global param_update
global param_momentum

param_grad = []
param_pdata = []
param_z = []
param_y = []
param_update = []
param_momentum = []


# Training
def train(epoch):
    print('\nEpoch: %d' % epoch)
    net.train()
    train_loss = 0
    correct = 0
    total = 0


    for batch_idx, (inputs, targets) in enumerate(trainloader):
        inputs, targets = inputs.to(device), targets.to(device)
        optimizer.zero_grad()
        outputs = net(inputs)
        loss = criterion(outputs, targets)
        loss.backward()
        
        params_track = optimizer.step()
        if args.debug is True:
            param_grad.append(params_track['grad_dist'].cpu().item())
            param_pdata.append(params_track['pdata_dist'].cpu().item())
            param_update.append(params_track['update_dist'].cpu().item())
            param_y.append(params_track['y_dist'].cpu().item())
            param_z.append(params_track['z_dist'].cpu().item())
            param_momentum.append(params_track['momentum_dist'].cpu().item())


        train_loss += loss.item()
        _, predicted = outputs.max(1)
        total += targets.size(0)
        correct += predicted.eq(targets).sum().item()

        progress_bar(batch_idx, len(trainloader), 'Loss: %.3f | Acc: %.3f%% (%d/%d)'
                     % (train_loss/(batch_idx+1), 100.*correct/total, correct, total))

    acc_final = 100.*correct/total
    loss_final = train_loss/(batch_idx+1)
    train_acc_list.append(acc_final)
    train_loss_list.append(loss_final)

    wandb.log({
        'train_loss': loss_final,
        'train_acc': acc_final
    }, step=epoch)


def test(epoch):
    global best_acc
    net.eval()
    test_loss = 0
    correct = 0
    total = 0

    with torch.no_grad():
        for batch_idx, (inputs, targets) in enumerate(testloader):
            inputs, targets = inputs.to(device), targets.to(device)
            outputs = net(inputs)
            loss = criterion(outputs, targets)

            test_loss += loss.item()
            _, predicted = outputs.max(1)
            total += targets.size(0)
            correct += predicted.eq(targets).sum().item()

            progress_bar(batch_idx, len(testloader), 'Loss: %.3f | Acc: %.3f%% (%d/%d)'
                         % (test_loss/(batch_idx+1), 100.*correct/total, correct, total))

    # Save checkpoint.
    acc = 100.*correct/total
    if acc > best_acc:
        print('Saving..')
        state = {
            'net': net.state_dict(),
            'acc': acc,
            'epoch': epoch,
        }
        if not os.path.isdir('checkpoint'):
            os.mkdir('checkpoint')
        torch.save(state, './checkpoint/ckpt.pth')
        best_acc = acc

    acc_final = 100.*correct/total
    loss_final = test_loss/(batch_idx+1)
    val_acc_list.append(acc_final)
    val_loss_list.append(loss_final)

    wandb.log({
        'test_loss': loss_final,
        'test_acc': acc_final
    }, step=epoch)


for epoch in range(start_epoch, start_epoch+200):
    train(epoch)
    test(epoch)
    if args.scheduler == 'Cosine':
        scheduler1.step()
    elif args.scheduler != 'None_scheduler':
        scheduler1.step()
        scheduler2.step()
    

train_loss_np = np.array(train_loss_list)
# np.save('./results/train/AGM_Momentum_train_loss.npy', train_loss_np)
np.save(f"{results_folder}/{args.optimizer}_train_loss.npy", train_loss_np)
train_acc_np = np.array(train_acc_list)
# np.save('./results/train/AGM_Momentum_train_acc.npy', train_acc_np)
np.save(f"{results_folder}/{args.optimizer}_train_acc.npy", train_acc_np)

val_loss_np = np.array(val_loss_list)
# np.save('./results/val/AMG_Momentum_val_loss.npy', val_loss_np)
np.save(f"{results_folder}/{args.optimizer}_val_loss.npy", val_loss_np)
val_acc_np = np.array(val_acc_list)
# np.save('./results/val/AMG_Momentum_val_acc.npy', val_acc_np)
np.save(f"{results_folder}/{args.optimizer}_val_acc.npy", val_acc_np)


if args.debug is True:
    # Create a new folder under results_folder to save the parameters tracking
    os.makedirs(f'{results_folder}/parameters_track', exist_ok=True)
    # np.save('./results/parameters_track/param_grad.npy', np.array(param_grad))
    # np.save('./results/parameters_track/param_pdata.npy', np.array(param_pdata))
    # np.save('./results/parameters_track/param_z.npy', np.array(param_z))
    # np.save('./results/parameters_track/param_y.npy', np.array(param_y))
    # np.save('./results/parameters_track/param_update.npy', np.array(param_update))
    # np.save('./results/parameters_track/param_momentum.npy', np.array(param_momentum))

    np.save(f'{results_folder}/parameters_track/param_grad.npy', np.array(param_grad))
    np.save(f'{results_folder}/parameters_track/param_pdata.npy', np.array(param_pdata))
    np.save(f'{results_folder}/parameters_track/param_z.npy', np.array(param_z))
    np.save(f'{results_folder}/parameters_track/param_y.npy', np.array(param_y))
    np.save(f'{results_folder}/parameters_track/param_update.npy', np.array(param_update))
    np.save(f'{results_folder}/parameters_track/param_momentum.npy', np.array(param_momentum))

