import json
import os
import sys
from datetime import datetime
from pathlib import Path

from flask import (
    Flask,
    jsonify,
    redirect,
    render_template,
    request,
    session,
    url_for,
    send_file,
)


def save_stage_file(
    filename: str, data: any, task_id: str, subdirectory: str = None
) -> str:
    """Save stage file with exact filename in stage-specific directory with optional subdirectory"""
    output_dir = get_stage_output_dir(task_id, subdirectory)
    filepath = output_dir / filename

    with open(filepath, "w") as f:
        json.dump(data, f, indent=4)

    print(f"✓ Saved file to: {filepath}")
    return str(filepath)


def get_task_output_dir(task_id: str) -> Path:
    """Get or create output directory for task"""
    output_dir = Path(f".generated_tasks/{task_id}")
    output_dir.mkdir(parents=True, exist_ok=True)
    return output_dir


def get_stage_output_dir(task_id: str, subdirectory: str = None) -> Path:
    """Get or create stage-specific output directory for task with optional subdirectory"""
    output_dir = Path(f".generated_tasks/{task_id}/{subdirectory}")
    output_dir.mkdir(parents=True, exist_ok=True)
    return output_dir


def get_stage_files(task_id: str, subdirectory: str = None) -> list:
    """Get list of all files in stage directory with optional subdirectory"""
    output_dir = get_stage_output_dir(task_id, subdirectory)

    files = []
    if output_dir.exists():
        # If no subdirectory specified, search recursively through all subdirectories
        if not subdirectory:
            for filepath in output_dir.rglob("*.json"):
                # Get relative path from stage directory
                relative_path = filepath.relative_to(output_dir)
                files.append(str(relative_path))
        else:
            # If subdirectory specified, only search in that subdirectory
            for filepath in output_dir.glob("*.json"):
                files.append(filepath.name)

    return sorted(files)


def load_stage_file(filename: str, task_id: str, subdirectory: str = None) -> any:
    """Load stage file by exact filename from stage-specific directory with optional subdirectory"""
    output_dir = get_stage_output_dir(task_id, subdirectory)

    filepath = output_dir / filename

    if not filepath.exists():
        raise FileNotFoundError(
            f"File '{filename}' not found for subdirectory {subdirectory}"
        )

    with open(filepath, "r") as f:
        data = json.load(f)

    print(f"✓ Loaded file from: {filepath}")
    return data


def login():
    # Redirect to data_gen instead of showing login
    return redirect(url_for("data_gen"))


def set_name():
    # Keep this for compatibility but redirect to data_gen
    name = request.form.get("name", "Default User")
    session["user_name"] = name.strip() if name.strip() else "Default User"

    return jsonify(
        {
            "success": True,
            "redirect": url_for("data_gen"),
            "message": f"Welcome, {session['user_name']}! Ready to generate data.",
        }
    )


def set_task_id():
    # Set the task ID from the form
    task_id = request.form.get("task_id", "DR0001")
    session["task_id"] = task_id.strip() if task_id.strip() else "DR0001"

    return jsonify(
        {
            "success": True,
            "redirect": url_for("data_gen"),
            "message": f"Task ID set to {session['task_id']}. Ready to generate data.",
        }
    )


def data_gen():
    # Set default task id if not already set
    if "task_id" not in session:
        session["task_id"] = "DR0001"

    return data_gen_stage1()


def data_gen_stage1():
    # Set default task id if not already set
    if "task_id" not in session:
        session["task_id"] = "DR0001"

    return render_template(
        "data_gen/data_gen_stage_1.html",
        task_id=session["task_id"],
        active_stage=1,
    )


def data_gen_stage2():
    # Set default task id if not already set
    if "task_id" not in session:
        session["task_id"] = "DR0001"

    return render_template(
        "data_gen/data_gen_stage_2.html",
        task_id=session["task_id"],
        active_stage=2,
    )


def data_gen_stage3():
    # Set default task id if not already set
    if "task_id" not in session:
        session["task_id"] = "DR0001"

    return render_template(
        "data_gen/data_gen_stage_3.html",
        task_id=session["task_id"],
        active_stage=3,
    )


def data_gen_stage4():
    # Set default task id if not already set
    if "task_id" not in session:
        session["task_id"] = "DR0001"

    return render_template(
        "data_gen/data_gen_stage_4.html",
        task_id=session["task_id"],
        active_stage=4,
    )


def data_gen_stage5():
    # Set default task id if not already set
    if "task_id" not in session:
        session["task_id"] = "DR0001"

    return render_template(
        "data_gen/data_gen_stage_5.html",
        task_id=session["task_id"],
        active_stage=5,
    )


def data_gen_stage6():
    # Set default task id if not already set
    if "task_id" not in session:
        session["task_id"] = "DR0001"

    return render_template(
        "data_gen/data_gen_stage_6.html",
        task_id=session["task_id"],
        active_stage=6,
    )


# ===============================
# STAGE 1: EXTERNAL INSIGHTS
# ===============================
def stage1_generate_external_insights():
    try:
        data = request.json
        urls = data.get("urls", [])
        count = data.get("count", 10)
        filename = data.get("filename", "stage_1_external_insights.json")
        task_id = data.get("task_id", "DR0001")

        # Validate URLs and proceed with LLM generation
        if not urls or len(urls) == 0:
            return jsonify({"success": False, "error": "At least one URL is required"})

        # Validate URLs
        invalid_urls = []
        for url in urls:
            if not url or not url.strip():  # Skip empty URLs
                continue
            try:
                from urllib.parse import urlparse

                result = urlparse(url)
                if not all([result.scheme, result.netloc]):
                    invalid_urls.append(url)
            except:
                invalid_urls.append(url)

        # Filter out empty URLs
        valid_urls = [url for url in urls if url and url.strip()]

        if not valid_urls:
            return jsonify(
                {"success": False, "error": "At least one valid URL is required"}
            )

        if invalid_urls:
            return jsonify(
                {
                    "success": False,
                    "error": f"Invalid URLs detected: {', '.join(invalid_urls)}",
                }
            )

        # LLM USED HERE: Use the new data generation module
        insights = DATA_GENERATOR.stage1_generate_external_insights(
            urls=valid_urls, count=count
        )

        # Save stage output
        stage_output = {
            "urls": valid_urls,
            "count": count,
            "insights": insights,
            "timestamp": datetime.now().isoformat(),
        }
        subdirectory = from_stage_number_to_subdirectory(1)
        saved_file = save_stage_file(filename, stage_output, task_id, subdirectory)

        return jsonify(
            {
                "success": True,
                "insights": insights,
                "saved_file": saved_file,
                "stage_output": stage_output,
            }
        )

    except Exception as e:
        return jsonify({"success": False, "error": str(e)})


def stage2_generate_company_structure():
    try:
        data = request.json
        task_id = data.get("task_id", "DR0001")
        filename = data.get("filename", "stage_2_company_structure.json")

        # Generate company structure with LLM
        company_structure = DATA_GENERATOR.stage2_generate_company_structure(
            company_name=data.get("name", "Default Company"),
            industry=data.get("industry", "Technology"),
            size=data.get("size", "Medium"),
        )

        # Save company structure
        company_data = {"company_structure": company_structure}
        subdirectory = from_stage_number_to_subdirectory(2, "company_structures")
        save_stage_file(filename, company_data, task_id, subdirectory)

        return jsonify(
            {
                "success": True,
                "company_structure": company_structure,
                "message": f"Generated new company structure",
            }
        )

    except Exception as e:
        return jsonify({"success": False, "error": str(e)})


def stage2_generate_personas():
    try:
        data = request.json
        task_id = data.get("task_id", "DR0001")
        filename = data.get("filename", "stage_2_personas.json")

        # Generate personas with LLM
        personas = DATA_GENERATOR.stage2_generate_personas(
            company_name=data.get("name", "Default Company"),
            industry=data.get("industry", "Technology"),
            size=data.get("size", "Medium"),
            persona_count=data.get("persona_count", 5),
        )

        # Save personas
        personas_data = {"personas": personas}
        subdirectory = from_stage_number_to_subdirectory(2, "personas")
        save_stage_file(filename, personas_data, task_id, subdirectory)

        return jsonify(
            {
                "success": True,
                "personas": personas,
                "message": f"Generated new personas",
            }
        )

    except Exception as e:
        return jsonify({"success": False, "error": str(e)})


def stage3_generate_internal_insights():
    try:
        data = request.json
        task_id = data.get("task_id", "DR0001")
        filename = data.get("filename", "stage_3_internal_insights.json")

        # Get file selections from dropdowns
        external_insights_file = data.get("external_insights_file", "")
        company_structure_file = data.get("company_structure_file", "")
        personas_file = data.get("personas_file", "")
        selected_persona_json = data.get("selected_persona", "")
        focus_area = data.get(
            "focus_area", "operations"
        )  # Single value instead of list
        insights_config = data.get(
            "insights", []
        )  # Get insights configuration from the web interface

        # Validate file selections
        if not external_insights_file:
            return jsonify(
                {"success": False, "error": "External insights file is required"}
            )

        if not company_structure_file:
            return jsonify(
                {"success": False, "error": "Company structure file is required"}
            )

        if not personas_file:
            return jsonify({"success": False, "error": "Personas file is required"})

        if not selected_persona_json:
            return jsonify({"success": False, "error": "Selected persona is required"})

        # Validate insights configuration
        if not insights_config or len(insights_config) == 0:
            return jsonify(
                {"success": False, "error": "Insights configuration is required"}
            )

        # The insight count is now determined by the number of configuration rows
        insight_count = len(insights_config)

        # Parse the selected persona JSON
        try:
            import json

            selected_persona = json.loads(selected_persona_json)
        except (json.JSONDecodeError, TypeError):
            return jsonify({"success": False, "error": "Invalid persona selection"})

        # Load external insights from selected file
        try:
            external_insights_data = load_stage_file(
                external_insights_file, task_id, from_stage_number_to_subdirectory(1)
            )
            external_insights = external_insights_data.get("insights", [])
        except FileNotFoundError:
            return jsonify(
                {
                    "success": False,
                    "error": f"External insights file '{external_insights_file}' not found",
                }
            )

        # Load company structure from selected file
        try:
            company_structure_data = load_stage_file(
                company_structure_file,
                task_id,
                from_stage_number_to_subdirectory(2, "company_structures"),
            )
            company_data = company_structure_data.get("company_structure", {})
        except FileNotFoundError:
            return jsonify(
                {
                    "success": False,
                    "error": f"Company structure file '{company_structure_file}' not found",
                }
            )

        # Load personas from selected file
        try:
            personas_data = load_stage_file(
                personas_file, task_id, from_stage_number_to_subdirectory(2, "personas")
            )
            personas = personas_data.get("personas", [])
        except FileNotFoundError:
            return jsonify(
                {
                    "success": False,
                    "error": f"Personas file '{personas_file}' not found",
                }
            )

        if company_data is None:
            company_data = {}

        # Use the new data generation module with single focus area, insights config, and selected persona
        insights = DATA_GENERATOR.stage3_generate_internal_insights(
            external_insights,
            company_data,
            insight_count,
            focus_area,
            insights_config,
            selected_persona,
        )

        # Save stage output
        stage_output = {
            "insights": insights,
            "insights_count": len(insights),
            "focus_area": focus_area,  # Single value instead of list
            "external_insights": external_insights_data,
            "company_structure": company_structure_data,
            "persona": selected_persona,
            "insights_config": insights_config,
            "timestamp": datetime.now().isoformat(),
        }
        subdirectory = from_stage_number_to_subdirectory(3)
        saved_file = save_stage_file(filename, stage_output, task_id, subdirectory)

        return jsonify(
            {
                "success": True,
                "insights": insights,
                "saved_file": saved_file,
                "stage_output": stage_output,
            }
        )

    except Exception as e:
        return jsonify({"success": False, "error": str(e)})


def stage4_generate_dr_question():
    try:
        data = request.json
        task_id = data.get("task_id", "DR0001")
        filename = data.get("filename", "stage_4_dr_question.json")

        selected_internal_insights = data.get("selected_internal_insights", [])
        selected_external_insights = data.get("selected_external_insights", [])
        persona = data.get("persona", {})
        company_data = data.get("company_data", {})

        # Validate inputs
        if not selected_internal_insights or len(selected_internal_insights) == 0:
            return jsonify(
                {"success": False, "error": "At least one internal insight is required"}
            )

        if not persona:
            return jsonify({"success": False, "error": "Persona is required"})

        if not company_data:
            return jsonify({"success": False, "error": "Company data is required"})

        # Use the new data generation module
        dr_question = DATA_GENERATOR.stage4_generate_dr_question(
            selected_internal_insights,
            selected_external_insights,
            persona,
            company_data,
        )

        # Save stage output
        stage_output = {
            "dr_question": dr_question,
            "selected_persona": persona,
            "internal_insights": selected_internal_insights,
            "external_insights": selected_external_insights,
            "company_data": company_data,
            "generation_params": {
                "internal_insights_used": len(selected_internal_insights),
                "external_insights_used": len(selected_external_insights),
            },
            "timestamp": datetime.now().isoformat(),
        }
        subdirectory = from_stage_number_to_subdirectory(4)
        saved_file = save_stage_file(filename, stage_output, task_id, subdirectory)

        return jsonify(
            {
                "success": True,
                "dr_question": dr_question,
                "saved_file": saved_file,
                "stage_output": stage_output,
            }
        )

    except Exception as e:
        return jsonify({"success": False, "error": str(e)})


def stage5_generate_files_and_evaluation():
    try:
        data = request.json
        task_id = data.get("task_id", "DR0001")
        filename = data.get("filename", "stage_5_files_evaluation.json")

        dr_question = data.get("dr_question", "")
        selected_persona = data.get("selected_persona", {})
        internal_insights = data.get("internal_insights", [])
        external_insights = data.get("external_insights", [])
        company_data = data.get("company_data", {})
        selected_file_types = data.get(
            "selected_file_types", []
        )  # Array of {fileType, insightType}
        file_configs = data.get(
            "file_configs", {}
        )  # Object with configs for each file type

        # Validate inputs
        if not dr_question:
            return jsonify({"success": False, "error": "DR question is required"})

        if not selected_persona:
            return jsonify({"success": False, "error": "Persona is required"})

        if not internal_insights:
            return jsonify(
                {"success": False, "error": "Internal insights are required"}
            )

        if not company_data:
            return jsonify({"success": False, "error": "Company data is required"})

        if not selected_file_types or len(selected_file_types) == 0:
            return jsonify(
                {"success": False, "error": "At least one file type must be selected"}
            )

        # Separate insight and distractor file types
        insight_file_types = []
        distractor_file_types = []

        for file_selection in selected_file_types:
            file_type = file_selection.get("fileType")
            insight_type = file_selection.get("insightType", "insight")

            if insight_type == "insight":
                insight_file_types.append(file_type)
            else:  # distractor
                distractor_file_types.append(file_type)

        # Use the data generation module with the updated structure
        evaluation_data, supporting_files, distractor_files = (
            DATA_GENERATOR.stage5_generate_files_and_evaluation_with_distribution(
                dr_question,
                internal_insights,
                external_insights,
                company_data,
                selected_persona,
                insight_file_types,
                distractor_file_types,
            )
        )

        # Save stage output
        stage_output = {
            "evaluation_data": evaluation_data,
            "supporting_files": supporting_files,
            "distractor_files": distractor_files,
            "generation_params": {
                "selected_file_types": selected_file_types,
                "file_configs": file_configs,
                "insight_file_types": insight_file_types,
                "distractor_file_types": distractor_file_types,
                "total_supporting_files": len(insight_file_types),
                "total_distractor_files": len(distractor_file_types),
            },
            "timestamp": datetime.now().isoformat(),
        }

        subdirectory = from_stage_number_to_subdirectory(5)
        saved_file = save_stage_file(filename, stage_output, task_id, subdirectory)

        return jsonify(
            {
                "success": True,
                "evaluation_data": evaluation_data,
                "supporting_files": supporting_files,
                "distractor_files": distractor_files,
                "saved_file": saved_file,
                "stage_output": stage_output,
            }
        )

    except Exception as e:
        return jsonify({"success": False, "error": str(e)})


def api_package_complete_task():
    # Set default user if not in session
    if "user_name" not in session:
        session["user_name"] = "Default User"

    try:
        data = request.json
        task_json = data.get("task_json", {})
        eval_json = data.get("eval_json", {})
        company_data = data.get("company_data", {})
        personas = data.get("personas", [])
        supporting_files = data.get("supporting_files", [])
        distractor_files = data.get("distractor_files", [])

        user_name = session["user_name"]
        output_dir = get_user_output_dir(user_name)

        # Generate final outputs
        all_files = supporting_files + distractor_files

        # Generate task.json
        task_json_data = {
            "name": f"{company_data.get('name', 'Company')}_{datetime.now().strftime('%Y%m%d_%H%M%S')}",
            "level": task_json.get("level", "medium"),
            "context": "company_info.json",
            "dr_question": task_json.get("dr_question", ""),
            "date": datetime.now().strftime("%Y-%m-%d"),
            "persona": task_json.get("persona", {}),
            "user_guidance": "Generated automatically by DR Bench Data Generator",
            "env_files": all_files,
        }

        # Generate eval.json with separate supporting and distractor file paths
        eval_json_data = {
            "supporting_file_paths": [
                f["destination"] for f in supporting_files if f.get("destination")
            ],
            "distractor_file_paths": [
                f["destination"] for f in distractor_files if f.get("destination")
            ],
            "dr_report_evaluation_qa": eval_json.get("dr_report_evaluation_qa", []),
            "dr_report_intermediate_steps": eval_json.get(
                "dr_report_intermediate_steps", []
            ),
        }

        # Save final outputs
        with open(output_dir / "task.json", "w") as f:
            json.dump(task_json_data, f, indent=2)

        with open(output_dir / "eval.json", "w") as f:
            json.dump(eval_json_data, f, indent=2)

        # Save summary
        summary = {
            "generation_summary": {
                "company_name": company_data.get("name", "Unknown"),
                "industry": company_data.get("industry", "Unknown"),
                "dr_question": task_json_data.get("dr_question", ""),
                "persona": f"{task_json_data.get('persona', {}).get('name', 'Unknown')} ({task_json_data.get('persona', {}).get('role', 'Unknown')})",
                "supporting_files_generated": len(supporting_files),
                "distractor_files_generated": len(distractor_files),
                "total_files_generated": len(all_files),
                "generation_timestamp": datetime.now().isoformat(),
            },
            "files_created": [
                "task.json",
                "eval.json",
                "company_info.json",
                "personas.jsonl",
                "stage_1_external_insights.json",
                "stage_2_company_structure.json",
                "stage_2_personas.json",
                "stage_3_internal_insights.json",
                "stage_4_dr_question.json",
                "stage_5_files_evaluation.json",
                "stage_6_file_generation.json",
            ],
        }

        with open(output_dir / "generation_summary.json", "w") as f:
            json.dump(summary, f, indent=2)

        return jsonify(
            {
                "success": True,
                "output_path": str(output_dir),
                "task_json": task_json_data,
                "eval_json": eval_json_data,
                "summary": summary,
                "files_created": [
                    "task.json",
                    "eval.json",
                    "company_info.json",
                    "personas.jsonl",
                    "generation_summary.json",
                ],
                "supporting_files_count": len(supporting_files),
                "distractor_files_count": len(distractor_files),
            }
        )

    except Exception as e:
        return jsonify({"success": False, "error": str(e)})


def api_view_stage_output(stage_name):
    """View saved stage output JSON file"""
    # Set default user if not in session
    if "user_name" not in session:
        session["user_name"] = "Default User"

    try:
        user_name = session["user_name"]
        output_dir = get_user_output_dir(user_name)

        if check_stage_exists(stage_name, output_dir):
            stage_data = load_stage_output(stage_name, output_dir)
            return jsonify(
                {
                    "success": True,
                    "stage_data": stage_data,
                    "stage_name": stage_name,
                    "file_path": str(
                        output_dir / f"{stage_name.lower().replace(' ', '_')}.json"
                    ),
                }
            )
        else:
            return jsonify(
                {"success": False, "error": f"Stage {stage_name} output not found"}
            )

    except Exception as e:
        return jsonify({"success": False, "error": str(e)})


def api_list_user_files():
    """List all generated files for the current user"""
    # Set default user if not in session
    if "user_name" not in session:
        session["user_name"] = "Default User"

    try:
        user_name = session["user_name"]
        output_dir = get_user_output_dir(user_name)

        files = []
        for file_path in output_dir.glob("*.json"):
            files.append(
                {
                    "name": file_path.name,
                    "path": str(file_path),
                    "size": file_path.stat().st_size,
                    "modified": datetime.fromtimestamp(
                        file_path.stat().st_mtime
                    ).isoformat(),
                }
            )

        return jsonify({"success": True, "files": files, "output_dir": str(output_dir)})

    except Exception as e:
        return jsonify({"success": False, "error": str(e)})


def api_generate_sample_internal_insights():
    # Set default user if not in session
    if "user_name" not in session:
        session["user_name"] = "Default User"

    try:
        data = request.json
        company_data = data.get("company_data", {})
        persona = data.get("persona", {})

        # Generate sample insight configurations (just id and file_type)
        sample_configs = [
            {"id": "INT001", "file_type": "pdf"},
            {"id": "INT002", "file_type": "mattermost_jsonl"},
            {"id": "INT003", "file_type": "roundcube_jsonl"},
            {"id": "INT004", "file_type": "docx"},
        ]

        return jsonify({"success": True, "insights": sample_configs})

    except Exception as e:
        return jsonify({"success": False, "error": str(e)})


def api_generate_insights_from_config():
    # Set default user if not in session
    if "user_name" not in session:
        session["user_name"] = "Default User"

    try:
        data = request.json
        insights_config = data.get("insights_config", [])
        external_insights = data.get("external_insights", [])
        company_data = data.get("company_data", {})

        if not insights_config:
            return jsonify(
                {"success": False, "error": "Insights configuration is required"}
            )

        # Import the data generator
        from drbench.data_gen import get_data_generator

        generator = get_data_generator()

        # Generate full insights from minimal config
        generated_insights = []
        for insight_config in insights_config:
            if not isinstance(insight_config, dict):
                continue

            insight_id = insight_config.get("id", f"INT{len(generated_insights)+1:03d}")
            file_type = insight_config.get("file_type", "pdf")

            try:
                generated_insight = generator.generate_single_internal_insight(
                    external_insights, company_data, insight_id, file_type
                )
                generated_insights.append(generated_insight)
            except Exception as e:
                print(f"Error generating insight {insight_id}: {e}")
                # Continue with other insights if one fails
                continue

        return jsonify({"success": True, "insights": generated_insights})

    except Exception as e:
        return jsonify({"success": False, "error": str(e)})


def from_stage_number_to_subdirectory(stage_number: int, data_type=None) -> str:
    if stage_number == 1:
        subdirectory = "stage_1_external_insights"
    elif stage_number == 2:
        if data_type == "company_structures":
            subdirectory = "stage_2_company_structures"
        elif data_type == "personas":
            subdirectory = "stage_2_personas"
        else:
            subdirectory = "stage_2_company_structures"  # Default for stage 2
    elif stage_number == 3:
        subdirectory = "stage_3_internal_insights"
    elif stage_number == 4:
        subdirectory = "stage_4_dr_question"
    elif stage_number == 5:
        subdirectory = "stage_5_files_and_evaluation"
    elif stage_number == 6:
        subdirectory = "stage_6_file_content"
    else:
        raise ValueError(f"Invalid stage number: {stage_number}")
    return subdirectory


def api_list_stage_files():
    try:
        data = request.json
        task_id = data.get("task_id", "DR0001")
        stage_number = data.get("stage_number", 1)
        data_type = data.get(
            "data_type", None
        )  # e.g., "external_insights", "company_structures", "personas"

        # Determine subdirectory based on stage number and data type
        subdirectory = from_stage_number_to_subdirectory(stage_number, data_type)
        # If no data_type specified, search all subdirectories (no subdirectory parameter)

        files = get_stage_files(task_id, subdirectory)

        return jsonify(
            {
                "success": True,
                "files": files,
            }
        )

    except Exception as e:
        return jsonify({"success": False, "error": str(e)})


def api_load_stage_file():
    try:
        data = request.json
        task_id = data.get("task_id", "DR0001")
        stage_number = data.get("stage_number", 1)
        filename = data.get("filename", "")
        data_type = data.get(
            "data_type", None
        )  # e.g., "external_insights", "company_structures", "personas"

        if not filename:
            return jsonify({"success": False, "error": "Filename is required"})

        # Determine subdirectory based on stage number and data type
        subdirectory = from_stage_number_to_subdirectory(stage_number, data_type)

        loaded_data = load_stage_file(filename, task_id, subdirectory)

        return jsonify(
            {
                "success": True,
                "data": loaded_data,
                "filename": filename,
            }
        )

    except FileNotFoundError as e:
        return jsonify({"success": False, "error": str(e)})
    except Exception as e:
        return jsonify({"success": False, "error": str(e)})


def api_save_insights():
    try:
        data = request.json
        insights = data.get("insights", [])
        filename = data.get("filename", "stage_1_external_insights.json")
        task_id = data.get("task_id", "DR0001")

        # Create stage output with updated insights
        stage_output = {
            "urls": [],  # We don't have the original URLs in edit mode
            "count": len(insights),
            "insights": insights,
            "timestamp": datetime.now().isoformat(),
            "edited": True,
        }

        # Save the updated insights
        subdirectory = from_stage_number_to_subdirectory(1)
        saved_file = save_stage_file(filename, stage_output, task_id, subdirectory)

        return jsonify(
            {
                "success": True,
                "saved_file": saved_file,
                "message": f"Insights saved successfully to {saved_file}",
            }
        )

    except Exception as e:
        return jsonify({"success": False, "error": str(e)})


def api_save_company_structure():
    try:
        data = request.json
        company_structure = data.get("company_structure", {})
        filename = data.get("filename", "stage_2_company_structure.json")
        task_id = data.get("task_id", "DR0001")

        # Create stage output with updated company structure
        company_output = {
            "company_structure": company_structure,
            "generation_params": {
                "company_name": company_structure.get("name", "Unknown"),
                "industry": company_structure.get("industry", "Unknown"),
                "size": company_structure.get("size", "Unknown"),
                "persona_count": 0,  # We don't have this info in edit mode
            },
            "timestamp": datetime.now().isoformat(),
            "edited": True,
        }

        # Save the updated company structure
        subdirectory = from_stage_number_to_subdirectory(2, "company_structures")
        saved_file = save_stage_file(filename, company_output, task_id, subdirectory)

        return jsonify(
            {
                "success": True,
                "saved_file": saved_file,
                "message": f"Company structure saved successfully to {saved_file}",
            }
        )

    except Exception as e:
        return jsonify({"success": False, "error": str(e)})


def api_save_personas():
    try:
        data = request.json
        personas = data.get("personas", [])
        filename = data.get("filename", "stage_2_personas.json")
        task_id = data.get("task_id", "DR0001")

        # Create stage output with updated personas
        personas_output = {
            "personas": personas,
            "count": len(personas),
            "timestamp": datetime.now().isoformat(),
            "edited": True,
        }

        # Save the updated personas
        subdirectory = from_stage_number_to_subdirectory(2, "personas")
        saved_file = save_stage_file(filename, personas_output, task_id, subdirectory)

        return jsonify(
            {
                "success": True,
                "saved_file": saved_file,
                "message": f"Personas saved successfully to {saved_file}",
            }
        )

    except Exception as e:
        return jsonify({"success": False, "error": str(e)})


def api_download_file():
    """Download a file from the task's output directory"""
    try:
        file_path = request.args.get("path")
        task_id = request.args.get("task_id", "DR0001")

        if not file_path:
            return jsonify({"success": False, "error": "No file path provided"})

        # Security check: ensure the file is within the task's output directory
        task_output_dir = get_task_output_dir(task_id)

        # Convert to Path objects for secure comparison
        requested_path = Path(file_path).resolve()
        task_dir = task_output_dir.resolve()

        # Check if the requested file is within the task's directory
        try:
            requested_path.relative_to(task_dir)
        except ValueError:
            return jsonify(
                {
                    "success": False,
                    "error": "Access denied: File outside task directory",
                }
            )

        if not requested_path.exists():
            return jsonify({"success": False, "error": "File not found"})

        # Return the file for download
        return send_file(
            requested_path,
            as_attachment=True,
            download_name=requested_path.name,
            mimetype="application/json",
        )

    except Exception as e:
        return jsonify({"success": False, "error": str(e)})


def load_json(path):
    with open(path, "r") as f:
        return json.load(f)


def get_timestamp() -> str:
    from datetime import datetime, timezone, timedelta

    return (
        datetime.now(timezone.utc)
        .astimezone(timezone(timedelta(hours=-7)))
        .strftime("%Y-%m-%d %H:%M")
    )


def save_json(path, data):
    with open(path, "w") as f:
        json.dump(data, f, indent=4)
    print(f"\nSaved to {path}\n")


def print_dict(dict):
    print("\n")
    for k in dict:
        print(f"{k}:\n   {dict[k]}\n")
    print("--------------------------------\n")


def print_list(list):
    print("\n")
    for i, item in enumerate(list):
        print(f"ITEM {i+1}:")
        for k in item:
            print(f"{k}: {item[k]}")
        print("--------------------------------\n")


from typing import Dict, Any, List
import json
import re


def extract_json_from_response(response: str) -> List[Dict[str, Any]]:
    """Extract JSON from AI response text"""
    import re

    # Try to find JSON array or object
    json_match = re.search(r"(\[.*\]|\{.*\})", response, re.DOTALL)
    if json_match:
        try:
            return json.loads(json_match.group(1))
        except json.JSONDecodeError:
            pass

    # Try parsing the entire response
    try:
        return json.loads(response.strip())
    except json.JSONDecodeError:
        raise ValueError("Could not extract valid JSON from response")


# Helper: format persona context
def format_persona_context(persona):
    if not persona:
        return ""
    return f"- Name: {persona.get('name', 'N/A')}\n- Role: {persona.get('role', 'N/A')}\n- Department: {persona.get('department', 'N/A')}\n- Seniority: {persona.get('seniority', 'N/A')}\n- Responsibilities: {persona.get('responsibilities', 'N/A')}"


# Helper: format internal insights context
def format_insights_context(insights):
    if not insights:
        return "None"
    return "\n".join([f"- {insight.get('insight', '')}" for insight in insights])


# Helper: format external insights context
def format_external_context(insights):
    if not insights:
        return "None"
    return "\n".join([f"- {insight.get('insight', '')}" for insight in insights])
