from drbench.metrics.distractor_recall import DistractorRecall
from drbench.metrics.drgym_factuality import DRGymFactuality
from drbench.metrics.drgym_quality import DRGymQuality
from drbench.metrics.drgym_report_relevance import DRGymReportRelevance
from drbench.metrics.factuality import ClaimFactuality
from drbench.metrics.factuality_v2 import CitationFactuality
from drbench.metrics.qa_similarity_v2 import QASimilarityV2
from drbench.metrics.report_quality import ReportQuality


def get_metric(name: str, model: str = "gpt-4o"):
    """
    Get metric by name.

    Args:
        name: The name of the metric to retrieve
        **kwargs: Additional parameters for the metric constructor

    Returns:
        DrBenchMetric: An instance of the requested metric class
    """
    if name == "factuality":
        raise ValueError("factuality metric is deprecated. Use factuality_v2 instead.")

    available_metrics = [
        "insights_recall",
        "distractor_recall",
        "report_quality",
        "factuality_v2",
        "drgym_factuality",
        "drgym_quality",
        "drgym_report_relevance",
    ]
    assert (
        name in available_metrics
    ), f"Unknown metric name: {name} - only {available_metrics} are supported"

    if name == "qa_similarity_v2":
        # model="together_ai/meta-llama/Llama-3.3-70B-Instruct-Turbo",
        return QASimilarityV2(
            model=model,
            n=5,
        )

    elif name == "insights_recall":
        return QASimilarityV2(
            model=model,
            n=5,
            ignore_not_answerable=False,
        )

    elif name == "distractor_recall":
        return DistractorRecall(
            model=model,
        )

    elif name == "report_quality":
        return ReportQuality(
            model=model,
        )

    elif name == "factuality":
        return ClaimFactuality(
            model=model,
            chunk_size=1000,
            top_k_chunks=3,
            embedding_model="text-embedding-3-small",
        )

    elif name == "factuality_v2":
        return CitationFactuality(
            model=model,
        )

    elif name == "drgym_factuality":
        return DRGymFactuality(
            model=model,
        )

    elif name == "drgym_quality":
        return DRGymQuality(
            model=model,
        )

    elif name == "drgym_report_relevance":
        return DRGymReportRelevance(
            model=model,
        )

    else:
        raise ValueError(f"Unknown metric name: {name}")
