import os
import random
import warnings
import argparse

import torch
import torch.nn.parallel
import torch.backends.cudnn as cudnn
import torch.distributed as dist
import torch.optim
import torch.nn as nn
import torch.multiprocessing as mp
import torch.utils.data
import torch.utils.data.distributed
import torchvision.transforms as transforms
import torchvision.models as models
from torchvision.transforms import InterpolationMode

from torch.utils.data import Dataset, TensorDataset, DataLoader

import torch.nn.functional as F

from tqdm import tqdm
import clip
import peft
from peft import PeftConfig, PeftModel
from prefetch_generator import BackgroundGenerator
import torchvision.datasets as datasets
import torchvision
import time

class DataLoaderX(DataLoader):
    def __iter__(self):
        return BackgroundGenerator(super().__iter__())

parser = argparse.ArgumentParser(description='LoRA-Like Low-Rank Knowledge Transfer')
parser.add_argument('--data', metavar='DIR',
                    help='path to dataset')
parser.add_argument('-j', '--workers', default=4, type=int, metavar='N',
                    help='number of data loading workers (default: 4)')
parser.add_argument('-b', '--batch-size', default=4, type=int,
                    metavar='N',
                    help='mini-batch size (default: 256), this is the total '
                         'batch size of all GPUs on the current node when '
                         'using Data Parallel or Distributed Data Parallel')
parser.add_argument('--world-size', default=-1, type=int,
                    help='number of nodes for distributed training')
parser.add_argument('--temperature', default=20, type=float,
                    help='the temperature of FKD')
parser.add_argument('--class-number', default=1000, type=int)

parser.add_argument('--seed', default=None, type=int,
                    help='seed for initializing training. ')

parser.add_argument('--input-size', default=224, type=int, metavar='S')

parser.add_argument('--matrix-epoch', type=int, default=100)
parser.add_argument('--ce-weight', type=float, default=0.1)
parser.add_argument('--output-dir', type=str)
parser.add_argument('--lora-lr', type=float)
parser.add_argument('--loss-type', type=str, default='kl')

class AverageMeter(object):
    def __init__(self):
        self.reset()

    def reset(self):
        self.avg = 0
        self.sum = 0
        self.cnt = 0
        self.val = 0

    def update(self, val, n=1):
        self.val = val
        self.sum += val * n
        self.cnt += n
        self.avg = self.sum / self.cnt


def accuracy(output, target, topk=(1,)):
    maxk = max(topk)
    batch_size = target.size(0)

    _, pred = output.topk(maxk, 1, True, True)
    pred = pred.t()
    correct = pred.eq(target.reshape(1, -1).expand_as(pred))

    res = []
    for k in topk:
        correct_k = correct[:k].reshape(-1).float().sum(0)
        res.append(correct_k.mul_(100.0 / batch_size))
    return res


class ProjectionNet(nn.Module):
    def __init__(self, input_features_dim, output_features_dim):
        super(ProjectionNet, self).__init__()
        self.fc = nn.Linear(input_features_dim, output_features_dim)  
    def forward(self, x):
        return self.fc(x)
    
class Adapter(nn.Module):
    def __init__(self, c_in, c_out, reduction=4):
        super(Adapter, self).__init__()
        self.fc = nn.Sequential(
            nn.Linear(c_in, c_in // reduction, bias=False),
            nn.ReLU(inplace=True),
            nn.Linear(c_in // reduction, c_out, bias=False),
            nn.ReLU(inplace=True)
        )

    def forward(self, x):
        x = self.fc(x)
        return x
    
class CLIPCLS(nn.Module):
    def __init__(self, model, linear):
        super(CLIPCLS, self).__init__()
        self.model = model
        self.fc = linear
    
    def forward(self, x):
        feature = self.model(x)
        out = self.fc(feature)
        return feature, out

class MLP(nn.Module):
    def __init__(self, input_features_dim, output_features_dim):
        super(MLP, self).__init__()
        self.fc_1 = nn.Linear(input_features_dim, 128)
        self.fc_2 = nn.Linear(128, output_features_dim)

    def forward(self, x):
        out = F.relu(self.fc_1(x))
        out = self.fc_2(out)
        return out

class ProjectionDataset(Dataset):
    def __init__(self, images, soft_labels, targets):  
        self.images = images
        self.soft_labels = soft_labels
        self.targets = targets

    def __len__(self):
        return len(self.images)

    def __getitem__(self, index):
        image = self.images[index]
        soft_label = self.soft_labels[index]
        target = self.targets[index]
        return image, soft_label, target

def testvalidate(model, val_loader, epoch=None):
    objs = AverageMeter()
    top1 = AverageMeter()
    top5 = AverageMeter()
    loss_function = nn.CrossEntropyLoss()

    model.eval()
    t1 = time.time()
    with torch.no_grad():
        for data, target in val_loader:
            target = target.type(torch.LongTensor)
            data, target = data.cuda(), target.cuda()

            _, output = model(data)
            loss = loss_function(output, target)

            prec1, prec5 = accuracy(output, target, topk=(1, 5))
            n = data.size(0)
            objs.update(loss.item(), n)
            top1.update(prec1.item(), n)
            top5.update(prec5.item(), n)

    logInfo = 'TEST Iter {}: loss = {:.6f},\t'.format(epoch, objs.avg) + \
              'Top-1 err = {:.6f},\t'.format(100 - top1.avg) + \
              'Top-5 err = {:.6f},\t'.format(100 - top5.avg) + \
              'val_time = {:.6f}'.format(time.time() - t1)
    print(logInfo)

    metrics = {
        'val/loss': objs.avg,
        'val/top1': top1.avg,
        'val/top5': top5.avg,
        'val/epoch': epoch,
    }
  
    return top1.avg

def cosine_similarity(a, b, eps=1e-5):
    return (a * b).sum(1) / (a.norm(dim=1) * b.norm(dim=1) + eps)


def pearson_correlation(a, b, eps=1e-5):
    return cosine_similarity(a - a.mean(1).unsqueeze(1),
                             b - b.mean(1).unsqueeze(1), eps)


def inter_class_relation(y_s, y_t):
    return 1 - pearson_correlation(y_s, y_t).mean()


def intra_class_relation(y_s, y_t):
    return inter_class_relation(y_s.transpose(0, 1), y_t.transpose(0, 1))


class DISTLoss(nn.Module):
    def __init__(self, beta=2, gamma=2, tem=4):
        super(DISTLoss, self).__init__()
        self.beta = beta
        self.gamma = gamma
        self.tem = tem

    def forward(self, logits_student, logits_teacher):
        y_s = (logits_student / self.tem).softmax(dim=1)
        y_t = (logits_teacher / self.tem).softmax(dim=1)
        inter_loss = (self.tem ** 2) * inter_class_relation(y_s, y_t)
        intra_loss = (self.tem ** 2) * intra_class_relation(y_s, y_t)
        loss_kd = self.beta * inter_loss + self.gamma * intra_loss

        return loss_kd

def zeroshot_classifier(model, classnames, templates):
    with torch.no_grad():
        zeroshot_weights = []
        for classname in tqdm(classnames):
            texts = [template.format(classname) for template in templates] 
            texts = clip.tokenize(texts).cuda()
            class_embeddings = model.encode_text(texts) 
            class_embeddings /= class_embeddings.norm(dim=-1, keepdim=True)
            class_embedding = class_embeddings.mean(dim=0)
            class_embedding /= class_embedding.norm()
            zeroshot_weights.append(class_embedding)
        zeroshot_weights = torch.stack(zeroshot_weights, dim=1).cuda()
    return zeroshot_weights

def main():
    args = parser.parse_args()

    if not os.path.exists(args.output_dir):
        os.makedirs(args.output_dir, exist_ok=True)

    if args.seed is not None:
        random.seed(args.seed)
        torch.manual_seed(args.seed)
        cudnn.deterministic = True
        warnings.warn('You have chosen to seed training. '
                      'This will turn on the CUDNN deterministic setting, '
                      'which can slow down your training considerably! '
                      'You may see unexpected behavior when restarting '
                      'from checkpoints.')


    main_worker(args)

def _set_module(model, submodule_key, module):
    tokens = submodule_key.split('.')
    sub_tokens = tokens[:-1]
    cur_mod = model
    for s in sub_tokens:
        cur_mod = getattr(cur_mod, s)
    setattr(cur_mod, tokens[-1], module)


def main_worker(args):
    aux_teacher = ["resnet18"]
    print("=> using pytorch pre-trained model '{}'".format(aux_teacher))
    model_teacher = []
    model_list = os.listdir("/path/to/model/pool")
    for ml in model_list:
        model_path = os.path.join("/path/to/model/pool", ml)
        _model = models.__dict__['resnet18'](pretrained=False)
        checkpoint = torch.load(model_path)
        _model.load_state_dict(checkpoint['model'])
        _model = _model.cuda()
        _model.eval()
        for p in _model.parameters():
            p.requires_grad = False
        _model = nn.DataParallel(_model).cuda()
        model_teacher.append(_model)
    model = model_teacher


    clipm, preprocess = clip.load('RN50', device="cuda")
    clipm = clipm.to(dtype=torch.float32)
    image_encoder = clipm.visual

    imagenet_classes = ["tench", "goldfish", "great white shark", "tiger shark", "hammerhead shark", "electric ray", "stingray", "rooster", "hen", "ostrich", "brambling", "goldfinch", "house finch", "junco", "indigo bunting", "American robin", "bulbul", "jay", "magpie", "chickadee", "American dipper", "kite (bird of prey)", "bald eagle", "vulture", "great grey owl", "fire salamander", "smooth newt", "newt", "spotted salamander", "axolotl", "American bullfrog", "tree frog", "tailed frog", "loggerhead sea turtle", "leatherback sea turtle", "mud turtle", "terrapin", "box turtle", "banded gecko", "green iguana", "Carolina anole", "desert grassland whiptail lizard", "agama", "frilled-necked lizard", "alligator lizard", "Gila monster", "European green lizard", "chameleon", "Komodo dragon", "Nile crocodile", "American alligator", "triceratops", "worm snake", "ring-necked snake", "eastern hog-nosed snake", "smooth green snake", "kingsnake", "garter snake", "water snake", "vine snake", "night snake", "boa constrictor", "African rock python", "Indian cobra", "green mamba", "sea snake", "Saharan horned viper", "eastern diamondback rattlesnake", "sidewinder rattlesnake", "trilobite", "harvestman", "scorpion", "yellow garden spider", "barn spider", "European garden spider", "southern black widow", "tarantula", "wolf spider", "tick", "centipede", "black grouse", "ptarmigan", "ruffed grouse", "prairie grouse", "peafowl", "quail", "partridge", "african grey parrot", "macaw", "sulphur-crested cockatoo", "lorikeet", "coucal", "bee eater", "hornbill", "hummingbird", "jacamar", "toucan", "duck", "red-breasted merganser", "goose", "black swan", "tusker", "echidna", "platypus", "wallaby", "koala", "wombat", "jellyfish", "sea anemone", "brain coral", "flatworm", "nematode", "conch", "snail", "slug", "sea slug", "chiton", "chambered nautilus", "Dungeness crab", "rock crab", "fiddler crab", "red king crab", "American lobster", "spiny lobster", "crayfish", "hermit crab", "isopod", "white stork", "black stork", "spoonbill", "flamingo", "little blue heron", "great egret", "bittern bird", "crane bird", "limpkin", "common gallinule", "American coot", "bustard", "ruddy turnstone", "dunlin", "common redshank", "dowitcher", "oystercatcher", "pelican", "king penguin", "albatross", "grey whale", "killer whale", "dugong", "sea lion", "Chihuahua", "Japanese Chin", "Maltese", "Pekingese", "Shih Tzu", "King Charles Spaniel", "Papillon", "toy terrier", "Rhodesian Ridgeback", "Afghan Hound", "Basset Hound", "Beagle", "Bloodhound", "Bluetick Coonhound", "Black and Tan Coonhound", "Treeing Walker Coonhound", "English foxhound", "Redbone Coonhound", "borzoi", "Irish Wolfhound", "Italian Greyhound", "Whippet", "Ibizan Hound", "Norwegian Elkhound", "Otterhound", "Saluki", "Scottish Deerhound", "Weimaraner", "Staffordshire Bull Terrier", "American Staffordshire Terrier", "Bedlington Terrier", "Border Terrier", "Kerry Blue Terrier", "Irish Terrier", "Norfolk Terrier", "Norwich Terrier", "Yorkshire Terrier", "Wire Fox Terrier", "Lakeland Terrier", "Sealyham Terrier", "Airedale Terrier", "Cairn Terrier", "Australian Terrier", "Dandie Dinmont Terrier", "Boston Terrier", "Miniature Schnauzer", "Giant Schnauzer", "Standard Schnauzer", "Scottish Terrier", "Tibetan Terrier", "Australian Silky Terrier", "Soft-coated Wheaten Terrier", "West Highland White Terrier", "Lhasa Apso", "Flat-Coated Retriever", "Curly-coated Retriever", "Golden Retriever", "Labrador Retriever", "Chesapeake Bay Retriever", "German Shorthaired Pointer", "Vizsla", "English Setter", "Irish Setter", "Gordon Setter", "Brittany dog", "Clumber Spaniel", "English Springer Spaniel", "Welsh Springer Spaniel", "Cocker Spaniel", "Sussex Spaniel", "Irish Water Spaniel", "Kuvasz", "Schipperke", "Groenendael dog", "Malinois", "Briard", "Australian Kelpie", "Komondor", "Old English Sheepdog", "Shetland Sheepdog", "collie", "Border Collie", "Bouvier des Flandres dog", "Rottweiler", "German Shepherd Dog", "Dobermann", "Miniature Pinscher", "Greater Swiss Mountain Dog", "Bernese Mountain Dog", "Appenzeller Sennenhund", "Entlebucher Sennenhund", "Boxer", "Bullmastiff", "Tibetan Mastiff", "French Bulldog", "Great Dane", "St. Bernard", "husky", "Alaskan Malamute", "Siberian Husky", "Dalmatian", "Affenpinscher", "Basenji", "pug", "Leonberger", "Newfoundland dog", "Great Pyrenees dog", "Samoyed", "Pomeranian", "Chow Chow", "Keeshond", "brussels griffon", "Pembroke Welsh Corgi", "Cardigan Welsh Corgi", "Toy Poodle", "Miniature Poodle", "Standard Poodle", "Mexican hairless dog (xoloitzcuintli)", "grey wolf", "Alaskan tundra wolf", "red wolf or maned wolf", "coyote", "dingo", "dhole", "African wild dog", "hyena", "red fox", "kit fox", "Arctic fox", "grey fox", "tabby cat", "tiger cat", "Persian cat", "Siamese cat", "Egyptian Mau", "cougar", "lynx", "leopard", "snow leopard", "jaguar", "lion", "tiger", "cheetah", "brown bear", "American black bear", "polar bear", "sloth bear", "mongoose", "meerkat", "tiger beetle", "ladybug", "ground beetle", "longhorn beetle", "leaf beetle", "dung beetle", "rhinoceros beetle", "weevil", "fly", "bee", "ant", "grasshopper", "cricket insect", "stick insect", "cockroach", "praying mantis", "cicada", "leafhopper", "lacewing", "dragonfly", "damselfly", "red admiral butterfly", "ringlet butterfly", "monarch butterfly", "small white butterfly", "sulphur butterfly", "gossamer-winged butterfly", "starfish", "sea urchin", "sea cucumber", "cottontail rabbit", "hare", "Angora rabbit", "hamster", "porcupine", "fox squirrel", "marmot", "beaver", "guinea pig", "common sorrel horse", "zebra", "pig", "wild boar", "warthog", "hippopotamus", "ox", "water buffalo", "bison", "ram (adult male sheep)", "bighorn sheep", "Alpine ibex", "hartebeest", "impala (antelope)", "gazelle", "arabian camel", "llama", "weasel", "mink", "European polecat", "black-footed ferret", "otter", "skunk", "badger", "armadillo", "three-toed sloth", "orangutan", "gorilla", "chimpanzee", "gibbon", "siamang", "guenon", "patas monkey", "baboon", "macaque", "langur", "black-and-white colobus", "proboscis monkey", "marmoset", "white-headed capuchin", "howler monkey", "titi monkey", "Geoffroy's spider monkey", "common squirrel monkey", "ring-tailed lemur", "indri", "Asian elephant", "African bush elephant", "red panda", "giant panda", "snoek fish", "eel", "silver salmon", "rock beauty fish", "clownfish", "sturgeon", "gar fish", "lionfish", "pufferfish", "abacus", "abaya", "academic gown", "accordion", "acoustic guitar", "aircraft carrier", "airliner", "airship", "altar", "ambulance", "amphibious vehicle", "analog clock", "apiary", "apron", "trash can", "assault rifle", "backpack", "bakery", "balance beam", "balloon", "ballpoint pen", "Band-Aid", "banjo", "baluster / handrail", "barbell", "barber chair", "barbershop", "barn", "barometer", "barrel", "wheelbarrow", "baseball", "basketball", "bassinet", "bassoon", "swimming cap", "bath towel", "bathtub", "station wagon", "lighthouse", "beaker", "military hat (bearskin or shako)", "beer bottle", "beer glass", "bell tower", "baby bib", "tandem bicycle", "bikini", "ring binder", "binoculars", "birdhouse", "boathouse", "bobsleigh", "bolo tie", "poke bonnet", "bookcase", "bookstore", "bottle cap", "hunting bow", "bow tie", "brass memorial plaque", "bra", "breakwater", "breastplate", "broom", "bucket", "buckle", "bulletproof vest", "high-speed train", "butcher shop", "taxicab", "cauldron", "candle", "cannon", "canoe", "can opener", "cardigan", "car mirror", "carousel", "tool kit", "cardboard box / carton", "car wheel", "automated teller machine", "cassette", "cassette player", "castle", "catamaran", "CD player", "cello", "mobile phone", "chain", "chain-link fence", "chain mail", "chainsaw", "storage chest", "chiffonier", "bell or wind chime", "china cabinet", "Christmas stocking", "church", "movie theater", "cleaver", "cliff dwelling", "cloak", "clogs", "cocktail shaker", "coffee mug", "coffeemaker", "spiral or coil", "combination lock", "computer keyboard", "candy store", "container ship", "convertible", "corkscrew", "cornet", "cowboy boot", "cowboy hat", "cradle", "construction crane", "crash helmet", "crate", "infant bed", "Crock Pot", "croquet ball", "crutch", "cuirass", "dam", "desk", "desktop computer", "rotary dial telephone", "diaper", "digital clock", "digital watch", "dining table", "dishcloth", "dishwasher", "disc brake", "dock", "dog sled", "dome", "doormat", "drilling rig", "drum", "drumstick", "dumbbell", "Dutch oven", "electric fan", "electric guitar", "electric locomotive", "entertainment center", "envelope", "espresso machine", "face powder", "feather boa", "filing cabinet", "fireboat", "fire truck", "fire screen", "flagpole", "flute", "folding chair", "football helmet", "forklift", "fountain", "fountain pen", "four-poster bed", "freight car", "French horn", "frying pan", "fur coat", "garbage truck", "gas mask or respirator", "gas pump", "goblet", "go-kart", "golf ball", "golf cart", "gondola", "gong", "gown", "grand piano", "greenhouse", "radiator grille", "grocery store", "guillotine", "hair clip", "hair spray", "half-track", "hammer", "hamper", "hair dryer", "hand-held computer", "handkerchief", "hard disk drive", "harmonica", "harp", "combine harvester", "hatchet", "holster", "home theater", "honeycomb", "hook", "hoop skirt", "gymnastic horizontal bar", "horse-drawn vehicle", "hourglass", "iPod", "clothes iron", "carved pumpkin", "jeans", "jeep", "T-shirt", "jigsaw puzzle", "rickshaw", "joystick", "kimono", "knee pad", "knot", "lab coat", "ladle", "lampshade", "laptop computer", "lawn mower", "lens cap", "letter opener", "library", "lifeboat", "lighter", "limousine", "ocean liner", "lipstick", "slip-on shoe", "lotion", "music speaker", "loupe magnifying glass", "sawmill", "magnetic compass", "messenger bag", "mailbox", "tights", "one-piece bathing suit", "manhole cover", "maraca", "marimba", "mask", "matchstick", "maypole", "maze", "measuring cup", "medicine cabinet", "megalith", "microphone", "microwave oven", "military uniform", "milk can", "minibus", "miniskirt", "minivan", "missile", "mitten", "mixing bowl", "mobile home", "ford model t", "modem", "monastery", "monitor", "moped", "mortar and pestle", "graduation cap", "mosque", "mosquito net", "vespa", "mountain bike", "tent", "computer mouse", "mousetrap", "moving van", "muzzle", "metal nail", "neck brace", "necklace", "baby pacifier", "notebook computer", "obelisk", "oboe", "ocarina", "odometer", "oil filter", "pipe organ", "oscilloscope", "overskirt", "bullock cart", "oxygen mask", "product packet / packaging", "paddle", "paddle wheel", "padlock", "paintbrush", "pajamas", "palace", "pan flute", "paper towel", "parachute", "parallel bars", "park bench", "parking meter", "railroad car", "patio", "payphone", "pedestal", "pencil case", "pencil sharpener", "perfume", "Petri dish", "photocopier", "plectrum", "Pickelhaube", "picket fence", "pickup truck", "pier", "piggy bank", "pill bottle", "pillow", "ping-pong ball", "pinwheel", "pirate ship", "drink pitcher", "block plane", "planetarium", "plastic bag", "plate rack", "farm plow", "plunger", "Polaroid camera", "pole", "police van", "poncho", "pool table", "soda bottle", "plant pot", "potter's wheel", "power drill", "prayer rug", "printer", "prison", "missile", "projector", "hockey puck", "punching bag", "purse", "quill", "quilt", "race car", "racket", "radiator", "radio", "radio telescope", "rain barrel", "recreational vehicle", "fishing casting reel", "reflex camera", "refrigerator", "remote control", "restaurant", "revolver", "rifle", "rocking chair", "rotisserie", "eraser", "rugby ball", "ruler measuring stick", "sneaker", "safe", "safety pin", "salt shaker", "sandal", "sarong", "saxophone", "scabbard", "weighing scale", "school bus", "schooner", "scoreboard", "CRT monitor", "screw", "screwdriver", "seat belt", "sewing machine", "shield", "shoe store", "shoji screen / room divider", "shopping basket", "shopping cart", "shovel", "shower cap", "shower curtain", "ski", "balaclava ski mask", "sleeping bag", "slide rule", "sliding door", "slot machine", "snorkel", "snowmobile", "snowplow", "soap dispenser", "soccer ball", "sock", "solar thermal collector", "sombrero", "soup bowl", "keyboard space bar", "space heater", "space shuttle", "spatula", "motorboat", "spider web", "spindle", "sports car", "spotlight", "stage", "steam locomotive", "through arch bridge", "steel drum", "stethoscope", "scarf", "stone wall", "stopwatch", "stove", "strainer", "tram", "stretcher", "couch", "stupa", "submarine", "suit", "sundial", "sunglasses", "sunglasses", "sunscreen", "suspension bridge", "mop", "sweatshirt", "swim trunks / shorts", "swing", "electrical switch", "syringe", "table lamp", "tank", "tape player", "teapot", "teddy bear", "television", "tennis ball", "thatched roof", "front curtain", "thimble", "threshing machine", "throne", "tile roof", "toaster", "tobacco shop", "toilet seat", "torch", "totem pole", "tow truck", "toy store", "tractor", "semi-trailer truck", "tray", "trench coat", "tricycle", "trimaran", "tripod", "triumphal arch", "trolleybus", "trombone", "hot tub", "turnstile", "typewriter keyboard", "umbrella", "unicycle", "upright piano", "vacuum cleaner", "vase", "vaulted or arched ceiling", "velvet fabric", "vending machine", "vestment", "viaduct", "violin", "volleyball", "waffle iron", "wall clock", "wallet", "wardrobe", "military aircraft", "sink", "washing machine", "water bottle", "water jug", "water tower", "whiskey jug", "whistle", "hair wig", "window screen", "window shade", "Windsor tie", "wine bottle", "airplane wing", "wok", "wooden spoon", "wool", "split-rail fence", "shipwreck", "sailboat", "yurt", "website", "comic book", "crossword", "traffic or street sign", "traffic light", "dust jacket", "menu", "plate", "guacamole", "consomme", "hot pot", "trifle", "ice cream", "popsicle", "baguette", "bagel", "pretzel", "cheeseburger", "hot dog", "mashed potatoes", "cabbage", "broccoli", "cauliflower", "zucchini", "spaghetti squash", "acorn squash", "butternut squash", "cucumber", "artichoke", "bell pepper", "cardoon", "mushroom", "Granny Smith apple", "strawberry", "orange", "lemon", "fig", "pineapple", "banana", "jackfruit", "cherimoya (custard apple)", "pomegranate", "hay", "carbonara", "chocolate syrup", "dough", "meatloaf", "pizza", "pot pie", "burrito", "red wine", "espresso", "tea cup", "eggnog", "mountain", "bubble", "cliff", "coral reef", "geyser", "lakeshore", "promontory", "sandbar", "beach", "valley", "volcano", "baseball player", "bridegroom", "scuba diver", "rapeseed", "daisy", "yellow lady's slipper", "corn", "acorn", "rose hip", "horse chestnut seed", "coral fungus", "agaric", "gyromitra", "stinkhorn mushroom", "earth star fungus", "hen of the woods mushroom", "bolete", "corn cob", "toilet paper"]
    # places_classes = imagenet_classes = ["airfield", "airplane cabin", "airport terminal", "alcove", "alley", "amphitheater", "amusement arcade", "amusement park", "apartment building/outdoor", "aquarium", "aqueduct", "arcade", "arch", "archaelogical excavation", "archive", "arena/hockey", "arena/performance", "arena/rodeo", "army base", "art gallery", "art school", "art studio", "artists loft", "assembly line", "athletic field/outdoor", "atrium/public", "attic", "auditorium", "auto factory", "auto showroom", "badlands", "bakery/shop", "balcony/exterior", "balcony/interior", "ball pit", "ballroom", "bamboo forest", "bank vault", "banquet hall", "bar", "barn", "barndoor", "baseball field", "basement", "basketball court/indoor", "bathroom", "bazaar/indoor", "bazaar/outdoor", "beach", "beach house", "beauty salon", "bedchamber", "bedroom", "beer garden", "beer hall", "berth", "biology laboratory", "boardwalk", "boat deck", "boathouse", "bookstore", "booth/indoor", "botanical garden", "bow window/indoor", "bowling alley", "boxing ring", "bridge", "building facade", "bullring", "burial chamber", "bus interior", "bus station/indoor", "butchers shop", "butte", "cabin/outdoor", "cafeteria", "campsite", "campus", "canal/natural", "canal/urban", "candy store", "canyon", "car interior", "carrousel", "castle", "catacomb", "cemetery", "chalet", "chemistry lab", "childs room", "church/indoor", "church/outdoor", "classroom", "clean room", "cliff", "closet", "clothing store", "coast", "cockpit", "coffee shop", "computer room", "conference center", "conference room", "construction site", "corn field", "corral", "corridor", "cottage", "courthouse", "courtyard", "creek", "crevasse", "crosswalk", "dam", "delicatessen", "department store", "desert/sand", "desert/vegetation", "desert road", "diner/outdoor", "dining hall", "dining room", "discotheque", "doorway/outdoor", "dorm room", "downtown", "dressing room", "driveway", "drugstore", "elevator/door", "elevator lobby", "elevator shaft", "embassy", "engine room", "entrance hall", "escalator/indoor", "excavation", "fabric store", "farm", "fastfood restaurant", "field/cultivated", "field/wild", "field road", "fire escape", "fire station", "fishpond", "flea market/indoor", "florist shop/indoor", "food court", "football field", "forest/broadleaf", "forest path", "forest road", "formal garden", "fountain", "galley", "garage/indoor", "garage/outdoor", "gas station", "gazebo/exterior", "general store/indoor", "general store/outdoor", "gift shop", "glacier", "golf course", "greenhouse/indoor", "greenhouse/outdoor", "grotto", "gymnasium/indoor", "hangar/indoor", "hangar/outdoor", "harbor", "hardware store", "hayfield", "heliport", "highway", "home office", "home theater", "hospital", "hospital room", "hot spring", "hotel/outdoor", "hotel room", "house", "hunting lodge/outdoor", "ice cream parlor", "ice floe", "ice shelf", "ice skating rink/indoor", "ice skating rink/outdoor", "iceberg", "igloo", "industrial area", "inn/outdoor", "islet", "jacuzzi/indoor", "jail cell", "japanese garden", "jewelry shop", "junkyard", "kasbah", "kennel/outdoor", "kindergarden classroom", "kitchen", "lagoon", "lake/natural", "landfill", "landing deck", "laundromat", "lawn", "lecture room", "legislative chamber", "library/indoor", "library/outdoor", "lighthouse", "living room", "loading dock", "lobby", "lock chamber", "locker room", "mansion", "manufactured home", "market/indoor", "market/outdoor", "marsh", "martial arts gym", "mausoleum", "medina", "mezzanine", "moat/water", "mosque/outdoor", "motel", "mountain", "mountain path", "mountain snowy", "movie theater/indoor", "museum/indoor", "museum/outdoor", "music studio", "natural history museum", "nursery", "nursing home", "oast house", "ocean", "office", "office building", "office cubicles", "oilrig", "operating room", "orchard", "orchestra pit", "pagoda", "palace", "pantry", "park", "parking garage/indoor", "parking garage/outdoor", "parking lot", "pasture", "patio", "pavilion", "pet shop", "pharmacy", "phone booth", "physics laboratory", "picnic area", "pier", "pizzeria", "playground", "playroom", "plaza", "pond", "porch", "promenade", "pub/indoor", "racecourse", "raceway", "raft", "railroad track", "rainforest", "reception", "recreation room", "repair shop", "residential neighborhood", "restaurant", "restaurant kitchen", "restaurant patio", "rice paddy", "river", "rock arch", "roof garden", "rope bridge", "ruin", "runway", "sandbox", "sauna", "schoolhouse", "science museum", "server room", "shed", "shoe shop", "shopfront", "shopping mall/indoor", "shower", "ski resort", "ski slope", "sky", "skyscraper", "slum", "snowfield", "soccer field", "stable", "stadium/baseball", "stadium/football", "stadium/soccer", "stage/indoor", "stage/outdoor", "staircase", "storage room", "street", "subway station/platform", "supermarket", "sushi bar", "swamp", "swimming hole", "swimming pool/indoor", "swimming pool/outdoor", "synagogue/outdoor", "television room", "television studio", "temple/asia", "throne room", "ticket booth", "topiary garden", "tower", "toyshop", "train interior", "train station/platform", "tree farm", "tree house", "trench", "tundra", "underwater/ocean deep", "utility room", "valley", "vegetable garden", "veterinarians office", "viaduct", "village", "vineyard", "volcano", "volleyball court/outdoor", "waiting room", "water park", "water tower", "waterfall", "watering hole", "wave", "wet bar", "wheat field", "wind farm", "windmill", "yard", "youth hostel", "zen garden"]
    # imagenet_100_classes = imagenet_classes = ["robin", "Gila monster", "hognose snake", "garter snake", "green mamba", "garden spider", "Lorikeet", "goose", "rock crab", "fiddler crab", "American lobster", "little blue heron", "bittern", "Chihuahua", "Shih-Tzu", "papillon", "toy terrier", "Walker hound", "English foxhound", "borzoi", "Saluki", "American Staffordshire terrier", "Chesapeake Bay Retriever", "vizsla", "kuvasz", "komondor", "Rottweiler", "Doberman", "boxer", "Great Dane", "standard poodle", "Mexican hairless", "coyote", "African hunting dog", "red fox", "tabby", "meerkat", "dung beetle", "walking stick", "leafhopper", "hare", "wild boar", "gibbon", "langur", "ambulance", "comic book", "bassinet", "boathouse", "jump rope", "barber chair", "car wheel", "chime", "cinema", "cocktail shaker", "computer keyboard", "Dutch oven", "football helmet", "gasmask", "hard disc", "harmonica", "honeycomb", "iron", "jean", "lampshade", "laptop", "milk can", "minibus", "modem", "moped", "mortar", "mousetrap", "oboe", "park bench", "pedestal", "pickup", "ocarina", "purse", "reel", "rocking chair", "rotisserie", "safety pin", "sarong", "ski mask", "slide rule", "stretcher", "theater curtain", "throne", "tile roof", "trailer truck", "tub", "vacuum", "window screen", "wing", "head cabbage", "cauliflower", "pineapple", "spaghetti squash", "chocolate sauce", "gyromitra", "stinkhorn"]
    
    templates = [
        "itap of a {}.",
        "a bad photo of the {}.",
        "a origami {}.",
        "a photo of the large {}.",
        "a {} in a video game.",
        "art of the {}.",
        "a photo of the small {}.",
    ]
    zeroshot_weight = zeroshot_classifier(clipm, imagenet_classes, templates)
    print(zeroshot_weight.shape)
    zeroshot_weight = zeroshot_weight / zeroshot_weight.norm(dim=-1, keepdim=True)

    x = torch.randn(1, 3, 224, 224).to(next(image_encoder.parameters()).device)
    image_feature = image_encoder(x)
    feature_shape = image_feature.shape
   

    projection_net = ProjectionNet(feature_shape[1], args.class_number)


    projection_net.fc.weight.data = zeroshot_weight.t()
    projection_net.fc.bias.data = torch.zeros(args.class_number)

 


    clip_model = CLIPCLS(image_encoder, projection_net)
    clip_model = nn.DataParallel(clip_model).cuda()


    rank_pattern = {r".*conv.*": 8, "fc.fc": 128}
    config = peft.LoraConfig(rank_pattern=rank_pattern, target_modules=r".*(conv|fc\.fc).*")

    peft_model = peft.get_peft_model(clip_model, config).to("cuda")
    print(peft_model)
    for name, param in peft_model.named_parameters():
        print(f'Layer: {name}, Requires Grad: {param.requires_grad}')
       
        
    
    optimizer = torch.optim.AdamW(peft_model.parameters(), lr=args.lora_lr)
    peft_model.print_trainable_parameters()

    

    for _model in model:
        for name, param in _model.named_parameters():
            param.requires_grad = False

    for _model in model:
        _model = nn.DataParallel(_model).cuda()
   

    cudnn.benchmark = True

    print("process data from {}".format(args.data))
    normalize = transforms.Normalize(mean=[0.485, 0.456, 0.406],
                                     std=[0.229, 0.224, 0.225])

    train_dataset = torchvision.datasets.ImageFolder(
        args.data,
        transforms.Compose([
            transforms.RandomResizedCrop(224),
            transforms.RandomHorizontalFlip(),
            transforms.ToTensor(),
            normalize
        ])
    )
    
    generator = torch.Generator()
    generator.manual_seed(args.fkd_seed)
    sampler = torch.utils.data.RandomSampler(train_dataset, generator=generator)
    train_loader = torch.utils.data.DataLoader(
        train_dataset, batch_size=args.batch_size, shuffle=(sampler is None), sampler=sampler,
        num_workers=args.workers, pin_memory=True)
    
    val_loader = DataLoaderX(
        datasets.ImageFolder('/path/to/dataset/val', transforms.Compose([
            transforms.Resize(256),
            transforms.CenterCrop(224),
            transforms.ToTensor(),
            normalize,
        ])),
    batch_size=256, shuffle=False,
    num_workers=4, pin_memory=True)

    loss_function_kl = nn.KLDivLoss(reduction='batchmean')
    loss_function_dist = DISTLoss(tem=args.temperature)


    for epoch in tqdm(range(0, args.matrix_epoch)):
        peft_model.eval()
        total_acc = 0.
        total_loss = 0.
        for _model in model:
            _model.eval()

        for batch_idx, (images, target) in enumerate(train_loader):
           
            images = images.cuda()
            target = target.cuda()
            split_point = int(images.shape[0] // 2)
            total_output = []
            for _model in model:
                cat_output = []
                output = _model(images[:split_point])
                cat_output.append(output)
                output = _model(images[split_point:])
                cat_output.append(output)
                output = torch.cat(cat_output, 0)
                total_output.append(output)

            output = torch.stack(total_output, 0)
            output = output.mean(0)

            image_feature, output_trans = peft_model(images)
           
          
            optimizer.zero_grad()

            if args.loss_type == "kl":
                output_trans = F.log_softmax(output_trans / args.temperature, dim=1)
                output = F.softmax(output / args.temperature, dim=1)
                loss = loss_function_kl(output_trans, output)
            elif args.loss_type == "dist":
                loss = loss_function_dist(output_trans, output)
            elif args.loss_type == "mse_gt":
                loss = F.mse_loss(output_trans, output) + args.ce_weight * F.cross_entropy(output_trans, target)
            elif args.loss_type == "gt":
                loss = F.cross_entropy(output_trans, target)
            loss.backward()

            optimizer.step()

            import copy
            output_t = copy.deepcopy(output_trans.detach().cpu()).cuda()
            acc = (output_t.argmax(1) == target).float().sum() / output_t.shape[0]

            total_acc += acc
            total_loss += loss.item()
            
        test_acc = testvalidate(peft_model, val_loader, epoch)
        print("Top 1-Acc.:", round(total_acc.item() / (batch_idx+1) * 100, 3))
        print("Total Loss:", total_loss / (batch_idx+1))
        print("Test-Acc.:", test_acc)
        print("save..")
        peft_model.save_pretrained(args.output_dir+"/{}".format(epoch))

    
def validate(input, target):
    def accuracy1(output, target, topk=(1,)):
        maxk = max(topk)
        batch_size = target.size(0)

        _, pred = output.topk(maxk, 1, True, True)
        pred = pred.t()
        correct = pred.eq(target.reshape(1, -1).expand_as(pred))

        res = []
        for k in topk:
            correct_k = correct[:k].reshape(-1).float().sum(0)
            res.append(correct_k.mul_(100.0 / batch_size))
        return res

    with torch.no_grad():
        output = input
        prec1, prec5 = accuracy1(output.data, target, topk=(1, 5))

    print("Verifier accuracy: ", prec1.item())



if __name__ == '__main__':
    main()
