import os
import random
import argparse
import collections
import numpy as np
from PIL import Image
import shutil
from tqdm import tqdm
import torch
import torch.utils
import torch.nn as nn
import torch.optim as optim
import torch.utils.data.distributed
import torch.nn.functional as F
from torchvision import transforms
import torchvision.models as models
from select_utils import *

import clip
from tqdm import tqdm

import peft
from peft import PeftConfig, PeftModel

def lr_policy(lr_fn):
    def _alr(optimizer, iteration, epoch):
        lr = lr_fn(iteration, epoch)
        for param_group in optimizer.param_groups:
            param_group['lr'] = lr

    return _alr

def lr_cosine_policy(base_lr, warmup_length, epochs):
    def _lr_fn(iteration, epoch):
        if epoch < warmup_length:
            lr = base_lr * (epoch + 1) / warmup_length
        else:
            e = epoch - warmup_length
            es = epochs - warmup_length
            lr = 0.5 * (1 + np.cos(np.pi * e / es)) * base_lr
        return lr

    return lr_policy(_lr_fn)


def clip_color(image_tensor, use_fp16=False):
    if use_fp16:
        mean = np.array([0.485, 0.456, 0.406], dtype=np.float16)
        std = np.array([0.229, 0.224, 0.225], dtype=np.float16)
    else:
        mean = np.array([0.485, 0.456, 0.406])
        std = np.array([0.229, 0.224, 0.225])
    for c in range(3):
        m, s = mean[c], std[c]
        image_tensor[:, c] = torch.clamp(image_tensor[:, c], -m/s, (1 - m)/s)
    return image_tensor

def init_images(args, model=None, model_stu=None):
    trainset = ImageFolder(
        classes=args.classes,
        ipc=args.mipc,
        shuffle=True,
        root=args.train_dir,
        transform=None,
    )

    trainset.transform = transforms.Compose(
        [
            transforms.ToTensor(),
            MultiRandomCrop(
                num_crop=args.num_crop, size=args.input_size, factor=args.factor
            ),
            normalize,
        ]
    )
    train_loader = torch.utils.data.DataLoader(
        trainset,
        batch_size=args.mipc,
        shuffle=False,
        num_workers=args.workers,
        pin_memory=False,
    )
    augment = transforms.Compose([])
    for c, (images, labels) in enumerate(tqdm(train_loader)):
        images = selector(
            args.ipc * args.factor**2,
            model,
            images,
            labels,
            args.input_size,
            m=args.num_crop,
        )
    
        s = images.shape
        r = s[2] // args.factor
        images_updated = torch.randn((s[0], 3, r, r), requires_grad=True, device='cuda', dtype=float)
      
        images_resized = F.interpolate(images, size=(args.input_size // args.factor, args.input_size // args.factor), mode='bilinear', align_corners=True)
        images_updated.data = images_resized
      
        optimizer = optim.Adam([images_updated], lr=args.lr, betas=[0.5, 0.9], eps = 1e-8)
        lr_scheduler = lr_cosine_policy(args.lr, 0, args.epochs)


        for epoch in range(args.epochs):
            lr_scheduler(optimizer, epoch, epoch)
            images_upsize = F.interpolate(images_updated, size=(args.input_size, args.input_size))
            ori_feature, _ = model_stu(images)
            ds_feature, _ = model_stu(images_upsize)
        
            loss = F.mse_loss(ds_feature, ori_feature)
           
            optimizer.zero_grad()
            loss.backward()
        
            optimizer.step()
            print(loss.item())
        images_updated.data = clip_color(images_updated.data)
        images_save = mix_images(images_updated.detach().clone(), args.input_size, args.factor, args.ipc)
        save_images(args, denormalize(images_save), c)


def save_images(args, images, class_id):
    for id in range(images.shape[0]):
        dir_path = "{}/{:05d}".format(args.syn_data_path, class_id)
        place_to_store = dir_path + "/class{:05d}_id{:05d}.jpg".format(class_id, id)
        if not os.path.exists(dir_path):
            os.makedirs(dir_path)
        image_np = images[id].data.cpu().numpy().transpose((1, 2, 0))
        pil_image = Image.fromarray((image_np * 255).astype(np.uint8))
        pil_image.save(place_to_store)

class ProjectionNet(nn.Module):
    def __init__(self, input_features_dim, output_features_dim):
        super(ProjectionNet, self).__init__()
        self.fc = nn.Linear(input_features_dim, output_features_dim)  
    def forward(self, x):
        return self.fc(x)
    
class CLIPCLS(nn.Module):
    def __init__(self, model, linear):
        super(CLIPCLS, self).__init__()
        self.model = model
        self.fc = linear
    
    def forward(self, x):
        feature = self.model(x)
        out = self.fc(feature)
        return feature, out
    
def zeroshot_classifier(model, classnames, templates):
    with torch.no_grad():
        zeroshot_weights = []
        for classname in tqdm(classnames):
            texts = [template.format(classname) for template in templates]  # format with class
            texts = clip.tokenize(texts).cuda()  # tokenize
            class_embeddings = model.encode_text(texts)  # embed with text encoder
            class_embeddings /= class_embeddings.norm(dim=-1, keepdim=True)
            class_embedding = class_embeddings.mean(dim=0)
            class_embedding /= class_embedding.norm()
            zeroshot_weights.append(class_embedding)
        zeroshot_weights = torch.stack(zeroshot_weights, dim=1).cuda()
    return zeroshot_weights

def main(args):
    if not os.path.exists(args.syn_data_path):
        os.makedirs(args.syn_data_path)
    else:
        shutil.rmtree(args.syn_data_path)
        os.makedirs(args.syn_data_path)

    aux = ["resnet18"]
    model_teacher = []
    for name in aux:
        model = load_model(
            model_name=name,
            dataset=args.dataset,
            pretrained=True,
            classes=args.classes,
        )
        model = nn.DataParallel(model).cuda()
        model.eval()
        for p in model.parameters():
            p.requires_grad = False
        model_teacher.append(model)
    print(len(model_teacher))
    
    peft_model_id = args.adap_net
    peft_config = PeftConfig.from_pretrained(peft_model_id)
    print("peft_config: ", peft_config)

    clipm, preprocess = clip.load('RN50', device="cuda")
    clipm = clipm.to(dtype=torch.float32)
    image_encoder = clipm.visual
    if args.dataset == 'imagenet-1k':
        imagenet_classes = ["tench", "goldfish", "great white shark", "tiger shark", "hammerhead shark", "electric ray", "stingray", "rooster", "hen", "ostrich", "brambling", "goldfinch", "house finch", "junco", "indigo bunting", "American robin", "bulbul", "jay", "magpie", "chickadee", "American dipper", "kite (bird of prey)", "bald eagle", "vulture", "great grey owl", "fire salamander", "smooth newt", "newt", "spotted salamander", "axolotl", "American bullfrog", "tree frog", "tailed frog", "loggerhead sea turtle", "leatherback sea turtle", "mud turtle", "terrapin", "box turtle", "banded gecko", "green iguana", "Carolina anole", "desert grassland whiptail lizard", "agama", "frilled-necked lizard", "alligator lizard", "Gila monster", "European green lizard", "chameleon", "Komodo dragon", "Nile crocodile", "American alligator", "triceratops", "worm snake", "ring-necked snake", "eastern hog-nosed snake", "smooth green snake", "kingsnake", "garter snake", "water snake", "vine snake", "night snake", "boa constrictor", "African rock python", "Indian cobra", "green mamba", "sea snake", "Saharan horned viper", "eastern diamondback rattlesnake", "sidewinder rattlesnake", "trilobite", "harvestman", "scorpion", "yellow garden spider", "barn spider", "European garden spider", "southern black widow", "tarantula", "wolf spider", "tick", "centipede", "black grouse", "ptarmigan", "ruffed grouse", "prairie grouse", "peafowl", "quail", "partridge", "african grey parrot", "macaw", "sulphur-crested cockatoo", "lorikeet", "coucal", "bee eater", "hornbill", "hummingbird", "jacamar", "toucan", "duck", "red-breasted merganser", "goose", "black swan", "tusker", "echidna", "platypus", "wallaby", "koala", "wombat", "jellyfish", "sea anemone", "brain coral", "flatworm", "nematode", "conch", "snail", "slug", "sea slug", "chiton", "chambered nautilus", "Dungeness crab", "rock crab", "fiddler crab", "red king crab", "American lobster", "spiny lobster", "crayfish", "hermit crab", "isopod", "white stork", "black stork", "spoonbill", "flamingo", "little blue heron", "great egret", "bittern bird", "crane bird", "limpkin", "common gallinule", "American coot", "bustard", "ruddy turnstone", "dunlin", "common redshank", "dowitcher", "oystercatcher", "pelican", "king penguin", "albatross", "grey whale", "killer whale", "dugong", "sea lion", "Chihuahua", "Japanese Chin", "Maltese", "Pekingese", "Shih Tzu", "King Charles Spaniel", "Papillon", "toy terrier", "Rhodesian Ridgeback", "Afghan Hound", "Basset Hound", "Beagle", "Bloodhound", "Bluetick Coonhound", "Black and Tan Coonhound", "Treeing Walker Coonhound", "English foxhound", "Redbone Coonhound", "borzoi", "Irish Wolfhound", "Italian Greyhound", "Whippet", "Ibizan Hound", "Norwegian Elkhound", "Otterhound", "Saluki", "Scottish Deerhound", "Weimaraner", "Staffordshire Bull Terrier", "American Staffordshire Terrier", "Bedlington Terrier", "Border Terrier", "Kerry Blue Terrier", "Irish Terrier", "Norfolk Terrier", "Norwich Terrier", "Yorkshire Terrier", "Wire Fox Terrier", "Lakeland Terrier", "Sealyham Terrier", "Airedale Terrier", "Cairn Terrier", "Australian Terrier", "Dandie Dinmont Terrier", "Boston Terrier", "Miniature Schnauzer", "Giant Schnauzer", "Standard Schnauzer", "Scottish Terrier", "Tibetan Terrier", "Australian Silky Terrier", "Soft-coated Wheaten Terrier", "West Highland White Terrier", "Lhasa Apso", "Flat-Coated Retriever", "Curly-coated Retriever", "Golden Retriever", "Labrador Retriever", "Chesapeake Bay Retriever", "German Shorthaired Pointer", "Vizsla", "English Setter", "Irish Setter", "Gordon Setter", "Brittany dog", "Clumber Spaniel", "English Springer Spaniel", "Welsh Springer Spaniel", "Cocker Spaniel", "Sussex Spaniel", "Irish Water Spaniel", "Kuvasz", "Schipperke", "Groenendael dog", "Malinois", "Briard", "Australian Kelpie", "Komondor", "Old English Sheepdog", "Shetland Sheepdog", "collie", "Border Collie", "Bouvier des Flandres dog", "Rottweiler", "German Shepherd Dog", "Dobermann", "Miniature Pinscher", "Greater Swiss Mountain Dog", "Bernese Mountain Dog", "Appenzeller Sennenhund", "Entlebucher Sennenhund", "Boxer", "Bullmastiff", "Tibetan Mastiff", "French Bulldog", "Great Dane", "St. Bernard", "husky", "Alaskan Malamute", "Siberian Husky", "Dalmatian", "Affenpinscher", "Basenji", "pug", "Leonberger", "Newfoundland dog", "Great Pyrenees dog", "Samoyed", "Pomeranian", "Chow Chow", "Keeshond", "brussels griffon", "Pembroke Welsh Corgi", "Cardigan Welsh Corgi", "Toy Poodle", "Miniature Poodle", "Standard Poodle", "Mexican hairless dog (xoloitzcuintli)", "grey wolf", "Alaskan tundra wolf", "red wolf or maned wolf", "coyote", "dingo", "dhole", "African wild dog", "hyena", "red fox", "kit fox", "Arctic fox", "grey fox", "tabby cat", "tiger cat", "Persian cat", "Siamese cat", "Egyptian Mau", "cougar", "lynx", "leopard", "snow leopard", "jaguar", "lion", "tiger", "cheetah", "brown bear", "American black bear", "polar bear", "sloth bear", "mongoose", "meerkat", "tiger beetle", "ladybug", "ground beetle", "longhorn beetle", "leaf beetle", "dung beetle", "rhinoceros beetle", "weevil", "fly", "bee", "ant", "grasshopper", "cricket insect", "stick insect", "cockroach", "praying mantis", "cicada", "leafhopper", "lacewing", "dragonfly", "damselfly", "red admiral butterfly", "ringlet butterfly", "monarch butterfly", "small white butterfly", "sulphur butterfly", "gossamer-winged butterfly", "starfish", "sea urchin", "sea cucumber", "cottontail rabbit", "hare", "Angora rabbit", "hamster", "porcupine", "fox squirrel", "marmot", "beaver", "guinea pig", "common sorrel horse", "zebra", "pig", "wild boar", "warthog", "hippopotamus", "ox", "water buffalo", "bison", "ram (adult male sheep)", "bighorn sheep", "Alpine ibex", "hartebeest", "impala (antelope)", "gazelle", "arabian camel", "llama", "weasel", "mink", "European polecat", "black-footed ferret", "otter", "skunk", "badger", "armadillo", "three-toed sloth", "orangutan", "gorilla", "chimpanzee", "gibbon", "siamang", "guenon", "patas monkey", "baboon", "macaque", "langur", "black-and-white colobus", "proboscis monkey", "marmoset", "white-headed capuchin", "howler monkey", "titi monkey", "Geoffroy's spider monkey", "common squirrel monkey", "ring-tailed lemur", "indri", "Asian elephant", "African bush elephant", "red panda", "giant panda", "snoek fish", "eel", "silver salmon", "rock beauty fish", "clownfish", "sturgeon", "gar fish", "lionfish", "pufferfish", "abacus", "abaya", "academic gown", "accordion", "acoustic guitar", "aircraft carrier", "airliner", "airship", "altar", "ambulance", "amphibious vehicle", "analog clock", "apiary", "apron", "trash can", "assault rifle", "backpack", "bakery", "balance beam", "balloon", "ballpoint pen", "Band-Aid", "banjo", "baluster / handrail", "barbell", "barber chair", "barbershop", "barn", "barometer", "barrel", "wheelbarrow", "baseball", "basketball", "bassinet", "bassoon", "swimming cap", "bath towel", "bathtub", "station wagon", "lighthouse", "beaker", "military hat (bearskin or shako)", "beer bottle", "beer glass", "bell tower", "baby bib", "tandem bicycle", "bikini", "ring binder", "binoculars", "birdhouse", "boathouse", "bobsleigh", "bolo tie", "poke bonnet", "bookcase", "bookstore", "bottle cap", "hunting bow", "bow tie", "brass memorial plaque", "bra", "breakwater", "breastplate", "broom", "bucket", "buckle", "bulletproof vest", "high-speed train", "butcher shop", "taxicab", "cauldron", "candle", "cannon", "canoe", "can opener", "cardigan", "car mirror", "carousel", "tool kit", "cardboard box / carton", "car wheel", "automated teller machine", "cassette", "cassette player", "castle", "catamaran", "CD player", "cello", "mobile phone", "chain", "chain-link fence", "chain mail", "chainsaw", "storage chest", "chiffonier", "bell or wind chime", "china cabinet", "Christmas stocking", "church", "movie theater", "cleaver", "cliff dwelling", "cloak", "clogs", "cocktail shaker", "coffee mug", "coffeemaker", "spiral or coil", "combination lock", "computer keyboard", "candy store", "container ship", "convertible", "corkscrew", "cornet", "cowboy boot", "cowboy hat", "cradle", "construction crane", "crash helmet", "crate", "infant bed", "Crock Pot", "croquet ball", "crutch", "cuirass", "dam", "desk", "desktop computer", "rotary dial telephone", "diaper", "digital clock", "digital watch", "dining table", "dishcloth", "dishwasher", "disc brake", "dock", "dog sled", "dome", "doormat", "drilling rig", "drum", "drumstick", "dumbbell", "Dutch oven", "electric fan", "electric guitar", "electric locomotive", "entertainment center", "envelope", "espresso machine", "face powder", "feather boa", "filing cabinet", "fireboat", "fire truck", "fire screen", "flagpole", "flute", "folding chair", "football helmet", "forklift", "fountain", "fountain pen", "four-poster bed", "freight car", "French horn", "frying pan", "fur coat", "garbage truck", "gas mask or respirator", "gas pump", "goblet", "go-kart", "golf ball", "golf cart", "gondola", "gong", "gown", "grand piano", "greenhouse", "radiator grille", "grocery store", "guillotine", "hair clip", "hair spray", "half-track", "hammer", "hamper", "hair dryer", "hand-held computer", "handkerchief", "hard disk drive", "harmonica", "harp", "combine harvester", "hatchet", "holster", "home theater", "honeycomb", "hook", "hoop skirt", "gymnastic horizontal bar", "horse-drawn vehicle", "hourglass", "iPod", "clothes iron", "carved pumpkin", "jeans", "jeep", "T-shirt", "jigsaw puzzle", "rickshaw", "joystick", "kimono", "knee pad", "knot", "lab coat", "ladle", "lampshade", "laptop computer", "lawn mower", "lens cap", "letter opener", "library", "lifeboat", "lighter", "limousine", "ocean liner", "lipstick", "slip-on shoe", "lotion", "music speaker", "loupe magnifying glass", "sawmill", "magnetic compass", "messenger bag", "mailbox", "tights", "one-piece bathing suit", "manhole cover", "maraca", "marimba", "mask", "matchstick", "maypole", "maze", "measuring cup", "medicine cabinet", "megalith", "microphone", "microwave oven", "military uniform", "milk can", "minibus", "miniskirt", "minivan", "missile", "mitten", "mixing bowl", "mobile home", "ford model t", "modem", "monastery", "monitor", "moped", "mortar and pestle", "graduation cap", "mosque", "mosquito net", "vespa", "mountain bike", "tent", "computer mouse", "mousetrap", "moving van", "muzzle", "metal nail", "neck brace", "necklace", "baby pacifier", "notebook computer", "obelisk", "oboe", "ocarina", "odometer", "oil filter", "pipe organ", "oscilloscope", "overskirt", "bullock cart", "oxygen mask", "product packet / packaging", "paddle", "paddle wheel", "padlock", "paintbrush", "pajamas", "palace", "pan flute", "paper towel", "parachute", "parallel bars", "park bench", "parking meter", "railroad car", "patio", "payphone", "pedestal", "pencil case", "pencil sharpener", "perfume", "Petri dish", "photocopier", "plectrum", "Pickelhaube", "picket fence", "pickup truck", "pier", "piggy bank", "pill bottle", "pillow", "ping-pong ball", "pinwheel", "pirate ship", "drink pitcher", "block plane", "planetarium", "plastic bag", "plate rack", "farm plow", "plunger", "Polaroid camera", "pole", "police van", "poncho", "pool table", "soda bottle", "plant pot", "potter's wheel", "power drill", "prayer rug", "printer", "prison", "missile", "projector", "hockey puck", "punching bag", "purse", "quill", "quilt", "race car", "racket", "radiator", "radio", "radio telescope", "rain barrel", "recreational vehicle", "fishing casting reel", "reflex camera", "refrigerator", "remote control", "restaurant", "revolver", "rifle", "rocking chair", "rotisserie", "eraser", "rugby ball", "ruler measuring stick", "sneaker", "safe", "safety pin", "salt shaker", "sandal", "sarong", "saxophone", "scabbard", "weighing scale", "school bus", "schooner", "scoreboard", "CRT monitor", "screw", "screwdriver", "seat belt", "sewing machine", "shield", "shoe store", "shoji screen / room divider", "shopping basket", "shopping cart", "shovel", "shower cap", "shower curtain", "ski", "balaclava ski mask", "sleeping bag", "slide rule", "sliding door", "slot machine", "snorkel", "snowmobile", "snowplow", "soap dispenser", "soccer ball", "sock", "solar thermal collector", "sombrero", "soup bowl", "keyboard space bar", "space heater", "space shuttle", "spatula", "motorboat", "spider web", "spindle", "sports car", "spotlight", "stage", "steam locomotive", "through arch bridge", "steel drum", "stethoscope", "scarf", "stone wall", "stopwatch", "stove", "strainer", "tram", "stretcher", "couch", "stupa", "submarine", "suit", "sundial", "sunglasses", "sunglasses", "sunscreen", "suspension bridge", "mop", "sweatshirt", "swim trunks / shorts", "swing", "electrical switch", "syringe", "table lamp", "tank", "tape player", "teapot", "teddy bear", "television", "tennis ball", "thatched roof", "front curtain", "thimble", "threshing machine", "throne", "tile roof", "toaster", "tobacco shop", "toilet seat", "torch", "totem pole", "tow truck", "toy store", "tractor", "semi-trailer truck", "tray", "trench coat", "tricycle", "trimaran", "tripod", "triumphal arch", "trolleybus", "trombone", "hot tub", "turnstile", "typewriter keyboard", "umbrella", "unicycle", "upright piano", "vacuum cleaner", "vase", "vaulted or arched ceiling", "velvet fabric", "vending machine", "vestment", "viaduct", "violin", "volleyball", "waffle iron", "wall clock", "wallet", "wardrobe", "military aircraft", "sink", "washing machine", "water bottle", "water jug", "water tower", "whiskey jug", "whistle", "hair wig", "window screen", "window shade", "Windsor tie", "wine bottle", "airplane wing", "wok", "wooden spoon", "wool", "split-rail fence", "shipwreck", "sailboat", "yurt", "website", "comic book", "crossword", "traffic or street sign", "traffic light", "dust jacket", "menu", "plate", "guacamole", "consomme", "hot pot", "trifle", "ice cream", "popsicle", "baguette", "bagel", "pretzel", "cheeseburger", "hot dog", "mashed potatoes", "cabbage", "broccoli", "cauliflower", "zucchini", "spaghetti squash", "acorn squash", "butternut squash", "cucumber", "artichoke", "bell pepper", "cardoon", "mushroom", "Granny Smith apple", "strawberry", "orange", "lemon", "fig", "pineapple", "banana", "jackfruit", "cherimoya (custard apple)", "pomegranate", "hay", "carbonara", "chocolate syrup", "dough", "meatloaf", "pizza", "pot pie", "burrito", "red wine", "espresso", "tea cup", "eggnog", "mountain", "bubble", "cliff", "coral reef", "geyser", "lakeshore", "promontory", "sandbar", "beach", "valley", "volcano", "baseball player", "bridegroom", "scuba diver", "rapeseed", "daisy", "yellow lady's slipper", "corn", "acorn", "rose hip", "horse chestnut seed", "coral fungus", "agaric", "gyromitra", "stinkhorn mushroom", "earth star fungus", "hen of the woods mushroom", "bolete", "corn cob", "toilet paper"]
    elif args.dataset == 'imagenet-100':
        imagenet_classes = ["robin", "Gila monster", "hognose snake", "garter snake", "green mamba", "garden spider", "Lorikeet", "goose", "rock crab", "fiddler crab", "American lobster", "little blue heron", "bittern", "Chihuahua", "Shih-Tzu", "papillon", "toy terrier", "Walker hound", "English foxhound", "borzoi", "Saluki", "American Staffordshire terrier", "Chesapeake Bay Retriever", "vizsla", "kuvasz", "komondor", "Rottweiler", "Doberman", "boxer", "Great Dane", "standard poodle", "Mexican hairless", "coyote", "African hunting dog", "red fox", "tabby", "meerkat", "dung beetle", "walking stick", "leafhopper", "hare", "wild boar", "gibbon", "langur", "ambulance", "comic book", "bassinet", "boathouse", "jump rope", "barber chair", "car wheel", "chime", "cinema", "cocktail shaker", "computer keyboard", "Dutch oven", "football helmet", "gasmask", "hard disc", "harmonica", "honeycomb", "iron", "jean", "lampshade", "laptop", "milk can", "minibus", "modem", "moped", "mortar", "mousetrap", "oboe", "park bench", "pedestal", "pickup", "ocarina", "purse", "reel", "rocking chair", "rotisserie", "safety pin", "sarong", "ski mask", "slide rule", "stretcher", "theater curtain", "throne", "tile roof", "trailer truck", "tub", "vacuum", "window screen", "wing", "head cabbage", "cauliflower", "pineapple", "spaghetti squash", "chocolate sauce", "gyromitra", "stinkhorn"]
    elif args.dataset == 'places':
        imagenet_classes = ["airfield", "airplane cabin", "airport terminal", "alcove", "alley", "amphitheater", "amusement arcade", "amusement park", "apartment building/outdoor", "aquarium", "aqueduct", "arcade", "arch", "archaelogical excavation", "archive", "arena/hockey", "arena/performance", "arena/rodeo", "army base", "art gallery", "art school", "art studio", "artists loft", "assembly line", "athletic field/outdoor", "atrium/public", "attic", "auditorium", "auto factory", "auto showroom", "badlands", "bakery/shop", "balcony/exterior", "balcony/interior", "ball pit", "ballroom", "bamboo forest", "bank vault", "banquet hall", "bar", "barn", "barndoor", "baseball field", "basement", "basketball court/indoor", "bathroom", "bazaar/indoor", "bazaar/outdoor", "beach", "beach house", "beauty salon", "bedchamber", "bedroom", "beer garden", "beer hall", "berth", "biology laboratory", "boardwalk", "boat deck", "boathouse", "bookstore", "booth/indoor", "botanical garden", "bow window/indoor", "bowling alley", "boxing ring", "bridge", "building facade", "bullring", "burial chamber", "bus interior", "bus station/indoor", "butchers shop", "butte", "cabin/outdoor", "cafeteria", "campsite", "campus", "canal/natural", "canal/urban", "candy store", "canyon", "car interior", "carrousel", "castle", "catacomb", "cemetery", "chalet", "chemistry lab", "childs room", "church/indoor", "church/outdoor", "classroom", "clean room", "cliff", "closet", "clothing store", "coast", "cockpit", "coffee shop", "computer room", "conference center", "conference room", "construction site", "corn field", "corral", "corridor", "cottage", "courthouse", "courtyard", "creek", "crevasse", "crosswalk", "dam", "delicatessen", "department store", "desert/sand", "desert/vegetation", "desert road", "diner/outdoor", "dining hall", "dining room", "discotheque", "doorway/outdoor", "dorm room", "downtown", "dressing room", "driveway", "drugstore", "elevator/door", "elevator lobby", "elevator shaft", "embassy", "engine room", "entrance hall", "escalator/indoor", "excavation", "fabric store", "farm", "fastfood restaurant", "field/cultivated", "field/wild", "field road", "fire escape", "fire station", "fishpond", "flea market/indoor", "florist shop/indoor", "food court", "football field", "forest/broadleaf", "forest path", "forest road", "formal garden", "fountain", "galley", "garage/indoor", "garage/outdoor", "gas station", "gazebo/exterior", "general store/indoor", "general store/outdoor", "gift shop", "glacier", "golf course", "greenhouse/indoor", "greenhouse/outdoor", "grotto", "gymnasium/indoor", "hangar/indoor", "hangar/outdoor", "harbor", "hardware store", "hayfield", "heliport", "highway", "home office", "home theater", "hospital", "hospital room", "hot spring", "hotel/outdoor", "hotel room", "house", "hunting lodge/outdoor", "ice cream parlor", "ice floe", "ice shelf", "ice skating rink/indoor", "ice skating rink/outdoor", "iceberg", "igloo", "industrial area", "inn/outdoor", "islet", "jacuzzi/indoor", "jail cell", "japanese garden", "jewelry shop", "junkyard", "kasbah", "kennel/outdoor", "kindergarden classroom", "kitchen", "lagoon", "lake/natural", "landfill", "landing deck", "laundromat", "lawn", "lecture room", "legislative chamber", "library/indoor", "library/outdoor", "lighthouse", "living room", "loading dock", "lobby", "lock chamber", "locker room", "mansion", "manufactured home", "market/indoor", "market/outdoor", "marsh", "martial arts gym", "mausoleum", "medina", "mezzanine", "moat/water", "mosque/outdoor", "motel", "mountain", "mountain path", "mountain snowy", "movie theater/indoor", "museum/indoor", "museum/outdoor", "music studio", "natural history museum", "nursery", "nursing home", "oast house", "ocean", "office", "office building", "office cubicles", "oilrig", "operating room", "orchard", "orchestra pit", "pagoda", "palace", "pantry", "park", "parking garage/indoor", "parking garage/outdoor", "parking lot", "pasture", "patio", "pavilion", "pet shop", "pharmacy", "phone booth", "physics laboratory", "picnic area", "pier", "pizzeria", "playground", "playroom", "plaza", "pond", "porch", "promenade", "pub/indoor", "racecourse", "raceway", "raft", "railroad track", "rainforest", "reception", "recreation room", "repair shop", "residential neighborhood", "restaurant", "restaurant kitchen", "restaurant patio", "rice paddy", "river", "rock arch", "roof garden", "rope bridge", "ruin", "runway", "sandbox", "sauna", "schoolhouse", "science museum", "server room", "shed", "shoe shop", "shopfront", "shopping mall/indoor", "shower", "ski resort", "ski slope", "sky", "skyscraper", "slum", "snowfield", "soccer field", "stable", "stadium/baseball", "stadium/football", "stadium/soccer", "stage/indoor", "stage/outdoor", "staircase", "storage room", "street", "subway station/platform", "supermarket", "sushi bar", "swamp", "swimming hole", "swimming pool/indoor", "swimming pool/outdoor", "synagogue/outdoor", "television room", "television studio", "temple/asia", "throne room", "ticket booth", "topiary garden", "tower", "toyshop", "train interior", "train station/platform", "tree farm", "tree house", "trench", "tundra", "underwater/ocean deep", "utility room", "valley", "vegetable garden", "veterinarians office", "viaduct", "village", "vineyard", "volcano", "volleyball court/outdoor", "waiting room", "water park", "water tower", "waterfall", "watering hole", "wave", "wet bar", "wheat field", "wind farm", "windmill", "yard", "youth hostel", "zen garden"]


    templates = [
        "itap of a {}.",
        "a bad photo of the {}.",
        "a origami {}.",
        "a photo of the large {}.",
        "a {} in a video game.",
        "art of the {}.",
        "a photo of the small {}.",
    ]
    zeroshot_weight = zeroshot_classifier(clipm, imagenet_classes, templates)
    print(zeroshot_weight.shape)
    zeroshot_weight = zeroshot_weight / zeroshot_weight.norm(dim=-1, keepdim=True)
    
    x = torch.randn(1, 3, args.input_size, args.input_size).to(next(image_encoder.parameters()).device)
        
    image_feature = image_encoder(x)
    feature_shape = image_feature.shape

    projection_net = ProjectionNet(feature_shape[1], args.n_classes)


    projection_net.fc.weight.data = zeroshot_weight.t()
    projection_net.fc.bias.data = torch.zeros(args.n_classes)

    
    model_student = CLIPCLS(image_encoder, projection_net)
    model_student = nn.DataParallel(model_student).cuda()
    model_student = PeftModel.from_pretrained(model_student, peft_model_id)


    model_student.eval()
    for p in model_student.parameters():
        p.requires_grad = False

    init_images(args, model_teacher, model_student)


if __name__ == "__main__":
    parser = argparse.ArgumentParser("select and update")
    parser.add_argument("--syn_data_path", type=str, default="/path/to/syn/data")
    parser.add_argument("--arch_name", type=str, default="resnet18")
    parser.add_argument("--dataset", type=str, default="imagenet-1k")
    parser.add_argument("--mipc", type=int, default=300)
    parser.add_argument("--train_dir", type=str, default="/path/to/imagenet/train")
    parser.add_argument("--num_crop", type=int, default=5)
    parser.add_argument("--input_size", type=int)
    parser.add_argument("--factor", type=int, default=2)
    parser.add_argument("--workers", type=int, default=8)
    parser.add_argument("--ipc", type=int, default=10)
    parser.add_argument("--epochs", type=int, default=200)
    parser.add_argument("--lr", type=float, default=0.01)
    parser.add_argument("--adap_net", type=str)
    parser.add_argument("--n_classes")
    args = parser.parse_args()
    if args.dataset == "imagenet-1k":
        args.n_classes = 1000
        args.input_size = 224
    elif args.dataset == "places":
        args.n_classes = 365
        args.input_size = 224
    elif args.dataset == "imagenet-100":
        args.n_classes = 100
        args.input_size = 224
    args.classes = range(args.n_classes)
    main(args)
