import torch
from transformers import AutoModelForCausalLM, AutoTokenizer
from tqdm import tqdm
from datasets import Dataset
import logging


class scaled_module(torch.nn.Module):
    def __init__(self, module, id_, total, tep_len, name='llama'):
        super().__init__()
        self.module = module
        self.id = id_
        self.total = total
        self.tep_len = tep_len
        self.final_scale = 1
        self.get_final_scale(name)

    def get_final_scale(self, name):
        if '70b' in name.lower():
            self.final_scale = 2.0
            self.noise_scale = 0.425
        elif 'llama' in name.lower():
            self.final_scale = 1.0
            self.noise_scale = 0.2
        elif 'phi' in name.lower():
            self.final_scale = 2.0
            self.noise_scale = 0.5
        elif 'mistral' in name.lower():
            self.final_scale = 1.5
            self.noise_scale = 0.3
        elif 'gemma' in name.lower():
            self.final_scale = 1.5
            self.noise_scale = 0.35
        else:
            raise ValueError(f'no model scale setting for model {name}')

    def get_scales(self, x):
        scales = torch.ones_like(x[:, :, 0])
        if x.shape[1] > 1:
            if self.id < self.total - 1:
                scales[:, 0: self.tep_len] = 1 + self.noise_scale * torch.rand_like(scales[:, 0: self.tep_len])
            else:
                scales[:, 0: self.tep_len] = 1 / self.final_scale
        return scales[:, :, None]

    def forward(self, x, **kwargs):
        out = list(self.module(x, **kwargs))
        scales = self.get_scales(out[0])
        out[0] /= scales
        return out


class LLM_for_BoolQ:
    TOTAL_B = 10
    def __init__(self, model, tokenizer, model_name='', enable_privacy=False):
        self.model = model
        self.model.eval()
        self.tokenizer = tokenizer
        self.model_name = model_name
        self.enable_privacy = enable_privacy
        self.prompt_post = ("You are a helpful assistant. According to the passage above, answer the question from the user. You answer only with a 'The answer is: ' followed letter from the set {True., False.}: {LIKE SO: 'The answer is: True.'}.\n"
                           "user: ")
        self.prompt_ = ''
        self.TEP_LEN = 0

    def model_privacy_augment(self):
        print(f"**ENABLE PRIVACY BLOCK**")
        for i in range(self.TOTAL_B):
            self.model.model.layers[i] = scaled_module(self.model.model.layers[i], i, self.TOTAL_B, self.TEP_LEN, self.model_name)

    def remove_privacy_blocks(self):
        print(f"**REMOVE PRIVACY BLOCK**")
        for i in range(self.TOTAL_B):
            self.model.model.layers[i] = self.model.model.layers[i].module

    def generate(self, example):
        device = "cuda"
        passage = self.get_passage(example['passage'])
        prompt = passage + self.prompt_post + example['question']
        prompt_input = f"{prompt}?\nassistant: The answer is"
        model_inputs = self.tokenizer([prompt_input], return_tensors="pt").to(device)
        with torch.no_grad():
            generated_ids = self.model.generate(**model_inputs, max_new_tokens=4)
        generated_ids = [
            output_ids[len(input_ids):] for input_ids, output_ids in zip(model_inputs.input_ids, generated_ids)
        ]
        return self.tokenizer.batch_decode(generated_ids, skip_special_tokens=True)[0]

    def get_passage(self, passage):
        self.TEP_LEN = self.tokenizer([passage], return_tensors="pt").input_ids.shape[1]
        if self.enable_privacy:
            for i in range(self.TOTAL_B):
                self.model.model.layers[i].tep_len = self.TEP_LEN
        passage += "\n"
        return passage

    def evaluate(self, dataset):
        total = len(dataset)
        acc_count = 0
        fails = 0
        pbar = tqdm(range(total))
        for i in pbar:
            data = dataset[i]
            result = self.generate(data)
            answer = str(data['answer'])
            if answer.lower() in result.lower():
                acc_count += 1
            elif all(choice not in result.lower() for choice in ["true", "false"]):
                fails += 1
            pbar.set_postfix_str(f"acc: {round(acc_count/(i+1),2)}, fails: {fails}")
            if (i+1) % 10 == 0:
                log.info(f"{i+1} | accuracy: {acc_count/(i+1)} | fails: {fails}")
        log.info(f"{total} | accuracy: {acc_count / total} | fails: {fails}")
        return acc_count/total, fails, acc_count, total


if __name__ == "__main__":
    phi_path = "../pretrained_models/Phi-3-medium-128k-instruct"
    gemma_path = "../pretrained_models/gemma-2-9b-it"
    llama_path = "../pretrained_models/Meta-Llama-3-8B-Instruct"
    mistral_path = "../pretrained_models/Mistral-7B-Instruct-v0.3"
    llama70_path = "../pretrained_models/Meta-Llama-3-70B-Instruct-AWQ"

    model_path = gemma_path
    enable_privacy = True

    log = logging.getLogger(f'BoolQ')
    log.setLevel(level=logging.DEBUG)
    log_name = model_path.split('/')[-1].split('-')[0].lower()
    model_name = model_path.split('/')[-1]
    file_name = f'boolq_{log_name}.log' if not enable_privacy else f'boolq_{log_name}_privacy.log'
    handler = logging.FileHandler(f'./boolq_eva/{file_name}', encoding='utf-8', mode='w')
    handler.setLevel(logging.INFO)
    log.addHandler(handler)

    # init model
    model = AutoModelForCausalLM.from_pretrained(model_path, device_map='auto', torch_dtype=torch.bfloat16)
    tokenizer = AutoTokenizer.from_pretrained(model_path)
    model.generation_config.pad_token_id = tokenizer.eos_token_id
    llm_boolq = LLM_for_BoolQ(model=model, tokenizer=tokenizer, model_name=model_name, enable_privacy=enable_privacy)

    # init dataset
    test_path = "../dataset/boolq/validation-00000-of-00001.parquet"
    dataset = Dataset.from_parquet(test_path)

    # evaluating BoolQ
    print(f'start evaluating {model_name}!')
    if enable_privacy:
        llm_boolq.model_privacy_augment()
    acc, fails, acc_count_, total_ = llm_boolq.evaluate(dataset)
    if enable_privacy:
        llm_boolq.remove_privacy_blocks()
    logging.shutdown()
