from datasets import load_from_disk
from transformers import AutoModelForCausalLM, AutoTokenizer
import torch
from tqdm import tqdm
import logging


def get_eos_str(name):
    if 'llama' in name.lower():
        return "<|eot_id|>"
    elif 'phi' in name.lower():
        return "<|end|>"
    elif 'mistral' in name.lower():
        return "[/INST]"
    elif 'gemma' in name.lower():
        return "<end_of_turn>"
    else:
        raise ValueError(f'no eos_string for model {name}')


def format_number_with_commas(number):
    if isinstance(number, str):
        if ',' in number:
            return number.replace(',', '')
        number = float(number) if '.' in number else int(number)
    return "{:,}".format(number)


class scaled_module(torch.nn.Module):
    def __init__(self, module, id_, total, tep_len_pre, tep_len, name='llama'):
        super().__init__()
        self.module = module
        self.id = id_
        self.total = total
        self.len_init = tep_len_pre
        self.tep_len = tep_len
        self.final_scale = 1
        self.noise_scale = 0.5
        self.get_final_scale(name)

    def get_final_scale(self, name):
        if '70b' in name.lower():
            self.final_scale = 2.0
            self.noise_scale = 0.425
        elif 'llama' in name.lower():
            self.final_scale = 1.0
            self.noise_scale = 0.2
        elif 'phi' in name.lower():
            self.final_scale = 2.0
            self.noise_scale = 0.5
        elif 'mistral' in name.lower():
            self.final_scale = 1.5
            self.noise_scale = 0.3
        elif 'gemma' in name.lower():
            self.final_scale = 1.5
            self.noise_scale = 0.35
        else:
            raise ValueError(f'no model scale setting for model {name}')

    def get_scales(self, x):
        scales = torch.ones_like(x[:, :, 0])
        if x.shape[1] > 1:
            final = self.len_init + self.tep_len
            if self.id < self.total - 1:
                scales[:, self.len_init:final] = 1 + self.noise_scale * torch.rand_like(scales[:, self.len_init:final])
            else:
                scales[:, self.len_init:final] = 1 / self.final_scale
        return scales[:, :, None]

    def forward(self, x, **kwargs):
        out = list(self.module(x, **kwargs))
        scales = self.get_scales(out[0])
        out[0] /= scales
        return out


class LLM_for_GSM8K:
    TOTAL_B = 10
    def __init__(self, model, tokenizer, model_name='', enable_privacy=False):
        self.model = model
        self.model.eval()
        self.tokenizer = tokenizer
        self.model_name = model_name
        self.enable_privacy = enable_privacy
        eot_str = get_eos_str(model_name)
        self.prompt_pre = f"You are a helpful and concise digital assistant. You are required to solve the following question. The final answer should be given with '#### ' followed by the correct value and '{eot_str}', LIKE SO '#### 10 {eot_str}', OR '#### 123 {eot_str}', OR '#### 45 {eot_str}'.\n\n"
        self.prompt_ = self.prompt_pre
        self.TEP_LEN_pre = self.tokenizer([self.prompt_pre], return_tensors="pt").input_ids.shape[1]
        self.TEP_LEN = 1
        self.terminators = [
            self.tokenizer.eos_token_id,
            self.tokenizer.convert_tokens_to_ids(eot_str),
        ]

    def generate(self, example):
        device = "cuda"
        question = self.get_question(example['question'])
        prompt = self.prompt_ + question + "\nassistant: Let's think step by step.\n"
        prompt_input = f"{prompt}\n"
        model_inputs = self.tokenizer([prompt_input], return_tensors="pt").to(device)
        with torch.no_grad():
            generated_ids = self.model.generate(**model_inputs, max_new_tokens=512, eos_token_id=self.terminators)
        generated_ids = [
            output_ids[len(input_ids):] for input_ids, output_ids in zip(model_inputs.input_ids, generated_ids)
        ]
        return self.tokenizer.batch_decode(generated_ids, skip_special_tokens=True)[0]

    def get_question(self, passage):
        if self.enable_privacy:
            self.TEP_LEN = self.tokenizer([passage], return_tensors="pt").input_ids.shape[1]
            for i in range(self.TOTAL_B):
                self.model.model.layers[i].tep_len = self.TEP_LEN
        return passage

    def model_privacy_augment(self):
        print(f"**ENABLE PRIVACY BLOCK**")
        for i in range(self.TOTAL_B):
            self.model.model.layers[i] = scaled_module(self.model.model.layers[i], i, self.TOTAL_B, self.TEP_LEN_pre, self.TEP_LEN, self.model_name)

    def remove_privacy_blocks(self):
        print("**REMOVE PRIVACY BLOCK**")
        for i in range(self.TOTAL_B):
            self.model.model.layers[i] = self.model.model.layers[i].module

    def evaluate(self, dataset):
        total = len(dataset)
        acc_count = 0
        pbar = tqdm(range(total))
        for i in pbar:
            data = dataset[i]
            result = self.generate(data)
            answer = data['answer'].split('#')[-1].strip()
            if f'#### {answer}' in result or f'#### {format_number_with_commas(answer)}' in result:
                acc_count += 1
            pbar.set_postfix_str(f"acc: {round(acc_count/(i+1),2)}")
            if (i+1) % 1 == 0:
                log.info(f"{i} | reference: {answer} | response: {result}")
                log.info(f"***********{i+1} | accuracy: {acc_count/(i+1)}***********\n")
        log.info(f"{total} | accuracy: {acc_count / total}")


if __name__ == "__main__":
    phi_path = "../pretrained_models/Phi-3-medium-128k-instruct"
    gemma_path = "../pretrained_models/gemma-2-9b-it"
    llama_path = "../pretrained_models/Meta-Llama-3-8B-Instruct"
    mistral_path = "../pretrained_models/Mistral-7B-Instruct-v0.3"
    llama70_path = "../pretrained_models/Meta-Llama-3-70B-Instruct-AWQ"

    # init dataset
    path = "../dataset_save_disk/gsm8k"
    dataset_ = load_from_disk(path)
    dataset = dataset_['test']
    temp_data = dataset_['train']

    model_path = gemma_path
    enable_privacy = True

    log = logging.getLogger(f'gsm8k')
    log.setLevel(level=logging.DEBUG)
    log_name = model_path.split('/')[-1].split('-')[0].lower()
    model_name = model_path.split('/')[-1]
    file_name = f'gsm8k_{log_name}.log' if not enable_privacy else f'gsm8k_{log_name}_privacy.log'
    handler = logging.FileHandler(f'./gsm8k_eva/{file_name}', encoding='utf-8', mode='w')
    handler.setLevel(logging.INFO)
    log.addHandler(handler)

    model = AutoModelForCausalLM.from_pretrained(model_path, device_map='auto', torch_dtype=torch.bfloat16)
    model.eval()
    tokenizer = AutoTokenizer.from_pretrained(model_path)
    model.generation_config.pad_token_id = tokenizer.eos_token_id

    llm_gsm8k = LLM_for_GSM8K(model, tokenizer, model_name, enable_privacy)
    print(f'start evaluating {model_name}!')
    if enable_privacy:
        llm_gsm8k.model_privacy_augment()
    llm_gsm8k.evaluate(dataset)
