import torch
from transformers import AutoModelForCausalLM, AutoTokenizer
from tqdm import tqdm
from datasets import load_from_disk
import logging


class scaled_module(torch.nn.Module):
    def __init__(self, module, id_, total, tep_len_pre, tep_len, name='llama'):
        super().__init__()
        self.module = module
        self.id = id_
        self.total = total
        self.len_init = tep_len_pre
        self.tep_len = tep_len
        self.final_scale = 1
        self.get_final_scale(name)

    def get_final_scale(self, name):
        if '70b' in name.lower():
            self.final_scale = 2.0
            self.noise_scale = 0.425
        elif 'llama' in name.lower():
            self.final_scale = 1.0
            self.noise_scale = 0.2
        elif 'phi' in name.lower():
            self.final_scale = 2.0
            self.noise_scale = 0.5
        elif 'mistral' in name.lower():
            self.final_scale = 1.5
            self.noise_scale = 0.3
        elif 'gemma' in name.lower():
            self.final_scale = 1.5
            self.noise_scale = 0.35
        else:
            raise ValueError(f'no model scale setting for model {name}')

    def get_scales(self, x):
        scales = torch.ones_like(x[:, :, 0])
        if x.shape[1] > 1:
            final = self.len_init+self.tep_len
            if self.id < self.total - 1:
                scales[:, self.len_init:final] = 1 + self.noise_scale * torch.rand_like(scales[:, self.len_init:final])
            else:
                scales[:, self.len_init:final] = 1 / self.final_scale
        return scales[:, :, None]

    def forward(self, x, **kwargs):
        out = list(self.module(x, **kwargs))
        scales = self.get_scales(out[0])
        out[0] /= scales
        return out


class LLM_for_HSwag:
    TOTAL_B = 10
    def __init__(self, model, tokenizer, model_name='', enable_privacy=False):
        self.model = model
        self.model.eval()
        self.tokenizer = tokenizer
        self.model_name = model_name
        self.enable_privacy = enable_privacy
        self.prompt_pre = "You are a helpful and concise assistant. You need to choose the best choice for the second half of the given sentence. You reply only with a 'The best answer is: ' followed letter from the set {A., B., C., D.}: {REPLY WITH ONLY THE STRING 'The best answer is: ' FOLLOWED BY THE CORRECT ANSWER's LETTER, LIKE SO: 'The best answer is: B.'}. Any extra words are forbidden.\n\n"
        self.prompt_ = self.prompt_pre
        self.TEP_LEN_pre = self.tokenizer([self.prompt_pre], return_tensors="pt").input_ids.shape[1]
        self.TEP_LEN = 0

    def model_privacy_augment(self):
        print(f"**ENABLE PRIVACY BLOCK**")
        for i in range(self.TOTAL_B):
            self.model.model.layers[i] = scaled_module(self.model.model.layers[i], i, self.TOTAL_B, self.TEP_LEN_pre,
                                                       self.TEP_LEN, self.model_name)

    def remove_privacy_blocks(self):
        print(f"**REMOVE PRIVACY BLOCK**")
        for i in range(self.TOTAL_B):
            self.model.model.layers[i] = self.model.model.layers[i].module

    def generate(self, example):
        device = "cuda"
        context = self.get_context(example['ctx'])
        choices = self.get_choices(example['endings'])
        prompt = self.prompt_ + context + choices
        prompt_input = f"{prompt}assistant: The best answer is"
        model_inputs = self.tokenizer([prompt_input], return_tensors="pt").to(device)
        with torch.no_grad():
            generated_ids = self.model.generate(**model_inputs, max_new_tokens=4)
        generated_ids = [
            output_ids[len(input_ids):] for input_ids, output_ids in zip(model_inputs.input_ids, generated_ids)
        ]
        return self.tokenizer.batch_decode(generated_ids, skip_special_tokens=True)[0]

    def get_context(self, passage):
        self.TEP_LEN = self.tokenizer([passage], return_tensors="pt").input_ids.shape[1]
        if self.enable_privacy:
            for i in range(self.TOTAL_B):
                self.model.model.layers[i].tep_len = self.TEP_LEN
        passage += "\nChoices:\n"
        return passage

    def get_choices(self, choices_list):
        prompt_c = ""
        prompt_c += ("A. " + choices_list[0] + "\n")
        prompt_c += ("B. " + choices_list[1] + "\n")
        prompt_c += ("C. " + choices_list[2] + "\n")
        prompt_c += ("D. " + choices_list[3] + "\n\n")
        return prompt_c

    def evaluate(self, dataset):
        answer_map = ['A.', 'B.', 'C.', 'D.']
        answer_map2 = ['A ', 'B ', 'C ', 'D ']
        total = len(dataset)
        acc_count = 0
        fails = 0
        pbar = tqdm(range(total))
        for i in pbar:
            data = dataset[i]
            result = self.generate(data)
            answer = answer_map[int(data['label'])]
            answer2 = answer_map2[int(data['label'])]
            if answer in result or answer2 in result:
                acc_count += 1
            elif all(choice not in result for choice in answer_map+answer_map2):
                fails += 1
                print(result)
            pbar.set_postfix_str(f"acc: {round(acc_count/(i+1),2)}, fails: {fails}")
            if (i+1) % 10 == 0:
                # log.info(f"{i} | reference: {answer} | response: {result}")
                log.info(f"{i+1} | accuracy: {acc_count/(i+1)} | fails: {fails}")
        log.info(f"{total} | accuracy: {acc_count / total} | fails: {fails}")
        return acc_count/total, fails, acc_count, total


if __name__ == "__main__":
    phi_path = "../pretrained_models/Phi-3-medium-128k-instruct"
    gemma_path = "../pretrained_models/gemma-2-9b-it"
    llama_path = "../pretrained_models/Meta-Llama-3-8B-Instruct"
    mistral_path = "../pretrained_models/Mistral-7B-Instruct-v0.3"
    llama70_path = "../pretrained_models/Meta-Llama-3-70B-Instruct-AWQ"

    model_path = gemma_path
    enable_privacy = True

    # init dataset
    path = '../dataset_save_disk/hellaswag/'
    dataset_ = load_from_disk(path)
    dataset = dataset_['validation']
    dataset_shot = dataset_['train']
    # init logger
    log = logging.getLogger(f'testHSwag')
    log.setLevel(level=logging.DEBUG)
    log_name = model_path.split('/')[-1].split('-')[0].lower()
    model_name = model_path.split('/')[-1]
    file_name = f'hswag_{log_name}.log' if not enable_privacy else f'hswag_{log_name}_privacy.log'
    handler = logging.FileHandler(f'./hswag_eva/{file_name}', encoding='utf-8', mode='w')
    handler.setLevel(logging.INFO)
    log.addHandler(handler)

    # init model
    model = AutoModelForCausalLM.from_pretrained(model_path, device_map='auto', torch_dtype=torch.bfloat16)
    tokenizer = AutoTokenizer.from_pretrained(model_path)
    model.generation_config.pad_token_id = tokenizer.eos_token_id
    llm_hswag = LLM_for_HSwag(model=model, tokenizer=tokenizer, model_name=model_name, enable_privacy=enable_privacy)

    # evaluating HellaSwag
    print(f'start evaluating {model_name}!')
    if enable_privacy:
        llm_hswag.model_privacy_augment()
    acc, fails, acc_count_, total_ = llm_hswag.evaluate(dataset)
    if enable_privacy:
        llm_hswag.remove_privacy_blocks()
    logging.shutdown()
