import torch
from transformers import AutoModelForCausalLM, AutoTokenizer
from tqdm import tqdm
from datasets import load_from_disk
import logging
import json


def get_eos_str(name):
    if 'llama' in name.lower():
        return "<|eot_id|>"
    else:
        raise ValueError(f'no eos_string for model {name}')


class scaled_module(torch.nn.Module):
    def __init__(self, module, id_, total, len_init, tep_len, name='llama'):
        super().__init__()
        self.module = module
        self.id = id_
        self.total = total
        self.len_init = len_init
        self.tep_len = tep_len
        self.final_scale = 1
        self.noise_scale = 0.5
        self.model_name = name
        self.get_final_scale(name)

    def get_final_scale(self, name):
        if 'llama' in name.lower():
            self.final_scale = 1.0
            self.noise_scale = 0.2
        else:
            raise ValueError(f'no model scale setting for model {name}')

    def get_scales(self, x):
        scales = torch.ones_like(x[:, :, 0])
        if x.shape[1] > 1:
            if self.id < self.total - 1:
                scales[:, self.len_init:self.tep_len] = 1 + self.noise_scale * torch.rand_like(scales[:, self.len_init:self.tep_len])
            else:
                scales[:, self.len_init:self.tep_len] = 1 / self.final_scale
        return scales[:, :, None]

    def forward(self, x, **kwargs):
        out = list(self.module(x, **kwargs))
        scales = self.get_scales(out[0])
        out[0] /= scales
        return out


class LLM_for_HEVAL:
    TOTAL_B = 10
    def __init__(self, model, tokenizer, model_name='', privacy=False):
        self.model = model
        self.model.eval()
        self.tokenizer = tokenizer
        self.model_name = model_name
        self.enable_privacy = privacy
        self.eot_str = get_eos_str(model_name)
        self.terminators = [
            self.tokenizer.eos_token_id,
            self.tokenizer.convert_tokens_to_ids(self.eot_str),
        ]
        self.prompt_pre = f"You are a concise Python programming assistant. You are required to complete the code of the function.\n\n"
        self.prompt_ = self.prompt_pre
        self.LEN_INIT = self.tokenizer([self.prompt_pre], return_tensors="pt").input_ids.shape[1]
        self.TEP_LEN = 1

    def model_privacy_augment(self):
        print(f"**ENABLE PRIVACY BLOCK, from {self.LEN_INIT} to xxx**")
        for i in range(self.TOTAL_B):
            self.model.model.layers[i] = scaled_module(self.model.model.layers[i], i, self.TOTAL_B, self.LEN_INIT, self.TEP_LEN, self.model_name)

    def remove_privacy_blocks(self):
        print("**REMOVE PRIVACY BLOCK**")
        for i in range(self.TOTAL_B):
            self.model.model.layers[i] = self.model.model.layers[i].module

    def get_all_prompt(self, prompt):
        prompt = self.prompt_ + prompt
        if self.enable_privacy:
            self.TEP_LEN = self.tokenizer([prompt], return_tensors="pt").input_ids.shape[1]
            for i in range(self.TOTAL_B):
                self.model.model.layers[i].tep_len = self.TEP_LEN
        return prompt

    def generate(self, data):
        device = "cuda"
        prompt_input = self.get_all_prompt(data['prompt'])
        model_inputs = self.tokenizer([prompt_input], return_tensors="pt").to(device)
        with torch.no_grad():
            generated_ids = self.model.generate(**model_inputs, max_new_tokens=512, eos_token_id=self.terminators)
        generated_ids = [
            output_ids[len(input_ids):] for input_ids, output_ids in zip(model_inputs.input_ids, generated_ids)
        ]
        return self.tokenizer.batch_decode(generated_ids, skip_special_tokens=True)[0]

    def evaluate(self, dataset):
        total = len(dataset)
        acc_count = 0
        pbar = tqdm(range(total))
        saved = []
        for i in pbar:
            tmp_data = {}
            data = dataset[i]
            result = self.generate(data)
            log.info(f"{i}\nreference:\n{data['canonical_solution']}\nresponse:\n{result}\n")
            tmp_data['data'] = data
            tmp_data['result'] = result
            saved.append(tmp_data)
        file_name = f'heval_eva/heval_{self.model_name}.json' if not self.enable_privacy \
            else f'heval_eva/heval_{self.model_name}_privacy.json'
        json_file = open(file_name, encoding='utf-8', mode='w')
        json.dump(saved, json_file, indent=4)
        return acc_count/total, acc_count, total


if __name__ == "__main__":
    llama_path = "../pretrained_models/Meta-Llama-3-8B-Instruct"

    model_path = llama_path
    enable_privacy = True

    # init logger
    model_name = model_path.split('/')[-1]
    log_name = model_path.split('/')[-1].split('-')[0].lower()
    log = logging.getLogger(f'test{log_name}')
    log.setLevel(level=logging.DEBUG)
    file_name = f'heval_{log_name}.log' if not enable_privacy else f'heval_{log_name}_privacy.log'
    handler = logging.FileHandler(f'./heval_eva/{file_name}', encoding='utf-8', mode='w')
    handler.setLevel(logging.INFO)
    log.addHandler(handler)

    model = AutoModelForCausalLM.from_pretrained(model_path, device_map='auto', torch_dtype=torch.bfloat16)
    tokenizer = AutoTokenizer.from_pretrained(model_path)
    model.generation_config.pad_token_id = tokenizer.eos_token_id

    llm_heval = LLM_for_HEVAL(model=model, tokenizer=tokenizer, model_name=model_name, privacy=enable_privacy)

    path = '../dataset_save_disk/humaneval/'
    dataset = load_from_disk(path)
    print(f'start evaluating {model_name}!')
    if enable_privacy:
        llm_heval.model_privacy_augment()
    acc, acc_count_, total_ = llm_heval.evaluate(dataset["test"])
    if enable_privacy:
        llm_heval.remove_privacy_blocks()
    logging.shutdown()
