import os
import torch
from transformers import AutoModelForCausalLM, AutoTokenizer
from tqdm import tqdm
from datasets import load_from_disk
import logging


def get_eos_str(name):
    if 'llama' in name.lower():
        return "<|eot_id|>"
    elif 'phi' in name.lower():
        return "<|end|>"
    elif 'mistral' in name.lower():
        return "[/INST]"
    elif 'gemma' in name.lower():
        return "<end_of_turn>"
    else:
        raise ValueError(f'no eos_string for model {name}')


class scaled_module(torch.nn.Module):
    def __init__(self, module, id_, total, tep_len, name='llama'):
        super().__init__()
        self.module = module
        self.id = id_
        self.total = total
        self.tep_len = tep_len
        self.final_scale = 1
        self.noise_scale = 0.5
        self.get_final_scale(name)

    def get_final_scale(self, name):
        if '70b' in name.lower():
            self.final_scale = 2.0
            self.noise_scale = 0.425
        elif 'llama' in name.lower():
            self.final_scale = 1.0
            self.noise_scale = 0.2
        elif 'phi' in name.lower():
            self.final_scale = 2.0
            self.noise_scale = 0.5
        elif 'mistral' in name.lower():
            self.final_scale = 1.5
            self.noise_scale = 0.3
        elif 'gemma' in name.lower():
            self.final_scale = 1.5
            self.noise_scale = 0.35
        else:
            raise ValueError(f'no model scale setting for model {name}')

    def get_scales(self, x):
        scales = torch.ones_like(x[:, :, 0])
        if x.shape[1] > 1:
            if self.id < self.total - 1:
                scales[:, 0: self.tep_len] = 1 + self.noise_scale * torch.rand_like(scales[:, 0: self.tep_len])
            else:
                scales[:, 0: self.tep_len] = 1 / self.final_scale
        return scales[:, :, None]

    def forward(self, x, **kwargs):
        out = list(self.module(x, **kwargs))
        scales = self.get_scales(out[0])
        out[0] /= scales
        return out


class LLM_for_MMLU:
    TOTAL_B = 10
    def __init__(self, model, tokenizer, use_terminator=True, model_name=''):
        self.model = model
        self.model.eval()
        self.tokenizer = tokenizer
        self.model_name = model_name
        self.use_tm = use_terminator
        self.eot_str = get_eos_str(model_name)
        if use_terminator:
            self.terminators = [
                self.tokenizer.eos_token_id,
                self.tokenizer.convert_tokens_to_ids(self.eot_str)
            ]
        self.prompt_pre = "You are a helpful and concise assistant. You reply only with a 'The correct answer is: ' followed letter from the set {A., B., C., D.}: {REPLY WITH ONLY THE STRING 'The correct answer is: ' FOLLOWED BY THE CORRECT ANSWER's LETTER, LIKE SO: 'The correct answer is: B.'}. DO NOT add any additional words or explain your reasoning!\n"
        self.prompt_ = ''
        self.TEP_LEN = self.tokenizer([self.prompt_], return_tensors="pt").input_ids.shape[1]

    def model_privacy_augment(self):
        print(f"**ENABLE PRIVACY BLOCK: {self.TEP_LEN} context tokens are preserved!")
        for i in range(self.TOTAL_B):
            self.model.model.layers[i] = scaled_module(self.model.model.layers[i], i, self.TOTAL_B, self.TEP_LEN, self.model_name)

    def remove_privacy_blocks(self):
        print("**REMOVE PRIVACY BLOCK**")
        for i in range(self.TOTAL_B):
            self.model.model.layers[i] = self.model.model.layers[i].module

    def enable_nshots_prompt(self, n_shots, example_datasets):
        self.prompt_ = ''
        if n_shots > 0:
            self.prompt_ += "Here are some examples about the interactions between user and assistant:\n\n"
            for i in range(n_shots):
                example = example_datasets[i]
                self.prompt_ += self.question_template(example)
                self.prompt_ += f"assistant: The correct answer is: {example['target']}.\n\n"
        self.TEP_LEN = self.tokenizer([self.prompt_], return_tensors="pt").input_ids.shape[1]
        self.prompt_ += self.prompt_pre

    def generate(self, prompt):
        device = "cuda"
        prompt = self.prompt_ + prompt
        prompt_input = f"{prompt}assistant: The correct answer is"
        model_inputs = self.tokenizer([prompt_input], return_tensors="pt").to(device)
        with torch.no_grad():
            if self.use_tm:
                generated_ids = self.model.generate(**model_inputs, max_new_tokens=4, eos_token_id=self.terminators)
            else:
                generated_ids = self.model.generate(**model_inputs, max_new_tokens=4)
        generated_ids = [
            output_ids[len(input_ids):] for input_ids, output_ids in zip(model_inputs.input_ids, generated_ids)
        ]
        return self.tokenizer.batch_decode(generated_ids, skip_special_tokens=True)[0]

    def question_template(self, example):
        prompt_q = ("user: I have a question: " + example["input"] + "\n\nChoices:\n")
        prompt_q += ("A. " + example["A"] + "\n")
        prompt_q += ("B. " + example["B"] + "\n")
        prompt_q += ("C. " + example["C"] + "\n")
        prompt_q += ("D. " + example["D"] + "\n\n")
        return prompt_q

    def evaluate(self, dataset, sub='unknown'):
        total = len(dataset)
        acc_count = 0
        fails = 0
        pbar = tqdm(range(total))
        for i in pbar:
            data = dataset[i]
            prompt_q = self.question_template(data)
            result = self.generate(prompt_q)
            if (data['target']+'.' in result) or ('**'+data['target'] in result):
                acc_count += 1
            elif all(choice not in result for choice in ["A.", "B.", "C.", "D.", "**A", "**B", "**C", "**D"]):
                fails += 1
            pbar.set_postfix_str(f"acc: {round(acc_count/(i+1),2)}")
        log.info(f"task: {sub} | accuracy: {acc_count/total} | fails: {fails}")
        return acc_count/total, fails, acc_count, total


if __name__ == "__main__":
    phi_path = "../pretrained_models/Phi-3-medium-128k-instruct"
    gemma_path = "../pretrained_models/gemma-2-9b-it"
    llama_path = "../pretrained_models/Meta-Llama-3-8B-Instruct"
    mistral_path = "../pretrained_models/Mistral-7B-Instruct-v0.3"
    llama70_path = "../pretrained_models/Meta-Llama-3-70B-Instruct-AWQ"

    model_path = gemma_path
    enable_privacy = True

    # init logger
    log = logging.getLogger(f'mmlu')
    log.setLevel(level=logging.DEBUG)
    log_name = model_path.split('/')[-1].split('-')[0].lower()
    model_name = model_path.split('/')[-1]
    file_name = f'mmlu_{log_name}.log' if not enable_privacy else f'mmlu_{log_name}_privacy.log'
    handler = logging.FileHandler(f'./mmlu_eva/{file_name}', encoding='utf-8', mode='w')
    handler.setLevel(logging.INFO)
    log.addHandler(handler)

    model = AutoModelForCausalLM.from_pretrained(model_path, device_map='auto', torch_dtype=torch.bfloat16)
    tokenizer = AutoTokenizer.from_pretrained(model_path)
    model.generation_config.pad_token_id = tokenizer.eos_token_id

    llm_mmlu = LLM_for_MMLU(model=model, tokenizer=tokenizer, use_terminator=False, model_name=model_name)

    all_sub = sorted(os.listdir('../dataset_save_disk/mmlu_disk/'))
    acc_count, total = 0, 0
    n_shots = 5
    print(f'start evaluating {model_name}!')
    for i, sub in enumerate(all_sub):
        print(f"--------No. {i+1}/{len(all_sub)} | {sub}--------")
        path = os.path.join('../dataset_save_disk/mmlu_disk/', sub)
        dataset = load_from_disk(path)
        llm_mmlu.enable_nshots_prompt(n_shots=n_shots, example_datasets=dataset["train"])
        if enable_privacy:
            llm_mmlu.model_privacy_augment()
        acc, fails, acc_count_, total_ = llm_mmlu.evaluate(dataset["test"], sub=sub)
        acc_count += acc_count_
        total += total_
        print(f"task: {sub} | accuracy: {acc} | fails: {fails}")
        print(f"total accuracy: {acc_count/total} | acc_count: {acc_count} | total test: {total}")
        if enable_privacy:
            llm_mmlu.remove_privacy_blocks()

    log.info(f"total accuracy: {acc_count/total} | acc_count: {acc_count} | total test: {total}")
    logging.shutdown()
