import os
import json
from pathlib import Path

def compile_jsons(base_paths):
    """
    Compile JSON files from multiple directories into a single dictionary.
    
    Args:
        base_paths (list): List of paths to search for JSON files
        
    Returns:
        dict: Combined results with experiment type as key
    """
    combined_results = {
        "FARE": {},
        "LORE": {},
        "CLIP": {}
    }
    
    for base_path in base_paths:
        parent_dir = Path(base_path).parent.name
        experiment_type = "FARE" if "FARE" in parent_dir else "LORE" if "LORE" in parent_dir else "CLIP"
        
        # Read and parse each JSON file
        for json_file in Path(base_path).glob("*.json"):
            try:
                with open(json_file, 'r') as f:
                    data = json.load(f)
                    
                # Use dataset name as key
                dataset_name = data.get('dataset', 'unknown')
                if dataset_name not in combined_results[experiment_type]:
                    combined_results[experiment_type][dataset_name] = []
                    
                # Add experiment type to the data
                data['experiment_type'] = experiment_type
                combined_results[experiment_type][dataset_name].append(data)
                
            except json.JSONDecodeError:
                print(f"Error reading {json_file}")
                continue
            except Exception as e:
                print(f"Unexpected error with {json_file}: {str(e)}")
                continue
    
    return combined_results

def save_combined_results(results, output_path="results/combined_results.json"):
    """
    Save the combined results to a JSON file.
    
    Args:
        results (dict): Combined results to save
        output_path (str): Path to save the combined JSON file
    """
    with open(output_path, 'w') as f:
        json.dump(results, f, indent=2)

if __name__ == "__main__":
    # Define the paths to search
    paths_to_search = [
        # "results/ViT-B-32/CLIP/clean",
        # "results/ViT-B-32/FARE2/clean",
        # "results/ViT-B-32/FARE2/adv",
        # "results/ViT-B-32/LORE-d0.1-k10-imagenet-5epoch/adv",
        # "results/ViT-B-32/LORE-d0.1-k10-imagenet-5epoch/clean",
        # "results/ViT-B-32/FARE4/clean",
        # "results/ViT-B-32/FARE4/adv",
        # "results/ViT-B-32/LORE4/clean",
        # "results/ViT-B-32/LORE4/adv",
        # "results/ViT-B-32/LORE_Sched"
        "results/ViT-B-32/final"
    ]
    
    # Compile the results
    combined_results = compile_jsons(paths_to_search)
    
    # Save the results
    save_combined_results(combined_results)
    print("Results have been combined and saved to combined_results.json")