from __future__ import annotations

from collections.abc import Callable
from dataclasses import dataclass, field
from typing import Any


@dataclass
class Rule:
    f: Callable[..., tuple[Any, Any]]
    arguments: dict[str, str]
    loss_function: Callable[[Any, Any], Any]


@dataclass
class System:
    rules: dict[str, Rule] = field(default_factory=dict)

    def add_rule(self, name, f, loss_function, /, **arguments) -> None:
        # TODO add introspection
        self.rules[name] = Rule(f, arguments, loss_function)

    def __call__(self, **kwargs: Any) -> Any:
        results = {}
        for rule_name, rule in self.rules.items():
            try:
                arguments = {k: kwargs[v] for k, v in rule.arguments.items()}
            except KeyError as e:
                raise ValueError('Missing input') from e

            prediction, target = rule.f(**arguments)
            loss = rule.loss_function(prediction, target)
            results[rule_name] = {'loss': loss,
                                  'prediction': prediction,
                                  'target': target}
        return results
