#!/bin/bash

# Activate conda environment
source ~/miniconda3/etc/profile.d/conda.sh
conda activate lsd  # replace 'lsd' with your environment name

# ====== CONFIGURATION SETTINGS - EDIT THESE VALUES DIRECTLY ======

# Hardware settings
CUDA_DEVICE="0"                   # GPU device ID to use (e.g., "0", "1", or "0,1" for multiple GPUs)
DEVICE="cuda"                     # Device type (usually "cuda" or "cpu")

# Data and model paths
CLASSIFIER_PATH="/logs/face_classifier_noised/512b_noised/best_model.pt"
CHECKPOINT_PATH="/logs/faces/nolsd_256_trullyjoint/latent_slot_diffusion/checkpoint-200000-last/"

# Model settings
USE_SLOTS=false                   # Set to 'true' to use slot-based models
GUIDANCE_SCALE=1.0                # Classifier guidance strength (higher = follow target more closely)
CONDITIONING_SCALE=1.0            # Slot conditioning strength (for slot models)
WEIGHT_DTYPE="float32"            # Model precision: float32, float16, or bfloat16
TARGET_STEP=250                   # Target diffusion timestep for reconstruction
NUM_STEPS=200                     # Number of diffusion sampling steps

# Dataset and processing settings
IMG_SIZE=256                      # Image size for processing
NUM_IMAGES=256                    # Number of counterfactual examples per attribute
BATCH_SIZE=8                    # Batch size for processing
MASK_ATTRIBUTES=true              # Set to 'true' to only modify specified attributes
SAVE_ALL=true                     # Set to 'true' to save all modifications, not just successful ones
CALCULATE_FID=true                # Set to 'true' to calculate FID between original and counterfactuals

# Output settings
OUTPUT_DIR="counter_joint_noslots_noised_200steps_250target"

# Specific attributes to modify (leave empty for default set)
# Format: comma-separated list with no spaces, e.g., "Smiling,Blond_Hair,Eyeglasses"
ATTRIBUTES=""

# ====== END OF CONFIGURATION SETTINGS ======

# Set CUDA visible devices before running the script
export CUDA_VISIBLE_DEVICES=$CUDA_DEVICE
echo "Setting CUDA_VISIBLE_DEVICES=$CUDA_DEVICE"

# Prepare arguments for the Python script
ARGS="--checkpoint_path $CHECKPOINT_PATH --classifier_path $CLASSIFIER_PATH"
ARGS="$ARGS --num_images $NUM_IMAGES --guidance_scale $GUIDANCE_SCALE --conditioning_scale $CONDITIONING_SCALE"
ARGS="$ARGS --device $DEVICE --output_dir $OUTPUT_DIR --weight_dtype $WEIGHT_DTYPE"
ARGS="$ARGS --target_step $TARGET_STEP --num_steps $NUM_STEPS --batch_size $BATCH_SIZE"
ARGS="$ARGS --img_size $IMG_SIZE"

# Add boolean flags based on configuration
if [ "$USE_SLOTS" = true ]; then
  ARGS="$ARGS --use_slots"
fi

if [ "$MASK_ATTRIBUTES" = true ]; then
  ARGS="$ARGS --mask_attributes"
fi

if [ "$SAVE_ALL" = true ]; then
  ARGS="$ARGS --save_all"
fi

if [ "$CALCULATE_FID" = true ]; then
  ARGS="$ARGS --calculate_fid"
fi

# Add attributes if specified
if [ ! -z "$ATTRIBUTES" ]; then
  # Convert comma-separated list to space-separated for Python argparse
  IFS=',' read -ra ATTR_ARRAY <<< "$ATTRIBUTES"
  for attr in "${ATTR_ARRAY[@]}"; do
    ARGS="$ARGS --attributes $attr"
  done
fi

# Print the configuration
echo "====== Counterfactual Generation Configuration ======"
echo "CUDA Device(s): $CUDA_DEVICE"
echo "Classifier Path: $CLASSIFIER_PATH"
echo "Checkpoint Path: $CHECKPOINT_PATH"
echo "Using Slots: $USE_SLOTS"
echo "Number of Images: $NUM_IMAGES"
echo "Guidance Scale: $GUIDANCE_SCALE"
echo "Conditioning Scale: $CONDITIONING_SCALE"
echo "Target Step: $TARGET_STEP"
echo "Number of Steps: $NUM_STEPS"
echo "Batch Size: $BATCH_SIZE"
echo "Image Size: $IMG_SIZE"
echo "Mask Attributes: $MASK_ATTRIBUTES"
echo "Save All: $SAVE_ALL"
echo "Calculate FID: $CALCULATE_FID"
echo "Device: $DEVICE"
echo "Weight Data Type: $WEIGHT_DTYPE"
echo "Output Directory: $OUTPUT_DIR"
if [ ! -z "$ATTRIBUTES" ]; then
  echo "Attributes: $ATTRIBUTES"
else
  echo "Attributes: Using default list"
fi
echo "=================================================="

# Run the counterfactuals.py script
echo "Running counterfactuals.py..."
python src/counterfactuals.py $ARGS

echo "Counterfactual generation completed!"
echo "Results saved to $OUTPUT_DIR"
