"""Define Transformer for Clustering."""
import math

import torch
import torch.nn as nn
import torch.nn.functional as F

import dgl.geometry as dgl_geo
import segment.utils as seg_utils


class TransformerClustering(nn.Module):

    def __init__(self,
                 num_clusters=4,
                 d_model=512,
                 dropout=0.1,
                 activation="relu",
                 normalize_before=False,
                 detach_src_for_logit=True,
                 l2_normalize_for_fps=True,
                 return_intermediate_dec=False,
                 kmeans_iterations=5):
        """Initializes a Transformer for Clustering.

        Args:
          num_clusters: A scalar indicates the number of centroids.
          d_model: A scalar indicates the input channels to Transformer.
          dropout: A `float` indicates the dropout rate.
          activation: A string indicates the type of non-linear activation.
          normalize_before: A `bool` indicates if applying normalization first.
          return_intermediate_dec: A `bool` indicates if return intermediate
            results from decoders.
        """
        super().__init__()

        self._num_clusters = num_clusters
        self._detach_src_for_logit = detach_src_for_logit
        self._l2_normalize_for_fps = l2_normalize_for_fps
        self._kmeans_iterations = kmeans_iterations

    def _rank_significance(self, attn, mask, num_samples):
        """Rank significance score by attention score.
        Args:
          attn: A `tensor` of shape
            `[batch_size, source_sequence_length, source_sequence_length]`.

        Returns:
          sampled_inds: A `tensor` of shape `[batch_size, num_clusters]`.
        """
        significance = attn.sum(dim=2).sum(dim=1)
        if mask is None:
            significance = significance.masked_fill(mask, -1)
        sampled_vals, sampled_inds = torch.topk(significance, num_samples, dim=1)

        return sampled_inds

    def _fill_with_mean(self, src, mask):
        """A helper function to fill invalid entries with mean values.
        """
        bs, sl, cs = src.shape
        if mask is not None:
            mean_src = valid_mean(src, ~mask).unsqueeze(1).type_as(src)
            # Fill padded entries with mean values.
            fill_mask = mask.unsqueeze(2).expand(-1, -1, cs)
            filled_src = torch.where(fill_mask, mean_src.expand(-1, sl, -1), src)
        else:
            mean_src = torch.mean(src, dim=1, keepdim=True).type_as(src)
            filled_src = src

        return filled_src, mean_src

    def _collect_centroids(self, src, kmean_labels, mask, num_clusters):
        """Helper function to collect centroids given kmeans labels.
        """
        bs, sl, cs = src.shape

        if mask is None:
            mask = torch.zeros((bs, sl), dtype=torch.bool, device=src.device)

        # Mask out invalid datas given mask.
        kmean_labels = kmean_labels.masked_fill(mask, num_clusters)

        centroids = seg_utils.segment_mean_nd(src, kmean_labels, 1, 1)

        centroids = centroids[:, :num_clusters, :]

        return centroids

    def _kmeans(self, src, sampled_inds, mask, iterations, metric='l2'):
        """Get centroid by KMeans.
        """
        bs, sl, cs = src.shape
        num_samples = sampled_inds.shape[1]

        src = src.to(torch.float64)

        # Initialize seeds with sampled inds.
        kmean_labels = torch.zeros((bs, sl), dtype=torch.long, device=src.device)
        kmean_labels += num_samples

        sampled_values = torch.arange(num_samples, dtype=torch.long,
                                      device=kmean_labels.device)
        sampled_values = sampled_values.view(1, -1).expand(bs, -1)
        kmean_labels = kmean_labels.scatter(1, sampled_inds, sampled_values)

        # Collect centroids.
        centroids = self._collect_centroids(src, kmean_labels, mask, num_samples)

        for it in range(iterations):

            # Compute distances between datas and centroids.
            if metric == 'l2':
                # Calculate L2 distance.
                sqr_src = torch.sum(src * src, dim=-1).unsqueeze(2)
                sqr_centroids = torch.sum(centroids * centroids, dim=-1).unsqueeze(1)
                src_centroids = torch.einsum('bij,bjk->bik', src, centroids.transpose(1, 2))
                dists = sqr_src + sqr_centroids - 2 * src_centroids
            else:
                # Calculate cosine-similarity distance.
                normed_src = F.normalize(src, dim=-1)
                normed_centroids = F.normalize(centroids, dim=-1)
                dists = 1 - torch.einsum('bij,bjk->bik', normed_src, normed_centroids.transpose(1, 2))

            # Find nearest neighbors.
            kmean_labels = torch.argmin(dists, dim=-1)

            # Update centroids.
            centroids = self._collect_centroids(src, kmean_labels, mask, num_samples)

        return kmean_labels

    def forward(self, attn, src, mask, pos_embed):
        """Feedforward for clustering with Transformer.

        Args:
          attn: A `tensor` of shape
            `[batch_size, num_heads, source_sequence_length, source_sequence_length]`.
          src: A `tensor` of shape `[batch_size, source_sequence_length, channels]`.
          mask: A bool `tensor` of shape `[batch_size, sequence_length]`.
          pos_embed: A `tensor` of shape
            `[batch_size, source_sequence_length, channels]`.

        Returns:
          centroids: A `tensor` of shape `[batch_size, num_clusters, channels]`.
          logits: A `tensor` of shape
            `[batch_size, source_sequence_length, num_clusters]`.
          sampled_inds: A `tensor` of shape `[batch_size, num_clusters]`.
        """
        bs, sl, cs = src.shape

        # Sample indices based on significance score.
        sampled_inds = self._rank_significance(attn, mask, self._num_clusters)

        # Average pooling within clusters.
        kmean_labels = self._kmeans(src, sampled_inds, mask,
                                    self._kmeans_iterations, 'l2')
        centroids = self._collect_centroids(
            src, kmean_labels, mask, self._num_clusters)

        # Dummy variables.
        logits = seg_utils.one_hot(kmean_labels, self._num_clusters).type_as(src)

        return centroids, logits, sampled_inds


def valid_mean(x, mask):
     """Compute mean of x given valid mask.

     Args:
         x: A `float` tensor of shape `[batch_size, num_nodes, channels]`.
         mask: A `bool` tensor of shape `[batch_size, num_nodes]`, where
             `True` indicates the entry is valid.

     Returns:
         mean_x: A `float` tensor of shape `[batch_size, channels]`.
     """
     mask = mask.type_as(x).unsqueeze(2)
     sum_mask = torch.clamp(torch.sum(mask, dim=1), min=1)
     masked_x = x * mask
     mean_x = torch.sum(masked_x, dim=1) / sum_mask

     return mean_x

