"""Define Hierarchical Visual Transformer.

Modified from:
    https://github.com/facebookresearch/moco-v3/blob/main/vits.py
"""

import math
import torch
import torch.nn as nn
import torch.nn.functional as F
from functools import partial, reduce
from operator import mul

from timm.models.vision_transformer import VisionTransformer, _cfg
from timm.models.layers.helpers import to_2tuple
from timm.models.layers import PatchEmbed

import segment.utils as seg_utils
from segment.tf_clusters import TransformerClustering

__all__ = [
    'vit_conv_small',
    'vit_conv_base',
]


class Pooling(nn.Module):

    def __init__(self, pool_block):
        super(Pooling, self).__init__()
        self.pool_block = pool_block

    def forward(self, x, padding_mask=None, pos_emb=None):
        """Perform Pooling module.
        
        Args:
            x: A `float` tensor of shape `[batch_size, num_nodes, channels]`.
                Note that the first entry denotes the class token and the
                rest of nodes are segment features.
            padding_mask: A `binary` tensor of shape
                `[batch_size, num_nodes]`. `True` indicates the entry
                is padded; otherwise, should be `False`. Note that the
                first entry denotes the class token and the rest of
                nodes are segment features.
            pos_emb: A `float` tensor of shape
                `[batch_size, num_nodes, channels]`. Note that the first entry
                (others) denotes the positional encoding for the class token
                (segments).

        Returns:
            pool_x: A `float` tensor of shape
                `[batch_size, num_pooled_nodes, channels]`.
            pool_pos_emb: A `float` tensor of shape
                `[batch_size, num_pooled_nodes, channels]`.
            pool_logit: A `float` tensor of shape
                `[batch_size, num_nodes, num_pooled_nodes]`.
            centroid: A `float` tensor of shape
                `[batch_size, num_pooled_nodes, channels]`.
        """
        seg_x = x[:, 1:, :]
        seg_pos_emb = pos_emb[:, 1:, :] if pos_emb is not None else None
        seg_mask = padding_mask[:, 1:] if padding_mask is not None else None
        centroid, pool_logit, _ = self.pool_block(
            src=seg_x.permute(0, 2, 1),
            mask=seg_mask,
            pos_embed=seg_pos_emb.permute(0, 2, 1))
        centroid = centroid.permute(0, 2, 1)

        # TODO: Pooling by probability.
        pool_prob = torch.softmax(pool_logit, dim=1)
        pool_logit = pool_logit.permute(0, 2, 1)

        pool_seg_x = torch.einsum('bij,bjk->bik', pool_prob, seg_x)
        pool_x = torch.cat([x[:, [0], :], pool_seg_x], dim=1)

        if pos_emb is not None:
            pool_seg_pos_emb = torch.einsum(
                'bij,bjk->bik', pool_prob, seg_pos_emb)
            pool_pos_emb = torch.cat(
                [pos_emb[:, [0], :], pool_seg_pos_emb], dim=1)
        else:
            pool_pos_emb = None

        return pool_x, pool_pos_emb, pool_logit, centroid


class PositionEmbeddingLearned(nn.Module):
    """
    Absolute pos embedding, learned.
    """
    def __init__(self, num_pos_feats=256, h=28, w=28, ratio=3):
        super().__init__()
        self.row_embed = nn.Embedding(h, num_pos_feats // 2)
        self.col_embed = nn.Embedding(w, num_pos_feats // 2)
        self.reset_parameters(ratio)

    def reset_parameters(self, ratio):
        nn.init.uniform_(self.row_embed.weight, -ratio, ratio)
        nn.init.uniform_(self.col_embed.weight, -ratio, ratio)

    def forward(self, x):
        n, h, w, c = x.shape
        i = torch.arange(w, device=x.device)
        j = torch.arange(h, device=x.device)
        x_emb = self.col_embed(i)
        y_emb = self.row_embed(j)
        pos = torch.cat([
            x_emb.unsqueeze(0).repeat(h, 1, 1),
            y_emb.unsqueeze(1).repeat(1, w, 1),
        ], dim=-1).unsqueeze(0).repeat(x.shape[0], 1, 1, 1)
        return pos


class VisionTransformerMoCo(VisionTransformer):
    def __init__(self, stop_grad_conv1=False, **kwargs):
        assert len(kwargs['depth']) == 5, 'Depth must be a list of five integers'
        depth = kwargs['depth']
        kwargs['depth'] = sum(kwargs['depth'])
        super().__init__(**kwargs)
        # Split intermediate attentional blocks.
        cumsum_depth = [0]
        for d in depth:
            cumsum_depth.append(d + cumsum_depth[-1])
        self.blocks1 = self.blocks[cumsum_depth[0]:cumsum_depth[1]]
        self.blocks2 = self.blocks[cumsum_depth[1]:cumsum_depth[2]]
        self.blocks3 = self.blocks[cumsum_depth[2]:cumsum_depth[3]]
        self.blocks4 = self.blocks[cumsum_depth[3]:cumsum_depth[4]]
        self.blocks5 = self.blocks[cumsum_depth[4]:cumsum_depth[5]]

        self.pool1 = Pooling(
            pool_block=TransformerClustering(
                num_clusters=64,
                nhead=4,
                d_model=kwargs['embed_dim'],
                num_encoder_layers=2,
                num_decoder_layers=2,
                dim_feedforward=kwargs['embed_dim']))
        self.pool_pos_embed1 = PositionEmbeddingLearned(
            kwargs['embed_dim'], 28, 28, 3)
        self.pool2 = Pooling(
            pool_block=TransformerClustering(
                num_clusters=32,
                nhead=4,
                d_model=kwargs['embed_dim'],
                num_encoder_layers=2,
                num_decoder_layers=2,
                dim_feedforward=kwargs['embed_dim']))
        self.pool_pos_embed2 = PositionEmbeddingLearned(
            kwargs['embed_dim'], 28, 28)
        self.pool3 = Pooling(
            pool_block=TransformerClustering(
                num_clusters=16,
                nhead=4,
                d_model=kwargs['embed_dim'],
                num_encoder_layers=2,
                num_decoder_layers=2,
                dim_feedforward=kwargs['embed_dim']))
        self.pool_pos_embed3 = PositionEmbeddingLearned(
            kwargs['embed_dim'], 28, 28)
        self.pool4 = Pooling(
            pool_block=TransformerClustering(
                num_clusters=8,
                nhead=4,
                d_model=kwargs['embed_dim'],
                num_encoder_layers=2,
                num_decoder_layers=2,
                dim_feedforward=kwargs['embed_dim']))
        self.pool_pos_embed4 = PositionEmbeddingLearned(
            kwargs['embed_dim'], 28, 28)
        self.pool5 = Pooling(
            pool_block=TransformerClustering(
                num_clusters=4,
                nhead=4,
                d_model=kwargs['embed_dim'],
                num_encoder_layers=2,
                num_decoder_layers=2,
                dim_feedforward=kwargs['embed_dim']))
        self.pool_pos_embed5 = PositionEmbeddingLearned(
            kwargs['embed_dim'], 28, 28)

        # Use fixed 2D sin-cos position embedding
        self.build_2d_sincos_position_embedding()

        # weight initialization
        for name, m in self.named_modules():
            if isinstance(m, nn.Linear):
                if 'qkv' in name:
                    # treat the weights of Q, K, V separately
                    val = math.sqrt(6. / float(m.weight.shape[0] // 3 + m.weight.shape[1]))
                    nn.init.uniform_(m.weight, -val, val)
                else:
                    nn.init.xavier_uniform_(m.weight)
                nn.init.zeros_(m.bias)
        nn.init.normal_(self.cls_token, std=1e-6)

        if isinstance(self.patch_embed, PatchEmbed):
            # xavier_uniform initialization
            val = math.sqrt(6. / float(3 * reduce(mul, self.patch_embed.patch_size, 1) + self.embed_dim))
            nn.init.uniform_(self.patch_embed.proj.weight, -val, val)
            nn.init.zeros_(self.patch_embed.proj.bias)

            if stop_grad_conv1:
                self.patch_embed.proj.weight.requires_grad = False
                self.patch_embed.proj.bias.requires_grad = False

    def build_2d_sincos_position_embedding(self, temperature=10000.):
        h, w = self.patch_embed.grid_size
        grid_w = torch.arange(w, dtype=torch.float32)
        grid_h = torch.arange(h, dtype=torch.float32)
        grid_w, grid_h = torch.meshgrid(grid_w, grid_h)
        assert self.embed_dim % 4 == 0, 'Embed dimension must be divisible by 4 for 2D sin-cos position embedding'
        pos_dim = self.embed_dim // 4
        omega = torch.arange(pos_dim, dtype=torch.float32) / pos_dim
        omega = 1. / (temperature**omega)
        out_w = torch.einsum('m,d->md', [grid_w.flatten(), omega])
        out_h = torch.einsum('m,d->md', [grid_h.flatten(), omega])
        pos_emb = torch.cat([torch.sin(out_w), torch.cos(out_w), torch.sin(out_h), torch.cos(out_h)], dim=1)[None, :, :]

        assert self.num_tokens == 1, 'Assuming one and only one token, [cls]'
        pe_token = torch.zeros([1, 1, self.embed_dim], dtype=torch.float32)
        #self.pos_embed = nn.Parameter(torch.cat([pe_token, pos_emb], dim=1))
        #self.pos_embed.requires_grad = False
        self.pos_embed = nn.Parameter(pos_emb)
        self.pos_embed.requires_grad = False
        self.pe_token = nn.Parameter(pe_token)
        self.pe_token.requires_grad = False

    def forward_features(self, x, y, return_intermediate=False):
        x = self.patch_embed(x) # NxHxWxC
        N, H, W, C = x.shape

        pool_pos_embed1 = self.pool_pos_embed1(x)
        pool_pos_embed2 = self.pool_pos_embed2(x)
        pool_pos_embed3 = self.pool_pos_embed3(x)
        pool_pos_embed4 = self.pool_pos_embed4(x)
        pool_pos_embed5 = self.pool_pos_embed5(x)

        # Collect features within each segment.
        y = y.unsqueeze(1).float()
        y = F.interpolate(y, x.shape[1:3], mode='nearest')
        y = y.squeeze(1).long()
        x = seg_utils.segment_mean_nd(x, y)

        # Create padding mask.
        ones = torch.ones((N, H, W, 1), dtype=x.dtype, device=x.device)
        avg_ones = seg_utils.segment_mean_nd(ones, y).squeeze(-1)
        avg_ones = F.pad(avg_ones, (1, 0, 0, 0), value=1.0)
        x_padding_mask = avg_ones <= 0.5

        # Collect positional encodings within each segment.
        pos_embed = self.pos_embed.view(1, H, W, C).expand(N, -1, -1, -1)
        pos_embed = seg_utils.segment_mean_nd(pos_embed, y)
        pe_token = self.pe_token.expand(N, -1, -1)
        pos_embed = torch.cat([pe_token, pos_embed], dim=1)

        pool_pos_embed1 = seg_utils.segment_mean_nd(pool_pos_embed1, y)
        pool_pos_embed1 = F.pad(pool_pos_embed1, (0, 0, 1, 0, 0, 0))

        pool_pos_embed2 = seg_utils.segment_mean_nd(pool_pos_embed2, y)
        pool_pos_embed2 = F.pad(pool_pos_embed2, (0, 0, 1, 0, 0, 0))

        pool_pos_embed3 = seg_utils.segment_mean_nd(pool_pos_embed3, y)
        pool_pos_embed3 = F.pad(pool_pos_embed3, (0, 0, 1, 0, 0, 0))

        pool_pos_embed4 = seg_utils.segment_mean_nd(pool_pos_embed4, y)
        pool_pos_embed4 = F.pad(pool_pos_embed4, (0, 0, 1, 0, 0, 0))

        pool_pos_embed5 = seg_utils.segment_mean_nd(pool_pos_embed5, y)
        pool_pos_embed5 = F.pad(pool_pos_embed5, (0, 0, 1, 0, 0, 0))

        # Concatenate segment features with the class token.
        cls_token = self.cls_token.expand(x.shape[0], -1, -1)  # stole cls_tokens impl from Phil Wang, thanks
        if self.dist_token is None:
            x = torch.cat((cls_token, x), dim=1)
        else:
            x = torch.cat((cls_token, self.dist_token.expand(x.shape[0], -1, -1), x), dim=1)

        # TODO: Check the influence of paddings on clusterings.
        x = self.pos_drop(x + pos_embed)
        block1 = self.blocks1(x)
        block2 = self.blocks2(block1)
        block3 = self.blocks3(block2)
        block4 = self.blocks4(block3)
        block5 = self.blocks5(block4)

        pool_block1, pool_pos_embed1, pool_logit1, centroid1 = (
            self.pool1(block1.detach(), x_padding_mask, pool_pos_embed1))
        pool_block2, pool_pos_embed2, pool_logit2, centroid2 = (
            self.pool2(block2.detach(), x_padding_mask, pool_pos_embed2))
        pool_block3, pool_pos_embed3, pool_logit3, centroid3 = (
            self.pool3(block3.detach(), x_padding_mask, pool_pos_embed3))
        pool_block4, pool_pos_embed4, pool_logit4, centroid4 = (
            self.pool4(block4.detach(), x_padding_mask, pool_pos_embed4))
        pool_block5, pool_pos_embed5, pool_logit5, centroid5 = (
            self.pool5(block5.detach(), x_padding_mask, pool_pos_embed5))

        intermediates = {}
        intermediates = {
            'logit1': pool_logit1, 'centroid1': centroid1, 'block1': block1,
            'logit2': pool_logit2, 'centroid2': centroid2, 'block2': block2,
            'logit3': pool_logit3, 'centroid3': centroid3, 'block3': block3,
            'logit4': pool_logit4, 'centroid4': centroid4, 'block4': block4,
            'logit5': pool_logit5, 'centroid5': centroid5, 'block5': block5,
            'padding_mask': x_padding_mask}

        # Predictor.
        out = self.norm(block5)
        if self.dist_token is None:
            if return_intermediate:
                return self.pre_logits(out[:, 0]), intermediates
            else:
                return self.pre_logits(out[:, 0])
        else:
            if return_intermediate:
                return out[:, 0], out[:, 1], intermediates
            else:
                return out[:, 0], out[:, 1]

    def forward(self, x, y, return_intermediate=False):
        x, intermediates = self.forward_features(
            x, y, return_intermediate=True)
        if self.head_dist is not None:
            x, x_dist = self.head(x[0]), self.head_dist(x[1])  # x must be a tuple
            if self.training and not torch.jit.is_scripting():
                # during inference, return the average of both classifier predictions

                if return_intermediate:
                    return x, x_dist, intermediates
                else:
                    return x, x_dist
            else:
                if return_intermediate:
                    return (x + x_dist) / 2, intermediates
                else:
                    return (x + x_dist) / 2
        else:
            x = self.head(x)

        if return_intermediate:
            return x, intermediates
        else:
            return x


class ConvStem(nn.Module):
    """ 
    ConvStem, from Early Convolutions Help Transformers See Better, Tete et al. https://arxiv.org/abs/2106.14881
    """
    def __init__(self, img_size=224, patch_size=8, in_chans=3, embed_dim=768, norm_layer=None, flatten=False):
        super().__init__()

        assert patch_size == 8, 'ConvStem only supports patch size of 8'
        assert embed_dim % 8 == 0, 'Embed dimension must be divisible by 8 for ConvStem'

        img_size = to_2tuple(img_size)
        patch_size = to_2tuple(patch_size)
        self.img_size = img_size
        self.patch_size = patch_size
        self.grid_size = (img_size[0] // patch_size[0], img_size[1] // patch_size[1])
        self.num_patches = self.grid_size[0] * self.grid_size[1]
        self.flatten = flatten

        # build stem, similar to the design in https://arxiv.org/abs/2106.14881
        stem = []
        input_dim, output_dim = 3, embed_dim // 8
        for l in range(4):
            stride = 2 if l < 3 else 1
            stem.append(nn.Conv2d(input_dim, output_dim, kernel_size=3, stride=stride, padding=1, bias=False))
            stem.append(nn.BatchNorm2d(output_dim))
            stem.append(nn.ReLU(inplace=True))
            input_dim = output_dim
            output_dim *= 2
        stem.append(nn.Conv2d(input_dim, embed_dim, kernel_size=1))
        self.proj = nn.Sequential(*stem)

        self.norm = norm_layer(embed_dim) if norm_layer else nn.Identity()

    def forward(self, x):
        B, C, H, W = x.shape
        assert H == self.img_size[0] and W == self.img_size[1], \
            f"Input image size ({H}*{W}) doesn't match model ({self.img_size[0]}*{self.img_size[1]})."
        x = self.proj(x)
        if self.flatten:
            x = x.flatten(2).transpose(1, 2)  # BCHW -> BNC
        else:
            x = x.permute(0, 2, 3, 1) # BxCxHxW -> BxHxWXC
        x = self.norm(x)
        return x


def vit_conv_small(**kwargs):
    # minus one ViT block
    model = VisionTransformerMoCo(
        patch_size=8, embed_dim=384, depth=[2, 2, 2, 3, 2], num_heads=12, mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), embed_layer=ConvStem, **kwargs)
    model.default_cfg = _cfg()
    return model

def vit_conv_base(**kwargs):
    # minus one ViT block
    model = VisionTransformerMoCo(
        patch_size=8, embed_dim=768, depth=[2, 2, 2, 3, 2], num_heads=12, mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), embed_layer=ConvStem, **kwargs)
    model.default_cfg = _cfg()
    return model
