"""Define Hierarchical Visual Transformer.

Modified from:
    https://github.com/facebookresearch/moco-v3/blob/main/vits.py
"""

import math
import torch
import torch.nn as nn
import torch.nn.functional as F
from functools import partial, reduce
from operator import mul

from timm.models.vision_transformer import VisionTransformer, Block, _cfg
from timm.models.layers.helpers import to_2tuple
from timm.models.layers import PatchEmbed

import segment.utils as seg_utils
from segment.tf_clusters_v2 import TransformerClustering
from segment.transformer_v2 import _BatchNorm1d

__all__ = [
    'vit_conv_small',
    'vit_conv_base',
]


class Pooling(nn.Module):

    def __init__(self, pool_block):
        super(Pooling, self).__init__()
        self.pool_block = pool_block

    def forward(self, x, padding_mask=None):
        """Perform Pooling module.
        
        Args:
            x: A `float` tensor of shape `[batch_size, num_nodes, channels]`.
                Note that the first entry denotes the class token and the
                rest of nodes are segment features.
            padding_mask: A `binary` tensor of shape
                `[batch_size, num_nodes]`. `True` indicates the entry
                is padded; otherwise, should be `False`. Note that the
                first entry denotes the class token and the rest of
                nodes are segment features.
            pos_emb: A `float` tensor of shape
                `[batch_size, num_nodes, channels]`. Note that the first entry
                (others) denotes the positional encoding for the class token
                (segments).

        Returns:
            pool_x: A `float` tensor of shape
                `[batch_size, num_pooled_nodes, channels]`.
            pool_pos_emb: A `float` tensor of shape
                `[batch_size, num_pooled_nodes, channels]`.
            pool_logit: A `float` tensor of shape
                `[batch_size, num_nodes, num_pooled_nodes]`.
            centroid: A `float` tensor of shape
                `[batch_size, num_pooled_nodes, channels]`.
        """
        seg_x = x
        seg_mask = padding_mask if padding_mask is not None else None
        centroid, pool_logit, _ = self.pool_block(
            src=seg_x, mask=seg_mask, pos_embed=None)

        pool_padding_mask = torch.zeros(
            (pool_logit.shape[0], pool_logit.shape[-1]),
            dtype=torch.bool,
            device=pool_logit.device)

        return pool_logit, centroid, pool_padding_mask


class VisionTransformerMoCo(VisionTransformer):
    def __init__(self, stop_grad_conv1=False, **kwargs):
        assert len(kwargs['depth']) == 4, 'Depth must be a list of five integers'
        depths = kwargs['depth']
        num_clusters = kwargs.pop('num_clusters', 64)
        kwargs['depth'] = sum(kwargs['depth'])
        super().__init__(**kwargs)

        # Do not tackle dist_token.
        assert self.dist_token is None, 'dist_token is not None.'
        assert self.head_dist is None, 'head_dist is not None.'

        # Split intermediate attentional blocks.
        del self.blocks, self.head, self.norm, self.cls_token # overwrite with new blocks.
        dpr = [x.item() for x in torch.linspace(0, 0, sum(depths))]
        dpr = dpr[::-1]

        in_dim = kwargs['embed_dim']
        out_dim = in_dim

        blocks = []
        block_projs = []
        pools = []
        heads = []
        norms = []
        for ind, depth in enumerate(depths):

            # Build Linear Projection when dimension is changed.
            if in_dim != out_dim:
                block_projs.append(nn.Sequential(_BatchNorm1d(in_dim),
                                                 nn.Linear(in_dim, out_dim)))
            else:
                block_projs.append(nn.Identity())

            # Build Attention Blocks.
            block = []
            for _ in range(depth):
                block.append(Block(dim=out_dim,
                                   num_heads=kwargs['num_heads'],
                                   mlp_ratio=kwargs['mlp_ratio'],
                                   qkv_bias=kwargs['qkv_bias'],
                                   drop_path=dpr.pop(),
                                   norm_layer=kwargs['norm_layer']))
            blocks.append(nn.Sequential(*block))

            # Build Pooling layers.
            pool = Pooling(
                pool_block=TransformerClustering(
                    num_clusters=num_clusters,
                    nhead=kwargs['num_heads'],
                    d_model=out_dim,
                    num_decoder_layers=2,
                    normalize_before=True,
                    detach_src_for_logit=True,
                    l2_normalize_for_fps=False,
                    dim_feedforward=out_dim))
            pools.append(pool)

            # Build output heads.
            heads.append(nn.Linear(out_dim, kwargs['num_classes']))
            norms.append(nn.LayerNorm(out_dim))

            # Update next layer information.
            in_dim = out_dim
            num_clusters = num_clusters // 2

        self.blocks1, self.block_projs1, self.pool1, self.head1, self.norm1 = (
            blocks[0], block_projs[0], pools[0], heads[0], norms[0])
        self.blocks2, self.block_projs2, self.pool2, self.head2, self.norm2 = (
            blocks[1], block_projs[1], pools[1], heads[1], norms[1])
        self.blocks3, self.block_projs3, self.pool3, self.head3, self.norm3 = (
            blocks[2], block_projs[2], pools[2], heads[2], norms[2])
        self.blocks4, self.block_projs4, self.pool4, self.head4, self.norm4 = (
            blocks[3], block_projs[3], pools[3], heads[3], norms[3])
        self.avgpool = nn.AdaptiveAvgPool1d(1)

        # Use fixed 2D sin-cos position embedding
        self.build_2d_sincos_position_embedding()

        # weight initialization
        for name, m in self.named_modules():
            if isinstance(m, nn.Linear):
                if 'qkv' in name:
                    # treat the weights of Q, K, V separately
                    val = math.sqrt(6. / float(m.weight.shape[0] // 3 + m.weight.shape[1]))
                    nn.init.uniform_(m.weight, -val, val)
                else:
                    nn.init.xavier_uniform_(m.weight)
                nn.init.zeros_(m.bias)

        if isinstance(self.patch_embed, PatchEmbed):
            # xavier_uniform initialization
            val = math.sqrt(6. / float(3 * reduce(mul, self.patch_embed.patch_size, 1) + self.embed_dim))
            nn.init.uniform_(self.patch_embed.proj.weight, -val, val)
            nn.init.zeros_(self.patch_embed.proj.bias)

            if stop_grad_conv1:
                self.patch_embed.proj.weight.requires_grad = False
                self.patch_embed.proj.bias.requires_grad = False

    def build_2d_sincos_position_embedding(self, temperature=10000.):
        h, w = self.patch_embed.grid_size
        grid_w = torch.arange(w, dtype=torch.float32)
        grid_h = torch.arange(h, dtype=torch.float32)
        grid_w, grid_h = torch.meshgrid(grid_w, grid_h)
        assert self.embed_dim % 4 == 0, 'Embed dimension must be divisible by 4 for 2D sin-cos position embedding'
        pos_dim = self.embed_dim // 4
        omega = torch.arange(pos_dim, dtype=torch.float32) / pos_dim
        omega = 1. / (temperature**omega)
        out_w = torch.einsum('m,d->md', [grid_w.flatten(), omega])
        out_h = torch.einsum('m,d->md', [grid_h.flatten(), omega])
        pos_emb = torch.cat([torch.sin(out_w), torch.cos(out_w), torch.sin(out_h), torch.cos(out_h)], dim=1)[None, :, :]

        assert self.num_tokens == 1, 'Assuming one and only one token, [cls]'
        pe_token = torch.zeros([1, 1, self.embed_dim], dtype=torch.float32)
        self.pos_embed = nn.Parameter(pos_emb)
        self.pos_embed.requires_grad = False
        self.pe_token = nn.Parameter(pe_token)
        self.pe_token.requires_grad = False

    def forward_features(self, x, y):
        x = self.patch_embed(x) # NxHxWxC
        N, H, W, C = x.shape

        # Collect features within each segment.
        y = y.unsqueeze(1).float()
        y = F.interpolate(y, x.shape[1:3], mode='nearest')
        y = y.squeeze(1).long()
        x = seg_utils.segment_mean_nd(x, y)

        # Create padding mask.
        ones = torch.ones((N, H, W, 1), dtype=x.dtype, device=x.device)
        avg_ones = seg_utils.segment_mean_nd(ones, y).squeeze(-1)
        x_padding_mask = avg_ones <= 0.5

        # Collect positional encodings within each segment.
        pos_embed = self.pos_embed.view(1, H, W, C).expand(N, -1, -1, -1)
        pos_embed = seg_utils.segment_mean_nd(pos_embed, y)

        # TODO: Check the influence of paddings on clusterings.
        x = self.pos_drop(x + pos_embed)

        # Intermediate results.
        intermediates = {}

        # Block1.
        block1 = self.blocks1(self.block_projs1(x)).type_as(x)
        pool_logit1, centroid1, pool_padding_mask1 = self.pool1(block1, x_padding_mask)
        out1 = self.pre_logits(self.avgpool(self.norm1(block1).transpose(1, 2)).squeeze(-1))
        intermediates1 = {
            'logit1': pool_logit1, 'centroid1': centroid1, 'block1': block1,
            'padding_mask1': x_padding_mask, 'out1': out1,
        }
        intermediates.update(intermediates1)

        # Block2.
        block2 = self.blocks2(self.block_projs2(centroid1)).type_as(x)
        pool_logit2, centroid2, pool_padding_mask2 = self.pool2(block2, pool_padding_mask1)
        out2 = self.pre_logits(self.avgpool(self.norm2(block2).transpose(1, 2)).squeeze(-1))
        intermediates2 = {
            'logit2': pool_logit2, 'centroid2': centroid2, 'block2': block2,
            'padding_mask2': pool_padding_mask1, 'out2': out2,
        }
        intermediates.update(intermediates2)

        # Block3.
        block3 = self.blocks3(self.block_projs3(centroid2)).type_as(x)
        pool_logit3, centroid3, pool_padding_mask3 = self.pool3(block3, pool_padding_mask2)
        out3 = self.pre_logits(self.avgpool(self.norm3(block3).transpose(1, 2)).squeeze(-1))
        intermediates3 = {
            'logit3': pool_logit3, 'centroid3': centroid3, 'block3': block3,
            'padding_mask3': pool_padding_mask2, 'out3': out3,
        }
        intermediates.update(intermediates3)

        # Block4.
        block4 = self.blocks4(self.block_projs4(centroid3)).type_as(x)
        pool_logit4, centroid4, pool_padding_mask4 = self.pool4(block4, pool_padding_mask3)
        out4 = self.pre_logits(self.avgpool(self.norm4(block4).transpose(1, 2)).squeeze(-1))
        intermediates4 = {
            'logit4': pool_logit4, 'centroid4': centroid4, 'block4': block4,
            'padding_mask4': pool_padding_mask3, 'out4': out4,
        }
        intermediates.update(intermediates4)

        return intermediates

    def forward(self, x, y, return_intermediate=True):
        intermediates = self.forward_features(x, y)
        intermediates['out4'] = self.head4(intermediates['out4'])
        x = intermediates['out4']

        if return_intermediate:
            intermediates['out1'] = self.head1(intermediates['out1'])
            intermediates['out2'] = self.head2(intermediates['out2'])
            intermediates['out3'] = self.head3(intermediates['out3'])

            return x, intermediates
        else:
            return x


class ConvStem(nn.Module):
    """ 
    ConvStem, from Early Convolutions Help Transformers See Better, Tete et al. https://arxiv.org/abs/2106.14881
    """
    def __init__(self, img_size=224, patch_size=8, in_chans=3, embed_dim=768, norm_layer=None, flatten=False):
        super().__init__()

        assert patch_size == 8, 'ConvStem only supports patch size of 8'
        assert embed_dim % 2 == 0, 'Embed dimension must be divisible by 2 for ConvStem'

        img_size = to_2tuple(img_size)
        patch_size = to_2tuple(patch_size)
        self.img_size = img_size
        self.patch_size = patch_size
        self.grid_size = (img_size[0] // patch_size[0], img_size[1] // patch_size[1])
        self.num_patches = self.grid_size[0] * self.grid_size[1]
        self.flatten = flatten

        # build stem, similar to the design in https://arxiv.org/abs/2106.14881
        stem = []

        output_dim = embed_dim // 2
        stem.append(nn.Conv2d(3, output_dim, kernel_size=7, stride=2, padding=3, bias=False))
        stem.append(nn.BatchNorm2d(output_dim))
        stem.append(nn.ReLU(inplace=True))
        stem.append(nn.MaxPool2d(kernel_size=3, stride=2, padding=1))

        for l in range(3):
            stem.append(nn.Conv2d(output_dim, output_dim, kernel_size=3, stride=1, padding=1, bias=False))
            stem.append(nn.BatchNorm2d(output_dim))
            stem.append(nn.ReLU(inplace=True))
        stem.append(nn.MaxPool2d(kernel_size=3, stride=2, padding=1))
        stem.append(nn.Conv2d(output_dim, embed_dim, kernel_size=1))
        #input_dim, output_dim = 3, embed_dim // 8
        #for l in range(4):
        #    stride = 2 if l < 3 else 1
        #    stem.append(nn.Conv2d(input_dim, output_dim, kernel_size=3, stride=stride, padding=1, bias=False))
        #    stem.append(nn.BatchNorm2d(output_dim))
        #    stem.append(nn.ReLU(inplace=True))
        #    input_dim = output_dim
        #    output_dim *= 2
        #stem.append(nn.Conv2d(input_dim, embed_dim, kernel_size=1))
        self.proj = nn.Sequential(*stem)

        self.norm = norm_layer(embed_dim) if norm_layer else nn.Identity()

    def forward(self, x):
        B, C, H, W = x.shape
        assert H == self.img_size[0] and W == self.img_size[1], \
            f"Input image size ({H}*{W}) doesn't match model ({self.img_size[0]}*{self.img_size[1]})."
        x = self.proj(x)
        if self.flatten:
            x = x.flatten(2).transpose(1, 2)  # BCHW -> BNC
        else:
            x = x.permute(0, 2, 3, 1) # BxCxHxW -> BxHxWXC
        x = self.norm(x)
        return x


def vit_conv_small(**kwargs):
    # minus one ViT block
    model = VisionTransformerMoCo(
        patch_size=8, embed_dim=384, num_clusters=64,
        depth=[3, 3, 3, 2], num_heads=12, mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), embed_layer=ConvStem, **kwargs)
    model.default_cfg = _cfg()
    return model

def vit_conv_base(**kwargs):
    # minus one ViT block
    model = VisionTransformerMoCo(
        patch_size=8, embed_dim=240, num_clusters=64,
        depth=[3, 3, 3, 2], num_heads=12, mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), embed_layer=ConvStem, **kwargs)
    model.default_cfg = _cfg()
    return model
