"""Define Hierarchical Visual Transformer.

Modified from:
    https://github.com/facebookresearch/moco-v3/blob/main/vits.py
"""

import math
import torch
import torch.nn as nn
import torch.nn.functional as F
from functools import partial, reduce
from operator import mul

from timm.models.vision_transformer import VisionTransformer, _cfg
from timm.models.layers.helpers import to_2tuple
from timm.models.layers import PatchEmbed, DropPath, Mlp

import segment.utils as seg_utils
from segment.tf_clusters_v3_5_1 import TransformerClustering

__all__ = [
    'vit_conv_small',
    'vit_conv_base',
    'vit_conv_small_p512',
]


class Pooling(nn.Module):

    def __init__(self, pool_block):
        super(Pooling, self).__init__()
        self.pool_block = pool_block

    def forward(self, x, padding_mask=None):
        """Perform Pooling module.
        
        Args:
            x: A `float` tensor of shape `[batch_size, num_nodes, channels]`.
                Note that the first entry denotes the class token and the
                rest of nodes are segment features.
            padding_mask: A `binary` tensor of shape
                `[batch_size, num_nodes]`. `True` indicates the entry
                is padded; otherwise, should be `False`. Note that the
                first entry denotes the class token and the rest of
                nodes are segment features.
            pos_emb: A `float` tensor of shape
                `[batch_size, num_nodes, channels]`. Note that the first entry
                (others) denotes the positional encoding for the class token
                (segments).

        Returns:
            pool_x: A `float` tensor of shape
                `[batch_size, num_pooled_nodes, channels]`.
            pool_pos_emb: A `float` tensor of shape
                `[batch_size, num_pooled_nodes, channels]`.
            pool_logit: A `float` tensor of shape
                `[batch_size, num_nodes, num_pooled_nodes]`.
            centroid: A `float` tensor of shape
                `[batch_size, num_pooled_nodes, channels]`.
        """
        seg_x = x
        seg_mask = padding_mask if padding_mask is not None else None
        centroid, pool_logit, sampled_x_inds = self.pool_block(
            src=seg_x, mask=seg_mask, pos_embed=None)

        pool_padding_mask = torch.zeros(
            (pool_logit.shape[0], pool_logit.shape[-1]),
            dtype=torch.bool,
            device=pool_logit.device)

        return pool_logit, centroid, pool_padding_mask, sampled_x_inds


class Attention(nn.Module):
    def __init__(self, dim, num_heads=8, qkv_bias=False, qk_scale=None, attn_drop=0., proj_drop=0.):
        super().__init__()
        self.num_heads = num_heads
        head_dim = dim // num_heads
        self.scale = qk_scale or head_dim ** -0.5

        self.qkv = nn.Linear(dim, dim * 3, bias=qkv_bias)
        self.attn_drop = nn.Dropout(attn_drop)
        self.proj = nn.Linear(dim, dim)
        self.proj_drop = nn.Dropout(proj_drop)

    def forward(self, x):
        B, N, C = x.shape
        qkv = self.qkv(x).reshape(B, N, 3, self.num_heads, C // self.num_heads).permute(2, 0, 3, 1, 4)
        q, k, v = qkv[0], qkv[1], qkv[2]   # make torchscript happy (cannot use tensor as tuple)

        attn = (q @ k.transpose(-2, -1)) * self.scale
        logit = attn
        attn = attn.softmax(dim=-1)
        attn = self.attn_drop(attn)

        x = (attn @ v).transpose(1, 2).reshape(B, N, C)
        x = self.proj(x)
        x = self.proj_drop(x)
        return x, logit


class Block(nn.Module):

    def __init__(self, dim, num_heads, mlp_ratio=4., qkv_bias=False, qk_scale=None, drop=0., attn_drop=0.,
                 drop_path=0., act_layer=nn.GELU, norm_layer=nn.LayerNorm):
        super().__init__()
        self.norm1 = norm_layer(dim)
        self.attn = Attention(
            dim, num_heads=num_heads, qkv_bias=qkv_bias, qk_scale=qk_scale, attn_drop=attn_drop, proj_drop=drop)
        # NOTE: drop path for stochastic depth, we shall see if this is better than dropout here
        self.drop_path = DropPath(drop_path) if drop_path > 0. else nn.Identity()
        self.norm2 = norm_layer(dim)
        mlp_hidden_dim = int(dim * mlp_ratio)
        self.mlp = Mlp(in_features=dim, hidden_features=mlp_hidden_dim, act_layer=act_layer, drop=drop)

    def forward(self, inputs):
        if isinstance(inputs, list) or isinstance(inputs, tuple):
            x = inputs[0]
        else:
            x = inputs
        #x = x + self.drop_path(self.attn(self.norm1(x)))
        attn_x, attn_logit = self.attn(self.norm1(x))
        x = x + self.drop_path(attn_x)
        x = x + self.drop_path(self.mlp(self.norm2(x)))
        return x, attn_logit


class VisionTransformerMoCo(VisionTransformer):
    def __init__(self, stop_grad_conv1=False, **kwargs):
        depths = kwargs['depth']
        # These entries do not exist in timm.VisionTransformer.
        num_clusters = kwargs.pop('num_clusters', [64, 32, 16, 8])
        kwargs['depth'] = sum(kwargs['depth'])
        super().__init__(**kwargs)

        # Do not tackle dist_token.
        assert self.dist_token is None, 'dist_token is not None.'
        assert self.head_dist is None, 'head_dist is not None.'

        del self.blocks
        self.blocks = nn.Sequential(*[
            Block(dim=kwargs['embed_dim'],
                  num_heads=kwargs['num_heads'],
                  mlp_ratio=kwargs['mlp_ratio'],
                  qkv_bias=kwargs['qkv_bias'],
                  norm_layer=kwargs['norm_layer'])
            for i in range(kwargs['depth'])])


        # --------------------------------------------------------------------------
        # Encoder specifics
        cumsum_depth = [0]
        for d in depths:
            cumsum_depth.append(d + cumsum_depth[-1])

        blocks = []
        pools = []
        for ind, depth in enumerate(depths):

            # Build Attention Blocks.
            blocks.append(self.blocks[cumsum_depth[ind]:cumsum_depth[ind+1]])

            # Build Pooling layers.
            pool = Pooling(
                pool_block=TransformerClustering(
                    num_clusters=num_clusters[ind],
                    d_model=kwargs['embed_dim'],
                    normalize_before=True,
                    detach_src_for_logit=True,
                    l2_normalize_for_fps=False))
            #if ind == len(depths) - 1:
            #    for param in pool.pool_block.fc.parameters():
            #        param.requires_grad = False
            pools.append(pool)

        self.blocks1, self.pool1 = blocks[0], pools[0]
        self.blocks2, self.pool2 = blocks[1], pools[1]
        self.blocks3, self.pool3 = blocks[2], pools[2]
        self.blocks4, self.pool4 = blocks[3], pools[3]
        # --------------------------------------------------------------------------

        # Use fixed 2D sin-cos position embedding
        self.build_2d_sincos_position_embedding()

        # weight initialization
        for name, m in self.named_modules():
            if isinstance(m, nn.Linear):
                if 'qkv' in name:
                    # treat the weights of Q, K, V separately
                    val = math.sqrt(6. / float(m.weight.shape[0] // 3 + m.weight.shape[1]))
                    nn.init.uniform_(m.weight, -val, val)
                else:
                    nn.init.xavier_uniform_(m.weight)
                nn.init.zeros_(m.bias)
        nn.init.normal_(self.cls_token, std=1e-6)

        if isinstance(self.patch_embed, PatchEmbed):
            # xavier_uniform initialization
            val = math.sqrt(6. / float(3 * reduce(mul, self.patch_embed.patch_size, 1) + self.embed_dim))
            nn.init.uniform_(self.patch_embed.proj.weight, -val, val)
            nn.init.zeros_(self.patch_embed.proj.bias)

            if stop_grad_conv1:
                self.patch_embed.proj.weight.requires_grad = False
                self.patch_embed.proj.bias.requires_grad = False

    def build_2d_sincos_position_embedding(self, temperature=10000.):
        h, w = self.patch_embed.grid_size
        grid_w = torch.arange(w, dtype=torch.float32)
        grid_h = torch.arange(h, dtype=torch.float32)
        grid_w, grid_h = torch.meshgrid(grid_w, grid_h)
        assert self.embed_dim % 4 == 0, 'Embed dimension must be divisible by 4 for 2D sin-cos position embedding'
        pos_dim = self.embed_dim // 4
        omega = torch.arange(pos_dim, dtype=torch.float32) / pos_dim
        omega = 1. / (temperature**omega)
        out_w = torch.einsum('m,d->md', [grid_w.flatten(), omega])
        out_h = torch.einsum('m,d->md', [grid_h.flatten(), omega])
        pos_emb = torch.cat([torch.sin(out_w), torch.cos(out_w), torch.sin(out_h), torch.cos(out_h)], dim=1)[None, :, :]

        assert self.num_tokens == 1, 'Assuming one and only one token, [cls]'
        pe_token = torch.zeros([1, 1, self.embed_dim], dtype=torch.float32)
        self.pos_embed = nn.Parameter(pos_emb)
        self.pos_embed.requires_grad = False
        self.pe_token = nn.Parameter(pe_token)
        self.pe_token.requires_grad = False

    def _block_operations(self, x, cls_token, x_pad_mask, nn_block, pool_block, norm_block):
        """Wrapper to define operations per block.
        """
        # Forward nn block with cls_token and x.
        cls_x = torch.cat([cls_token, x], dim=1)
        #cls_x = nn_block(cls_x).type_as(x)
        cls_x, cls_x_attn_logit = nn_block(cls_x)
        cls_x = cls_x.type_as(x)
        cls_token, x = cls_x[:, :1, :], cls_x[:, 1:, :]

        cls_x_attn_logit = cls_x_attn_logit.type_as(x)

        # Perform pooling only on x.
        pool_logit, centroid, pool_pad_mask, pool_inds = pool_block(x, x_pad_mask)

        # Generate output by cls_token.
        if norm_block is not None:
            out = norm_block(cls_x)[:, 0]
        else:
            out = cls_x[:, 0]

        return x, cls_token, pool_logit, centroid, pool_pad_mask, pool_inds, out, cls_x_attn_logit


    def forward_features(self, x, y):
        x = self.patch_embed(x) # NxHxWxC
        N, H, W, C = x.shape
        yH, yW = y.shape[-2:]

        # Collect features within each segment.
        #y = y.unsqueeze(1).float()
        #y = F.interpolate(y, x.shape[1:3], mode='nearest')
        #y = y.squeeze(1).long()
        x = F.interpolate(x.permute(0, 3, 1, 2), size=(yH, yW), mode='bilinear')
        x = x.permute(0, 2, 3, 1)
        x = seg_utils.segment_mean_nd(x, y)

        # Create padding mask.
        ones = torch.ones((N, yH, yW, 1), dtype=x.dtype, device=x.device)
        avg_ones = seg_utils.segment_mean_nd(ones, y).squeeze(-1)
        x_padding_mask = avg_ones <= 0.5

        # Collect positional encodings within each segment.
        pos_embed = self.pos_embed.view(1, H, W, C).expand(N, -1, -1, -1)
        pos_embed = F.interpolate(pos_embed.permute(0, 3, 1, 2), size=(yH, yW), mode='bicubic')
        pos_embed = pos_embed.permute(0, 2, 3, 1)
        pos_embed = seg_utils.segment_mean_nd(pos_embed, y)

        # TODO: Check the influence of paddings on clusterings.
        x = self.pos_drop(x + pos_embed)

        # Intermediate results.
        intermediates = {}

        # Add class token.
        cls_token = self.cls_token.expand(x.shape[0], -1, -1)
        cls_token = cls_token + self.pe_token.expand(x.shape[0], -1, -1)

        # Block1.
        (block1, cls_token1, pool_logit1, centroid1,
         pool_padding_mask1, pool_inds1, out1, cls_block_attn1) = self._block_operations(
            x, cls_token, x_padding_mask,
            self.blocks1, self.pool1, None)

        intermediates1 = {
            'logit1': pool_logit1, 'centroid1': centroid1, 'block1': block1,
            'padding_mask1': x_padding_mask, 'sampled_inds1': pool_inds1,
            'cls_block_attn1': cls_block_attn1,
        }
        intermediates.update(intermediates1)

        # Block2.
        (block2, cls_token2, pool_logit2, centroid2,
         pool_padding_mask2, pool_inds2, out2, cls_block_attn2) = self._block_operations(
            centroid1, cls_token1, pool_padding_mask1,
            self.blocks2, self.pool2, None)

        intermediates2 = {
            'logit2': pool_logit2, 'centroid2': centroid2, 'block2': block2,
            'padding_mask2': pool_padding_mask1, 'sampled_inds2': pool_inds2,
            'cls_block_attn2': cls_block_attn2,
        }
        intermediates.update(intermediates2)

        # Block3.
        (block3, cls_token3, pool_logit3, centroid3,
         pool_padding_mask3, pool_inds3, out3, cls_block_attn3) = self._block_operations(
            centroid2, cls_token2, pool_padding_mask2,
            self.blocks3, self.pool3, None)

        intermediates3 = {
            'logit3': pool_logit3, 'centroid3': centroid3, 'block3': block3,
            'padding_mask3': pool_padding_mask2, 'sampled_inds3': pool_inds3,
            'cls_block_attn3': cls_block_attn3,
        }
        intermediates.update(intermediates3)

        # Block4.
        (block4, cls_token4, pool_logit4, centroid4,
         pool_padding_mask4, pool_inds4, out4, cls_block_attn4) = self._block_operations(
            centroid3, cls_token3, pool_padding_mask3,
            self.blocks4, self.pool4, self.norm)
        out4 = self.pre_logits(out4)

        intermediates4 = {
            'logit4': pool_logit4, 'centroid4': centroid4, 'block4': block4,
            'padding_mask4': pool_padding_mask3, 'out4': out4, 'sampled_inds4': pool_inds4,
            'cls_block_attn4': cls_block_attn4,
        }
        intermediates.update(intermediates4)

        return intermediates

    def forward(self, x, y, return_intermediate=True):
        intermediates = self.forward_features(x, y)
        x = self.head(intermediates['out4'])

        if return_intermediate:
            return x, intermediates
        else:
            return x


class ConvStem(nn.Module):
    """ 
    ConvStem, from Early Convolutions Help Transformers See Better, Tete et al. https://arxiv.org/abs/2106.14881
    """
    def __init__(self, img_size=224, patch_size=8, in_chans=3, embed_dim=768, norm_layer=None, flatten=False):
        super().__init__()

        assert patch_size == 8, 'ConvStem only supports patch size of 8'
        assert embed_dim % 8 == 0, 'Embed dimension must be divisible by 2 for ConvStem'

        img_size = to_2tuple(img_size)
        patch_size = to_2tuple(patch_size)
        self.img_size = img_size
        self.patch_size = patch_size
        self.grid_size = (img_size[0] // patch_size[0], img_size[1] // patch_size[1])
        self.num_patches = self.grid_size[0] * self.grid_size[1]
        self.flatten = flatten

        # build stem, similar to the design in https://arxiv.org/abs/2106.14881
        stem = []
        input_dim, output_dim = 3, embed_dim // 8
        for l in range(4):
            stride = 2 if l < 3 else 1
            stem.append(nn.Conv2d(input_dim, output_dim, kernel_size=3, stride=stride, padding=1, bias=False))
            stem.append(nn.BatchNorm2d(output_dim))
            stem.append(nn.ReLU(inplace=True))
            input_dim = output_dim
            output_dim *= 2
        stem.append(nn.Conv2d(input_dim, embed_dim, kernel_size=1))
        self.proj = nn.Sequential(*stem)

        self.norm = norm_layer(embed_dim) if norm_layer else nn.Identity()

    def forward(self, x):
        B, C, H, W = x.shape
        assert H == self.img_size[0] and W == self.img_size[1], \
            f"Input image size ({H}*{W}) doesn't match model ({self.img_size[0]}*{self.img_size[1]})."
        x = self.proj(x)
        if self.flatten:
            x = x.flatten(2).transpose(1, 2)  # BCHW -> BNC
        else:
            x = x.permute(0, 2, 3, 1) # BxCxHxW -> BxHxWXC
        x = self.norm(x)
        return x


def vit_conv_small(**kwargs):
    # minus one ViT block
    model = VisionTransformerMoCo(
        patch_size=8, embed_dim=384, num_clusters=[64, 32, 16, 8],
        depth=[3, 3, 3, 2], num_heads=12, mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), embed_layer=ConvStem, **kwargs)
    model.default_cfg = _cfg()
    return model

def vit_conv_base(**kwargs):
    # minus one ViT block
    model = VisionTransformerMoCo(
        patch_size=8, embed_dim=768, num_clusters=[64, 32, 16, 8],
        depth=[3, 3, 3, 2], num_heads=12, mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), embed_layer=ConvStem, **kwargs)
    model.default_cfg = _cfg()
    return model

def vit_conv_small_p512(**kwargs):
  # minus one ViT block
  model = VisionTransformerMoCo(
      img_size=512, patch_size=8, embed_dim=384, num_clusters=[128, 64, 32, 16],
      depth=[3, 3, 3, 2], num_heads=12, mlp_ratio=4, qkv_bias=True,
      norm_layer=partial(nn.LayerNorm, eps=1e-6), embed_layer=ConvStem, **kwargs)
  model.default_cfg = _cfg()
  return model
