"""Define Hierarchical Visual Transformer.

Modified from:
    https://github.com/facebookresearch/moco-v3/blob/main/vits.py
"""

import math
import torch
import torch.nn as nn
import torch.nn.functional as F
from functools import partial, reduce
from operator import mul

from timm.models.vision_transformer import VisionTransformer, Block, _cfg
from timm.models.layers.helpers import to_2tuple
from timm.models.layers import PatchEmbed

import segment.utils as seg_utils
from segment.tf_clusters_v4 import TransformerClustering

__all__ = [
    'vit_conv_small',
    'vit_conv_base',
]


class Pooling(nn.Module):

    def __init__(self, pool_block):
        super(Pooling, self).__init__()
        self.pool_block = pool_block

    def forward(self, cls_x, padding_mask=None):
        """Perform Pooling module.
        
        Args:
            cls_x: A `float` tensor of shape `[batch_size, num_nodes, channels]`.
                Note that the first entry denotes the class token and the
                rest of nodes are segment features.
            padding_mask: A `binary` tensor of shape
                `[batch_size, num_nodes]`. `True` indicates the entry
                is padded; otherwise, should be `False`. Note that the
                first entry denotes the class token and the rest of
                nodes are segment features.
            pos_emb: A `float` tensor of shape
                `[batch_size, num_nodes, channels]`. Note that the first entry
                (others) denotes the positional encoding for the class token
                (segments).

        Returns:
            pool_x: A `float` tensor of shape
                `[batch_size, num_pooled_nodes, channels]`.
            pool_pos_emb: A `float` tensor of shape
                `[batch_size, num_pooled_nodes, channels]`.
            pool_logit: A `float` tensor of shape
                `[batch_size, num_nodes, num_pooled_nodes]`.
            centroid: A `float` tensor of shape
                `[batch_size, num_pooled_nodes, channels]`.
        """
        cls_centroid, pool_logit, sampled_x_inds = self.pool_block(
            cls_src=cls_x, mask=padding_mask, pos_embed=None)

        pool_padding_mask = torch.zeros(
            (pool_logit.shape[0], pool_logit.shape[-1]),
            dtype=torch.bool,
            device=pool_logit.device)

        return pool_logit, cls_centroid, pool_padding_mask, sampled_x_inds


class VisionTransformerMoCo(VisionTransformer):
    def __init__(self, stop_grad_conv1=False, **kwargs):
        depths = kwargs['depth']
        # These entries do not exist in timm.VisionTransformer.
        num_clusters = kwargs.pop('num_clusters', [64, 32, 16, 8])
        kwargs['depth'] = sum(kwargs['depth'])
        super().__init__(**kwargs)

        # Do not tackle dist_token.
        assert self.dist_token is None, 'dist_token is not None.'
        assert self.head_dist is None, 'head_dist is not None.'

        # --------------------------------------------------------------------------
        # Encoder specifics
        del self.blocks, self.norm # overwrite with new blocks.
        dpr = [x.item() for x in torch.linspace(0, 0, sum(depths))]
        dpr = dpr[::-1]

        blocks = []
        pools = []
        for ind, depth in enumerate(depths):

            # Build Attention Blocks.
            block = []
            for _ in range(depth):
                block.append(Block(dim=kwargs['embed_dim'],
                                   num_heads=kwargs['num_heads'],
                                   mlp_ratio=kwargs['mlp_ratio'],
                                   qkv_bias=kwargs['qkv_bias'],
                                   drop_path=dpr.pop(),
                                   norm_layer=kwargs['norm_layer']))
            blocks.append(nn.Sequential(*block))

            # Build Pooling layers.
            pool = Pooling(
                pool_block=TransformerClustering(
                    num_clusters=num_clusters[ind],
                    d_model=kwargs['embed_dim'],
                    normalize_before=True,
                    detach_src_for_logit=True,
                    l2_normalize_for_fps=False,
                    num_heads=kwargs['num_heads'],
                    qkv_bias=kwargs['qkv_bias'],
                    norm_layer=kwargs['norm_layer']))
            pools.append(pool)

        self.blocks1, self.pool1 = blocks[0], pools[0]
        self.blocks2, self.pool2 = blocks[1], pools[1]
        self.blocks3, self.pool3 = blocks[2], pools[2]
        self.blocks4, self.pool4 = blocks[3], pools[3]
        # --------------------------------------------------------------------------

        # Use fixed 2D sin-cos position embedding
        self.build_2d_sincos_position_embedding()

        # weight initialization
        for name, m in self.named_modules():
            if isinstance(m, nn.Linear):
                if 'qkv' in name:
                    # treat the weights of Q, K, V separately
                    val = math.sqrt(6. / float(m.weight.shape[0] // 3 + m.weight.shape[1]))
                    nn.init.uniform_(m.weight, -val, val)
                else:
                    nn.init.xavier_uniform_(m.weight)
                nn.init.zeros_(m.bias)
        nn.init.normal_(self.cls_token, std=1e-6)

        if isinstance(self.patch_embed, PatchEmbed):
            # xavier_uniform initialization
            val = math.sqrt(6. / float(3 * reduce(mul, self.patch_embed.patch_size, 1) + self.embed_dim))
            nn.init.uniform_(self.patch_embed.proj.weight, -val, val)
            nn.init.zeros_(self.patch_embed.proj.bias)

            if stop_grad_conv1:
                self.patch_embed.proj.weight.requires_grad = False
                self.patch_embed.proj.bias.requires_grad = False

    def build_2d_sincos_position_embedding(self, temperature=10000.):
        h, w = self.patch_embed.grid_size
        grid_w = torch.arange(w, dtype=torch.float32)
        grid_h = torch.arange(h, dtype=torch.float32)
        grid_w, grid_h = torch.meshgrid(grid_w, grid_h)
        assert self.embed_dim % 4 == 0, 'Embed dimension must be divisible by 4 for 2D sin-cos position embedding'
        pos_dim = self.embed_dim // 4
        omega = torch.arange(pos_dim, dtype=torch.float32) / pos_dim
        omega = 1. / (temperature**omega)
        out_w = torch.einsum('m,d->md', [grid_w.flatten(), omega])
        out_h = torch.einsum('m,d->md', [grid_h.flatten(), omega])
        pos_emb = torch.cat([torch.sin(out_w), torch.cos(out_w), torch.sin(out_h), torch.cos(out_h)], dim=1)[None, :, :]

        assert self.num_tokens == 1, 'Assuming one and only one token, [cls]'
        pe_token = torch.zeros([1, 1, self.embed_dim], dtype=torch.float32)
        self.pos_embed = nn.Parameter(pos_emb)
        self.pos_embed.requires_grad = False
        self.pe_token = nn.Parameter(pe_token)
        self.pe_token.requires_grad = False

    def _block_operations(self, cls_x, x_pad_mask, nn_block, pool_block):
        """Wrapper to define operations per block.
        """
        cls_x = nn_block(cls_x)

        # Perform pooling only on x.
        pool_logit, cls_centroid, pool_pad_mask, pool_inds = pool_block(cls_x, x_pad_mask)

        # Generate output by pooling.
        out = cls_x[:, 1:, :].mean(dim=1)

        return cls_x, pool_logit, cls_centroid, pool_pad_mask, pool_inds, out


    def forward_features(self, x, y):
        x = self.patch_embed(x) # NxHxWxC
        N, H, W, C = x.shape

        # Collect features within each segment.
        y = y.unsqueeze(1).float()
        y = F.interpolate(y, x.shape[1:3], mode='nearest')
        y = y.squeeze(1).long()
        x = seg_utils.segment_mean_nd(x, y)

        # Create padding mask.
        ones = torch.ones((N, H, W, 1), dtype=x.dtype, device=x.device)
        avg_ones = seg_utils.segment_mean_nd(ones, y).squeeze(-1)
        x_padding_mask = avg_ones <= 0.5

        # Collect positional encodings within each segment.
        pos_embed = self.pos_embed.view(1, H, W, C).expand(N, -1, -1, -1)
        pos_embed = seg_utils.segment_mean_nd(pos_embed, y)

        # TODO: Check the influence of paddings on clusterings.
        x = self.pos_drop(x + pos_embed)

        # Intermediate results.
        intermediates = {}

        # Add class token.
        cls_token = self.cls_token.expand(x.shape[0], -1, -1)
        cls_token = cls_token + self.pe_token.expand(x.shape[0], -1, -1)

        cls_x = torch.cat([cls_token, x], dim=1)

        # Block1.
        (cls_block1, pool_logit1, cls_centroid1,
         pool_padding_mask1, pool_inds1, out1) = self._block_operations(
            cls_x, x_padding_mask, self.blocks1, self.pool1)

        intermediates1 = {
            'logit1': pool_logit1, 'centroid1': cls_centroid1[:, 1:], 'block1': cls_block1[:, 1:],
            'padding_mask1': x_padding_mask, 'sampled_inds1': pool_inds1,
        }
        intermediates.update(intermediates1)

        # Block2.
        (cls_block2, pool_logit2, cls_centroid2,
         pool_padding_mask2, pool_inds2, out2) = self._block_operations(
            cls_centroid1, pool_padding_mask1, self.blocks2, self.pool2)

        intermediates2 = {
            'logit2': pool_logit2, 'centroid2': cls_centroid2[:, 1:], 'block2': cls_block2[:, 1:],
            'padding_mask2': pool_padding_mask1, 'sampled_inds2': pool_inds2,
        }
        intermediates.update(intermediates2)

        # Block3.
        (cls_block3, pool_logit3, cls_centroid3,
         pool_padding_mask3, pool_inds3, out3) = self._block_operations(
            cls_centroid2, pool_padding_mask2, self.blocks3, self.pool3)

        intermediates3 = {
            'logit3': pool_logit3, 'centroid3': cls_centroid3[:, 1:], 'block3': cls_block3[:, 1:],
            'padding_mask3': pool_padding_mask2, 'sampled_inds3': pool_inds3,
        }
        intermediates.update(intermediates3)

        # Block4.
        (cls_block4, pool_logit4, cls_centroid4,
         pool_padding_mask4, pool_inds4, out4) = self._block_operations(
            cls_centroid3, pool_padding_mask3, self.blocks4, self.pool4)
        out4 = self.pre_logits(out4)

        intermediates4 = {
            'logit4': pool_logit4, 'centroid4': cls_centroid4[:, 1:], 'block4': cls_block4[:, 1:],
            'padding_mask4': pool_padding_mask3, 'out4': out4, 'sampled_inds4': pool_inds4,
        }
        intermediates.update(intermediates4)

        return intermediates

    def forward(self, x, y, return_intermediate=True):
        intermediates = self.forward_features(x, y)
        x = self.head(intermediates['out4'])

        if return_intermediate:
            return x, intermediates
        else:
            return x


class ConvStem(nn.Module):
    """ 
    ConvStem, from Early Convolutions Help Transformers See Better, Tete et al. https://arxiv.org/abs/2106.14881
    """
    def __init__(self, img_size=224, patch_size=8, in_chans=3, embed_dim=768, norm_layer=None, flatten=False):
        super().__init__()

        assert patch_size == 8, 'ConvStem only supports patch size of 8'
        assert embed_dim % 8 == 0, 'Embed dimension must be divisible by 2 for ConvStem'

        img_size = to_2tuple(img_size)
        patch_size = to_2tuple(patch_size)
        self.img_size = img_size
        self.patch_size = patch_size
        self.grid_size = (img_size[0] // patch_size[0], img_size[1] // patch_size[1])
        self.num_patches = self.grid_size[0] * self.grid_size[1]
        self.flatten = flatten

        # build stem, similar to the design in https://arxiv.org/abs/2106.14881
        stem = []
        input_dim, output_dim = 3, embed_dim // 8
        for l in range(4):
            stride = 2 if l < 3 else 1
            stem.append(nn.Conv2d(input_dim, output_dim, kernel_size=3, stride=stride, padding=1, bias=False))
            stem.append(nn.BatchNorm2d(output_dim))
            stem.append(nn.ReLU(inplace=True))
            input_dim = output_dim
            output_dim *= 2
        stem.append(nn.Conv2d(input_dim, embed_dim, kernel_size=1))
        self.proj = nn.Sequential(*stem)

        self.norm = norm_layer(embed_dim) if norm_layer else nn.Identity()

    def forward(self, x):
        B, C, H, W = x.shape
        assert H == self.img_size[0] and W == self.img_size[1], \
            f"Input image size ({H}*{W}) doesn't match model ({self.img_size[0]}*{self.img_size[1]})."
        x = self.proj(x)
        if self.flatten:
            x = x.flatten(2).transpose(1, 2)  # BCHW -> BNC
        else:
            x = x.permute(0, 2, 3, 1) # BxCxHxW -> BxHxWXC
        x = self.norm(x)
        return x


def vit_conv_small(**kwargs):
    # minus one ViT block
    model = VisionTransformerMoCo(
        patch_size=8, embed_dim=384, num_clusters=[64, 32, 16, 8],
        depth=[3, 3, 3, 2], num_heads=12, mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), embed_layer=ConvStem, **kwargs)
    model.default_cfg = _cfg()
    return model

def vit_conv_base(**kwargs):
    # minus one ViT block
    model = VisionTransformerMoCo(
        patch_size=8, embed_dim=768, num_clusters=[64, 32, 16, 8],
        depth=[3, 3, 3, 2], num_heads=12, mlp_ratio=4, qkv_bias=True,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), embed_layer=ConvStem, **kwargs)
    model.default_cfg = _cfg()
    return model
