%% Stochastic Zeroth-Order Constraint Extrapolation (SZO-ConEX) method
% This code implements a distributed zeroth-order optimization algorithm
% for power systems with inequality constraints using Gaussian smoothing
% and constraint handling techniques. It features multi-loop structure.

preprocess;
%% Algorithm Parameters
n_trial = 50;               % Number of independent trials
M=200;                      % Outer loop iterations
T =100;                     % Inner loop iterations

% Initialize recording matrices
fvals = zeros(T*M, n_trial); 
lvals=zeros(T*M,n_trial); 
gnorms = zeros(T*M, n_trial); 
consv= zeros(T*M,n_trial); 

rk = repmat(min(1e-2./((4000:T+4000-1).^1.1)', 1e-5), 1, n_trial);  % Smoothing radius
etak = 5e-6 * ones(T, n_trial);                                     % Stepsize
deltak = repmat(min(50 ./ (1:T)', 0.1), 1, n_trial);                % Constraint handling technique parameter
thetak=ones(T, n_trial);                                            % Constant for extrapolation step
mu0=5;
mu1=10;

x0 = [res_pf_original.bus(idx_load_p, 3); res_pf_original.bus(idx_load_q, 4)] / baseMVA;    
%% Main Optimization Loop (Parallel Trials)
parfor pp = 1:n_trial
    %%% Initialize variables for current trial
    rng(pp);
    x_cur= x0*pp/n_trial;
    stationarity=zeros(dim+1, 1);
    p_bus = zeros(n_bus, 1);
    q_bus = zeros(n_bus, 1);
    p_bus_pert1 = zeros(n_bus, 1);
    q_bus_pert1 = zeros(n_bus, 1);
    p_bus_pert2 = zeros(n_bus, 1);
    q_bus_pert2 = zeros(n_bus, 1);
    p_bus_last_pert = zeros(n_bus, 1);
    q_bus_last_pert = zeros(n_bus, 1);
    lambd=0;
    l_cur=0;
    l_last=0;
    x_ref = x_cur % Reference point for outer loop

    %%% Reference point for outer loop
    fvals_pp = zeros(T*M,1);
    lvals_pp = zeros(T*M,1);
    gnorms_pp = zeros(T*M,1);
    consv_pp  = zeros(T*M,1);

    for m =1 : M %Outer Loop (Handles Non-Convexity)
        %%% Calculate objective components
        p_bus(idx_load_p) = x_cur(1:dim_real);
        q_bus(idx_load_p) = x_cur(dim_real+1:end);
        [V_phasor, S_slack] = pf_radial(p_bus, q_bus, V_slack, Mat_pf_back, Mat_pf_forward, branch_z, ...
            idx_slack, n_bus);
        Vm = abs(V_phasor);
        Va = angle(V_phasor);
        p_slack = real(S_slack);
        phi_cur = global_obj(p_slack, Vm, p_desired, Vm_lb, Vm_ub, dev_alpha, pen_alpha);
        localf_cur = sum(local_a .* (x_cur-x_nom).^2 + local_b .* (x_nom-x_cur));
        f_cur = phi_cur + localf_cur;
        x_last=x_cur;
        p_slack_last=p_slack;
        for k = 1:T % Inner Loop (Main Optimization)
            index = (m-1)*T + k;
            %%% Extrapolation step
            tilde_z_pert = randn(dim, 1);
            r_cur = rk(k, pp);
            z_pert = (max(min(x_last + r_cur * tilde_z_pert, x_ub), x_lb) - x_last) / r_cur;
            x_last_pert = x_last + r_cur * z_pert;
        
            p_bus_last_pert(idx_load_p) = x_last_pert(1:dim_real);
            q_bus_last_pert(idx_load_p) = x_last_pert(dim_real+1:end);
            [V_phasor_last_pert, S_slack_last_pert] = pf_radial(p_bus_last_pert, q_bus_last_pert, V_slack, Mat_pf_back, Mat_pf_forward, branch_z, ...
                idx_slack, n_bus);
            Vm_last_pert = abs(V_phasor_last_pert);
            Va_last_pert = angle(V_phasor_last_pert);
            p_slack_last_pert = real(S_slack_last_pert);
            phi_last_pert = global_obj(p_slack_last_pert, Vm_last_pert, p_desired, Vm_lb, Vm_ub, dev_alpha, pen_alpha);
            
            %   Linearization with the Gaussian smoothing-based stochastic
            %   zeroth-order gradients estimate
            l_cur = p_slack_last_pert-p_desired + mu1*norm(x_last_pert-x_ref)^2 ...% F_v
                    +((p_slack_last_pert-p_slack_last)/r_cur*z_pert+2*mu1*(x_last-x_ref))'*(x_cur-x_last); % G_v
            if k<2
                l_last=l_cur;
            end
            s_cur = (1+thetak(k,pp))*l_cur - thetak(k,pp) * l_last;

            %%% Dual Variable Update 
            lambd=max(lambd+etak(k,pp)*s_cur,0);

            %%% Primal Variable Update (Decision Variables)
            % Evaluate current point
            p_bus(idx_load_p) = x_cur(1:dim_real);
            q_bus(idx_load_p) = x_cur(dim_real+1:end);
            [V_phasor, S_slack] = pf_radial(p_bus, q_bus, V_slack, Mat_pf_back, Mat_pf_forward, branch_z, ...
                idx_slack, n_bus);
            Vm = abs(V_phasor);
            Va = angle(V_phasor);
            p_slack = real(S_slack);
            phi_cur = global_obj(p_slack, Vm, p_desired, Vm_lb, Vm_ub, dev_alpha, pen_alpha);
            localf_cur = sum(local_a .* (x_cur-x_nom).^2 + local_b .* (x_nom-x_cur));
            f_cur = phi_cur + localf_cur;

            % First perturbation for objective gradient
            tilde_z_pert = randn(dim, 1);
            r_cur = rk(k, pp);
            z_pert1 = (max(min(x_cur + r_cur * tilde_z_pert, x_ub), x_lb) - x_cur) / r_cur;
            x_pert1 = x_cur + r_cur * z_pert1;

            p_bus_pert1(idx_load_p) = x_pert1(1:dim_real);
            q_bus_pert1(idx_load_p) = x_pert1(dim_real+1:end);
            [V_phasor_pert1, S_slack_pert1] = pf_radial(p_bus_pert1, q_bus_pert1, V_slack, Mat_pf_back, Mat_pf_forward, branch_z, ...
                idx_slack, n_bus);
            Vm_pert1 = abs(V_phasor_pert1);
            Va_pert1 = angle(V_phasor_pert1);
            p_slack_pert1 = real(S_slack_pert1);
            phi_pert1 = global_obj(p_slack_pert1, Vm_pert1, p_desired, Vm_lb, Vm_ub, dev_alpha, pen_alpha);

            % Second perturbation for constraint gradient
            tilde_z_pert = randn(dim, 1);
            r_cur = rk(k, pp);
            z_pert2 = (max(min(x_cur + r_cur * tilde_z_pert, x_ub), x_lb) - x_cur) / r_cur;
            x_pert2 = x_cur + r_cur * z_pert2;
        
            p_bus_pert2(idx_load_p) = x_pert2(1:dim_real);
            q_bus_pert2(idx_load_p) = x_pert2(dim_real+1:end);
            [V_phasor_pert2, S_slack_pert2] = pf_radial(p_bus_pert2, q_bus_pert2, V_slack, Mat_pf_back, Mat_pf_forward, branch_z, ...
                idx_slack, n_bus);
            Vm_pert2 = abs(V_phasor_pert2);
            Va_pert2 = angle(V_phasor_pert2);
            p_slack_pert2 = real(S_slack_pert2);
            phi_pert2 = global_obj(p_slack_pert2, Vm_pert2, p_desired, Vm_lb, Vm_ub, dev_alpha, pen_alpha);
               
            % Combined gradient estimation
            grad = (phi_pert1-phi_cur)*z_pert1/r_cur + 2 * local_a .* (x_cur - x_nom) - local_b + 2*mu0*(x_cur-x_ref);
            grad = grad + lambd*((p_slack_pert2-p_slack)*z_pert2/r_cur + 2*mu1*(x_cur-x_ref));
            
            % Projected gradient step
            x_next = x_cur - etak(k, pp) * grad;
            x_next = max(min(x_next, x_ub), x_lb);
            
            %%% Metrics Calculation
            fvals_pp(index) = f_cur;                        % Objective value record  
            grad_x = grad_phi(Vm, Va, p_slack, p_desired, dev_alpha, pen_alpha, Vm_lb, Vm_ub, V_slack, Ybus, ...
                idx_slack, idx_nonslack, n_nonslack, idx_var) + ...
                2 * local_a .* (x_cur - x_nom) - local_b + ...
                lambd*grad_p_slack(Vm, Va, V_slack, Ybus,idx_slack, idx_nonslack, n_nonslack, idx_var);
            grad_y = p_slack-p_desired;
            stationarity(1:dim)=(x_cur-max(min(x_cur-etak(k,pp)*grad_x, x_ub), x_lb))/etak(k,pp)
            stationarity(dim+1)=(lambd-max(lambd+etak(k,pp) *grad_y,0) )/etak(k,pp);
            gnorms_pp(index) = norm(stationarity);          % Stationarity measure record
            lvals_pp(index) = lambd;                        % Dual variable record
            cons_error=max(p_slack-p_desired, 0);
            consv_pp(index)=norm(cons_error) ;              % Constraint violation recor
            
            
            x_last=x_cur;
            x_cur =x_next;
            l_last=l_cur;
            p_slack_last=p_slack;

        end
        x_ref=x_cur;
    end

    % Store trial results
    fvals(:,pp)  = fvals_pp;
    gnorms(:,pp) = gnorms_pp;
    lvals(:,pp)  = lvals_pp;
    consv(:,pp)  = consv_pp;  
end

save res_SZO_ConEx_50times;

