clearvars;
% Load the IEEE 141-bus system case and extract base power (MVA)
mpc_original = loadcase('case141');
baseMVA = mpc_original.baseMVA;
% Get number of buses
n_bus = length(mpc_original.bus(:, 1));
% Identify and process slack bus
idx_slack = find(mpc_original.bus(:, 2) == 3);
mpc_original.bus(idx_slack, 3) = 0; % no real load at slack bus
mpc_original.bus(idx_slack, 4) = 0; % no reactive load at slack bus
mpc_original.bus(idx_slack, 9) = 0; % zero phase at slack bus
V_slack = mpc_original.bus(idx_slack, 8);
% Identify non-slack buses
idx_nonslack = find((1:n_bus)' ~= idx_slack);
n_nonslack = length(idx_nonslack);
% Create grid graph and compute minimum spanning tree
grid_graph = graph(mpc_original.branch(:, 1), mpc_original.branch(:, 2), mpc_original.branch(:, 3));
grid_graph_X = graph(mpc_original.branch(:, 1), mpc_original.branch(:, 2), mpc_original.branch(:, 4));
% Compute minimum spanning tree with slack bus as root
[~, pred] = minspantree(grid_graph, 'Root', idx_slack);
grid_digraph = digraph(pred(pred~=0),find(pred~=0));
grid_adj = adjacency(grid_digraph);
% Construct voltage magnitude forward and backward propagation matrices
Mat_pf_back = sparse(eye(n_bus)) - grid_adj;
Mat_pf_forward = Mat_pf_back'; %sparse(eye(n_bus)) - sparse(find(pred~=0), pred(pred~=0), ones(n_nonslack, 1), n_bus, n_bus);
% Calculate branch impedances (considering both resistance and reactance)）
branch_z = zeros(n_nonslack, 1);
for k = 1:n_bus
    if k == idx_slack
        branch_z(k) = NaN;
        continue;
    end
    idx_edge = findedge(grid_graph, k, pred(k));
    branch_z(k) = grid_graph.Edges.Weight(idx_edge) + 1j * grid_graph_X.Edges.Weight(idx_edge);
end
% Clear temporary variables
clear pred grid_graph grid_graph_X grid_digraph grid_adj idx_edge k;
% Generate power system admittance matrix
Ybus = makeYbus(mpc_original);
% Identify buses with active/reactive power loads
idx_load_p = find(mpc_original.bus(:, 3) ~= 0);
idx_load_q = find(mpc_original.bus(:, 4) ~= 0);
% Identify buses with controllable loads
idx_var = find([mpc_original.bus(idx_nonslack, 3) ~= 0; mpc_original.bus(idx_nonslack, 4) ~= 0]);
% Dimensions for active/reactive power loads
dim_real = length(idx_load_p);
dim_reactive = length(idx_load_q);
dim = dim_real + dim_reactive;
% Lower/upper bounds for active/reactive power
lb_real = zeros(dim_real, 1);
lb_reactive = zeros(dim_reactive, 1);
ub_real = mpc_original.bus(idx_load_p, 3) / baseMVA;
ub_reactive = mpc_original.bus(idx_load_q, 4) / baseMVA;
% Combine bounds
x_lb = [lb_real; lb_reactive];
x_ub = [ub_real; ub_reactive];
x_nom = x_ub;
% Voltage magnitude bounds
Vm_lb = 0.96 * ones(n_bus, 1);
Vm_ub = 1.04 * ones(n_bus, 1);
% Penalty parameters
pen_alpha = 20;
dev_alpha = 20;
% Configure power flow options and run initial power flow
mpopt_std = mpoption('pf.alg', 'PQSUM', 'verbose', 0, 'out.all', 0);
res_pf_original = runpf(mpc_original, mpopt_std);
% Calculate nominal active power and desired power reduction target
p_nom = sum(res_pf_original.gen(:, 2)) / baseMVA;
p_reduction = 0.15;
p_desired = p_nom - p_reduction;
% Initialize random seed and generate random power coefficients
rng(0);
local_a = rand(dim, 1) + 0.5;
local_b = rand(dim, 1) * 5;
rng('shuffle');
