%% ZO-MinMax: Zeroth-Order Min-Max Optimization
% % This code implements a zeroth-order min-max optimization algorithm 
% using Gaussian distribution for gradient estimation and constraint handling techniques.

preprocess;
%% Algorithm Parameters
n_trial = 50;                       % Number of independent trials
T = 20000;                          % Number of iterations

% Initialize recording matrices
fvals = zeros(T, n_trial) ;
lvals = zeros(T, n_trial);
gnorms = zeros(T, n_trial);
consv = zeros(T, n_trial) ;


rk = repmat(min(1e-2./((4000:T+4000-1).^1.1)', 1e-5), 1, n_trial);  % Smoothing radius
etak = 5e-6 * ones(T, n_trial);                                     % Stepsize
deltak = repmat(min(50 ./ (1:T)', 0.1), 1, n_trial);                % Constraint handling technique parameter
x0 = [res_pf_original.bus(idx_load_p, 3); res_pf_original.bus(idx_load_q, 4)] / baseMVA;


%% Main Optimization Loop (Parallel Trials)
parfor pp = 1:n_trial
    %%% Initialize variables for current trial
    rng(pp);  
    x_cur= x0*pp/n_trial;
    stationarity=zeros(dim+1, 1);
    p_bus = zeros(n_bus, 1);
    q_bus = zeros(n_bus, 1);
    p_bus_pert = zeros(n_bus, 1);
    q_bus_pert = zeros(n_bus, 1);
    lambd=0;
    fvals_pp  = zeros(T,1);
    lvals_pp  = zeros(T,1);
    gnorms_pp = zeros(T,1);
    consv_pp  = zeros(T,1);
    
    for k = 1:T
        %%% Evaluate Current Point
        p_bus(idx_load_p) = x_cur(1:dim_real);
        q_bus(idx_load_p) = x_cur(dim_real+1:end);
        [V_phasor, S_slack] = pf_radial(p_bus, q_bus, V_slack, Mat_pf_back, Mat_pf_forward, branch_z, ...
            idx_slack, n_bus);
        Vm = abs(V_phasor);
        Va = angle(V_phasor);
        p_slack = real(S_slack);
        
        phi_cur = global_obj(p_slack, Vm, p_desired, Vm_lb, Vm_ub, dev_alpha, pen_alpha);
        localf_cur = sum(local_a .* (x_cur-x_nom).^2 + local_b .* (x_nom-x_cur));
        f_cur = phi_cur + localf_cur;
        
        %%% Update Dual Variable
        lambd=max(lambd+etak(k,pp)*(p_slack-p_desired),0) 

        %%% Metrics Calculation
        cons_error=max(p_slack-p_desired, 0);
        grad_x = grad_phi(Vm, Va, p_slack, p_desired, dev_alpha, pen_alpha, Vm_lb, Vm_ub, V_slack, Ybus, ...
            idx_slack, idx_nonslack, n_nonslack, idx_var) + ...
            2 * local_a .* (x_cur - x_nom) - local_b + ...
            lambd*grad_p_slack(Vm, Va, V_slack, Ybus,idx_slack, idx_nonslack, n_nonslack, idx_var);
        grad_y = p_slack-p_desired;
        stationarity(1:dim)=(x_cur-max(min(x_cur-etak(k,pp)*grad_x, x_ub), x_lb))/etak(k,pp)
        stationarity(dim+1)=(lambd-max(lambd+etak(k,pp) *grad_y,0) )/etak(k,pp);       
        fvals_pp(k)  = f_cur;                   
        lvals_pp(k)  = lambd;
        consv_pp(k)  = norm(cons_error);
        gnorms_pp(k) = norm(stationarity);
        
        %%% Gaussian Perturbation and Update
        tilde_z_pert = randn(dim, 1);
        r_cur = rk(k, pp);
        z_pert = (max(min(x_cur + r_cur * tilde_z_pert, x_ub), x_lb) - x_cur) / r_cur;  
        x_pert = x_cur + r_cur * z_pert;

        % Evaluate perturbed point
        p_bus_pert(idx_load_p) = x_pert(1:dim_real);
        q_bus_pert(idx_load_p) = x_pert(dim_real+1:end);
        [V_phasor_pert, S_slack_pert] = pf_radial(p_bus_pert, q_bus_pert, V_slack, Mat_pf_back, Mat_pf_forward, branch_z, ...
            idx_slack, n_bus);
        Vm_pert = abs(V_phasor_pert);
        Va_pert = angle(V_phasor_pert);
        p_slack_pert = real(S_slack_pert);
        phi_pert = global_obj(p_slack_pert, Vm_pert, p_desired, Vm_lb, Vm_ub, dev_alpha, pen_alpha);

        % Primal variable update
        x_next = x_cur - etak(k, pp) * ...
            (2 * local_a .* (x_cur - x_nom) - local_b ...
            + (phi_pert+lambd*p_slack_pert - phi_cur-lambd*p_slack) / r_cur * z_pert);
        x_next = max(min(x_next, x_ub), x_lb);
        x_cur = x_next;

    end
    % Store trial results
    fvals(:,pp)  = fvals_pp;
    lvals(:,pp)  = lvals_pp;
    gnorms(:,pp) = gnorms_pp;
    consv(:,pp)  = consv_pp;
     
end

save res_ZO_MinMax_50times;