# coding=utf-8
# Copyright 2023-present the HuggingFace Inc. team.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import warnings
from typing import Any, List, Optional
import math
import torch
import torch.nn as nn

from peft.tuners.tuners_utils import BaseTunerLayer
from .utils import circulant, BlockCircularConvolution, get_circulant_fast

class C3ALayer(BaseTunerLayer):
    # All names of layers that may contain (trainable) adapter weights
    adapter_layer_names = ("c3a_kernel",)
    # All names of other parameters that may contain adapter-related parameters
    other_param_names = ("block_size",)

    def __init__(self, base_layer: nn.Module, **kwargs) -> None:
        self.base_layer = base_layer
        self.block_size = {}
        self.c3a_kernel = nn.ParameterDict({})
        self.c3a_magnitude_vector = nn.ParameterDict({})
        # Mark the weight as unmerged
        self._disable_adapters = False
        self.merged_adapters = []
        self.kwargs = kwargs

        base_layer = self.get_base_layer()
        if isinstance(base_layer, nn.Linear):
            self.in_features, self.out_features = base_layer.in_features, base_layer.out_features
        else:
            raise ValueError(f"Unsupported layer type {type(base_layer)}")

    def get_delta_weight(self, adapter) -> torch.Tensor:
        if adapter not in self.c3a_kernel.keys():
            raise ValueError(f"Adapter {adapter} not found.")
        base_layer_weight = self.get_base_layer().weight
        c3a_kernel = self.c3a_kernel[adapter]

        # Slower Version
        # block_size = self.block_size[adapter]
        # delta_weight = torch.zeros_like(base_layer_weight)
        # for i in range(c3a_kernel.size(0)):
        #     for j in range(c3a_kernel.size(1)):
        #         block_delta_weight = circulant(c3a_kernel[i, j])
        #         delta_weight[block_size * i : block_size * (i + 1), block_size * j : block_size * (j + 1)] = block_delta_weight
        
        # Faster Version
        previous_dtype = c3a_kernel.dtype
        eye_matrix = torch.eye(base_layer_weight.size(-1), device=base_layer_weight.device)
        delta_weight = get_circulant_fast(eye_matrix, c3a_kernel.float())
        return delta_weight.to(previous_dtype) / base_layer_weight.size(-1)

    def update_layer(self, adapter_name, block_size, init_weights, use_dc3a):
        if block_size <= 0:
            raise ValueError(f"`block_size` should be a positive integer value but the value passed is {block_size}")
        assert self.in_features % block_size == 0, f"The block size should be a factor of the input size. However, the input size is {self.in_features} and the block size is {block_size}"
        assert self.out_features % block_size == 0, f"The block size should be a factor of the output size. However, the output size is {self.out_features} and the block size is {block_size}"
        self.block_size[adapter_name] = block_size

        weight = getattr(self.get_base_layer(), "weight", None)
        self.c3a_kernel[adapter_name] = nn.Parameter(
            torch.zeros(self.out_features//block_size, self.in_features//block_size, block_size, dtype=weight.dtype, device=weight.device)
        )

        if weight is not None:
            # the layer is already completely initialized, this is an update
            if weight.dtype.is_floating_point or weight.dtype.is_complex:
                self.to(weight.device, dtype=weight.dtype)
            else:
                self.to(weight.device)

        self.reset_c3a_parameters(adapter_name, init_weights)
        
        delta_weight = self.get_delta_weight(adapter_name)
        
        if use_dc3a:
            if not self.c3a_magnitude_vector:
                # first dora layer being added, add c3a_magnitude_vector to the list of learnable parameters
                self.adapter_layer_names = self.adapter_layer_names[:] + ("c3a_magnitude_vector",)

            w_new = delta_weight.clone().detach() + weight.clone().detach()
            self.c3a_magnitude_vector[adapter_name] = nn.Parameter(
                w_new.norm(dim=1, keepdim=False), requires_grad=True
            )

        self.set_adapter(self.active_adapters)

    @torch.no_grad()
    def reset_c3a_parameters(self, adapter_name, init_weights):
        if init_weights is False:
            return

        if adapter_name in self.c3a_kernel.keys():
            if init_weights == "gaussian":
                nn.init.normal_(self.c3a_kernel[adapter_name])
            elif init_weights in ["xavier_uniform", True]:
                fan_in, fan_out = self.in_features, self.out_features
                std = 1.0 * math.sqrt(2.0 / float(fan_in + fan_out))
                a = math.sqrt(3.0) * std  # Calculate uniform bounds from standard deviation
                nn.init.uniform_(self.c3a_kernel[adapter_name], -a, a)
            elif init_weights == "kaiming_uniform":
                fan_in = self.in_features
                a = 1.0 * math.sqrt(1.0 / float(fan_in))
                nn.init.uniform_(self.c3a_kernel[adapter_name], -a, a)
            else:
                raise ValueError(f"Unknown init_weights: {init_weights}")

#  ------------------------------------------------------------------------------------------
#  Copyright (c) Microsoft Corporation. All rights reserved.
#  Licensed under the MIT License (MIT). See LICENSE in the repo root for license information.
#  ------------------------------------------------------------------------------------------


class C3ALinear(nn.Module, C3ALayer):
    # Lora implemented in a dense layer
    def __init__(
        self,
        base_layer,
        adapter_name: str,
        block_size: int,
        init_weights: bool = True,
        use_dc3a: bool = False,
        **kwargs,
    ) -> None:
        super().__init__()
        C3ALayer.__init__(self, base_layer, **kwargs)
        self._active_adapter = adapter_name
        self.update_layer(adapter_name, block_size, init_weights, use_dc3a)

    def merge(self, safe_merge: bool = False, adapter_names: Optional[List[str]] = None) -> None:
        """
        Merge the active adapter weights into the base weights

        Args:
            safe_merge (`bool`, *optional*):
                If True, the merge operation will be performed in a copy of the original weights and check for NaNs
                before merging the weights. This is useful if you want to check if the merge operation will produce
                NaNs. Defaults to `False`.
            adapter_names (`List[str]`, *optional*):
                The list of adapter names that should be merged. If None, all active adapters will be merged. Defaults
                to `None`.
        """
        if self.merged:
            warnings.warn(
                f"Already following adapters were merged {','.join(self.merged_adapters)}. "
                f"You are now additionally merging {','.join(self.active_adapters)}."
            )

        if adapter_names is None:
            adapter_names = self.active_adapters

        for active_adapter in adapter_names:
            if active_adapter in self.c3a_kernel.keys():
                base_layer = self.get_base_layer()
                if safe_merge:
                    # Note that safe_merge will be slower than the normal merge
                    # because of the copy operation.
                    orig_weights = base_layer.weight.data.clone()
                    # orig_weights += self.get_delta_weight(active_adapter)
                    delta_weight = self.get_delta_weight(active_adapter)
                    if not active_adapter in self.c3a_magnitude_vector.keys():
                        orig_weights = orig_weights + delta_weight
                    else:
                        # handle dora
                        # since delta_weight already includes scaling, set it to 1 here
                        weight_norm = (
                            (orig_weights + delta_weight).norm(dim=1, keepdim=False)
                        )
                        # We need to cache weight_norm because it has to be based on the original weights. We
                        # cannot calculate it on the fly based on the merged weights when unmerging because its a
                        # different value
                        # self._cache_store(f"{active_adapter}-weight_norm", weight_norm)
                        dora_factor = self.c3a_magnitude_vector[active_adapter] / weight_norm
                        dora_factor = dora_factor.view(-1, 1)
                        orig_weights = dora_factor * (orig_weights + delta_weight)

                    if not torch.isfinite(orig_weights).all():
                        raise ValueError(
                            f"NaNs detected in the merged weights. The adapter {active_adapter} seems to be broken"
                        )

                    base_layer.weight.data = orig_weights
                else:
                    delta_weight = self.get_delta_weight(active_adapter)
                    if not active_adapter in self.c3a_magnitude_vector.keys():
                        base_layer.weight.data = base_layer.weight.data + delta_weight
                    else:
                        weight_norm = (
                            (base_layer.weight + delta_weight).norm(dim=1, keepdim=False)
                        )
                        # We need to cache weight_norm because it has to be based on the original weights. We
                        # cannot calculate it on the fly based on the merged weights when unmerging because its a
                        # different value
                        # self._cache_store(f"{active_adapter}-weight_norm", weight_norm)
                        dora_factor = self.c3a_magnitude_vector[active_adapter] / weight_norm
                        dora_factor = dora_factor.view(-1, 1)
                        new_weight = dora_factor * (base_layer.weight.data + delta_weight)
                        base_layer.weight.data = new_weight
                        
                    # base_layer.weight.data += self.get_delta_weight(active_adapter)
                self.merged_adapters.append(active_adapter)

    def unmerge(self) -> None:
        """
        This method unmerges all merged adapter layers from the base weights.
        """
        if not self.merged:
            warnings.warn("Already unmerged. Nothing to do.")
            return
        while len(self.merged_adapters) > 0:
            active_adapter = self.merged_adapters.pop()
            if active_adapter in self.c3a_kernel.keys():
                self.get_base_layer().weight.data -= self.get_delta_weight(active_adapter)
            
    def forward(self, x: torch.Tensor, *args: Any, **kwargs: Any) -> torch.Tensor:
        previous_dtype = x.dtype

        if self.disable_adapters:
            if self.merged:
                self.unmerge()
            result = self.base_layer(x, *args, **kwargs)
        elif self.merged:
            result = self.base_layer(x, *args, **kwargs)
        else:
            result = self.base_layer(x, *args, **kwargs)
            for active_adapter in self.active_adapters:
                if active_adapter not in self.c3a_kernel.keys():
                    continue
                c3a_kernel = self.c3a_kernel[active_adapter]
                x = BlockCircularConvolution.apply(x.float(), c3a_kernel.float()) / x.size(-1)
                x = x.to(c3a_kernel.dtype)
                result += x
                if active_adapter in self.c3a_magnitude_vector.keys():
                    # result *= self.c3a_magnitude_vector[active_adapter].view(1, -1)
                    weight_norm = (self.base_layer.weight.clone().detach() + self.get_delta_weight(active_adapter).clone().detach()).norm(dim=1, keepdim=False)
                    # result /= weight_norm.view(1, -1)
                    dora_factor = (self.c3a_magnitude_vector[active_adapter] / weight_norm).view(1, -1)
                    result = result * dora_factor
                    

        result = result.to(previous_dtype)
        return result

    def __repr__(self) -> str:
        rep = super().__repr__()
        return "c3a." + rep