import networkx as nx
from typing import Callable, Optional
import numpy as np
import os.path as osp
import pickle as pkl
import scipy.sparse as sp
import sys
import torch
from torch_sparse import coalesce, SparseTensor
from torch_geometric.data import InMemoryDataset, download_url, Data
import torch_geometric.transforms as T
from torch_geometric.utils import from_scipy_sparse_matrix
from torch_geometric.utils.undirected import is_undirected, to_undirected
from time import perf_counter

from .normalization import fetch_normalization, row_normalize, AugNorm
from .utils import sp_mtx_to_sp_tnsr


def preprocess_citation(adj, features, normalization=['AugNorm']):
    if isinstance(normalization,list):
        for norm in normalization:
            adj_normalizer = fetch_normalization(norm)
            adj = adj_normalizer(adj)
    else:
        adj_normalizer = fetch_normalization(normalization)
        adj = adj_normalizer(adj)
    features = row_normalize(features)
    return adj, features

def load_chain(normalization, cuda, num_chains, chain_len, num_class=2,
                                noise=0.00, noise_type=None, need_orig=False, adj_pow=1):
    """load the synthetic dataset: chain"""
    c = num_class # num of classes
    n = num_chains # chains for each class
    l = chain_len # length of chain
    f = 100 # feature dimension
    num_nodes = c*n*l
    tn = int(num_nodes*0.05)
    vl = int(num_nodes*0.1)
    tt = num_nodes - tn - vl
    noise = noise

    # directed chains
    chain_adj = sp.coo_matrix((np.ones(l-1), (np.arange(l-1), np.arange(1, l))), shape=(l, l))
    adj = sp.block_diag([chain_adj for _ in range(c*n)]) # square matrix N = c*n*l

    if noise_type is None:
        features = np.random.uniform(-noise, noise, size=(c,n,l,f))
    elif noise_type == 'normal':
        features = np.random.normal(0, noise, size=(c,n,l,f))
    elif noise_type == 'normal_v2':
        features = np.random.normal(0, 1, size=(c,n,l,f)) * noise
    else:
        raise RuntimeError(f'Cannot find noise type {noise_type}')

    features[:, :, 0, :c] += np.eye(c).reshape(c, 1, c) # add class info to the first node of chains.
    features = features.reshape(-1, f)

    labels = np.eye(c).reshape(c, 1, 1, c).repeat(n, axis=1).repeat(l, axis=2) # one-hot labels
    labels = labels.reshape(-1, c)

    idx_random = np.arange(c*n*l)
    np.random.shuffle(idx_random)
    idx_train = idx_random[:tn]
    idx_val = idx_random[tn:tn+vl]
    idx_test = idx_random[tn+vl:tn+vl+tt]

    new_adj = [adj]
    for _ in range(1,adj_pow):
        new_adj += [new_adj[-1] @ adj]
    adj = sum(new_adj)

    adj, features = preprocess_citation(adj, features, normalization)
    features = torch.FloatTensor(np.array(features.todense() if sp.issparse(features) else features)).float()
    labels = torch.LongTensor(labels)
    labels = torch.max(labels, dim=1)[1]
    sp_adj = adj
    edge_index, _ = from_scipy_sparse_matrix(adj)
    adj = sp_mtx_to_sp_tnsr(adj).float()
    idx_train = torch.LongTensor(idx_train)
    idx_val = torch.LongTensor(idx_val)
    idx_test = torch.LongTensor(idx_test)

    if cuda:
        features = features.cuda()
        adj = adj.cuda()
        labels = labels.cuda()
        idx_train = idx_train.cuda()
        idx_val = idx_val.cuda()
        idx_test = idx_test.cuda()
    

    return adj, sp_adj, features, labels, idx_train, idx_val, idx_test, edge_index


def load_raw_graph(adj_pow,dataset_str = "amazon-all", normalization=['AugNorm']):
    txt_file = '/root/workspace/data/' + dataset_str + '/adj_list.txt'
    graph = {}
    with open(txt_file, 'r') as f:
        cur_idx = 0
        for row in f:
            row = row.strip().split()
            adjs = []
            for j in range(1, len(row)):
                adjs.append(int(row[j]))
            graph[cur_idx] = adjs
            cur_idx += 1
    adj = nx.adjacency_matrix(nx.from_dict_of_lists(graph))

    # DIFFUSION
    new_adj = [adj]
    for i in range(1,adj_pow):
        new_adj += [new_adj[-1] @ adj]
    adj = sum(new_adj)

    adj_normalizer = fetch_normalization(normalization)
    adj = adj_normalizer(adj)
    sp_adj = adj
    adj = sp_mtx_to_sp_tnsr(adj).float()
    return adj, sp_adj

def load_txt_data(adj_pow, dataset_str = "amazon-all", portion = '0.06', normalization=['AugNorm']):
    adj, sp_adj = load_raw_graph(adj_pow,dataset_str,normalization=normalization)
    idx_train = list(np.loadtxt('data/' + dataset_str + '/train_idx-' + str(portion) + '.txt', dtype=int))
    idx_val = list(np.loadtxt('data/' + dataset_str + '/test_idx.txt', dtype=int))
    idx_test = list(np.loadtxt('data/' + dataset_str + '/test_idx.txt', dtype=int))
    labels = np.loadtxt('data/' + dataset_str + '/label.txt')
    with open('data/' + dataset_str + '/meta.txt', 'r') as f:
        num_nodes, num_class = [int(w) for w in f.readline().strip().split()]

    features = sp.identity(num_nodes)
    
    # porting to pytorch
    features = sp_mtx_to_sp_tnsr(features).float()
    labels = torch.FloatTensor(labels)
    #labels = torch.max(labels, dim=1)[1]
    idx_train = torch.LongTensor(idx_train)
    idx_val = torch.LongTensor(idx_val)
    idx_test = torch.LongTensor(idx_test)

    return adj, sp_adj, features, labels, idx_train, idx_val, idx_test, num_nodes, num_class


def parse_index_file(filename):
    """Parse index file."""
    index = []
    for line in open(filename):
        index.append(int(line.strip()))
    return index


def load_citation(dataset_str="cora", normalization="AugNorm", cuda=True, need_orig=False):
    names = ['x', 'y', 'tx', 'ty', 'allx', 'ally', 'graph']
    objects = []
    for i in range(len(names)):
        with open("/root/workspace/data/{}/ind.{}.{}".format(dataset_str.lower(),dataset_str.lower(), names[i]), 'rb') as f:
            if sys.version_info > (3, 0):
                objects.append(pkl.load(f, encoding='latin1'))
            else:
                objects.append(pkl.load(f))

    x, y, tx, ty, allx, ally, graph = tuple(objects)
    test_idx_reorder = parse_index_file("/root/workspace/data/{}/ind.{}.test.index".format(dataset_str.lower(),dataset_str))
    test_idx_range = np.sort(test_idx_reorder)

    if dataset_str == 'citeseer':
        # Fix citeseer dataset (there are some isolated nodes in the graph)
        # Find isolated nodes, add them as zero-vecs into the right position
        test_idx_range_full = range(min(test_idx_reorder), max(test_idx_reorder)+1)
        tx_extended = sp.lil_matrix((len(test_idx_range_full), x.shape[1]))
        tx_extended[test_idx_range-min(test_idx_range), :] = tx
        tx = tx_extended
        ty_extended = np.zeros((len(test_idx_range_full), y.shape[1]))
        ty_extended[test_idx_range-min(test_idx_range), :] = ty
        ty = ty_extended

    features = sp.vstack((allx, tx)).tolil()
    features[test_idx_reorder, :] = features[test_idx_range, :]
    adj = nx.adjacency_matrix(nx.from_dict_of_lists(graph))
    adj = adj + adj.T.multiply(adj.T > adj) - adj.multiply(adj.T > adj)
    labels = np.vstack((ally, ty))
    labels[test_idx_reorder, :] = labels[test_idx_range, :]

    Laplace = torch.tensor(np.diag(adj.sum(1))-adj,dtype=torch.float)

    idx_test = test_idx_range.tolist()
    idx_train = range(len(y))
    idx_val = range(len(y), len(y)+500)

    if need_orig:
        adj_orig = AugNorm(adj, need_orig=True)
        adj_orig = sp_mtx_to_sp_tnsr(adj_orig).float()
        if cuda:
            adj_orig = adj_orig.cuda()

    adj, features = preprocess_citation(adj, features, normalization)

    # porting to pytorch
    features = torch.FloatTensor(np.array(features.todense())).float()
    labels = torch.LongTensor(labels)
    labels = torch.max(labels, dim=1)[1]
    adj = sp_mtx_to_sp_tnsr(adj).float()
    idx_train = torch.LongTensor(idx_train)
    idx_val = torch.LongTensor(idx_val)
    idx_test = torch.LongTensor(idx_test)

    if cuda:
        features = features.cuda()
        adj = adj.cuda()
        labels = labels.cuda()
        Laplace = Laplace.cuda()
        idx_train = idx_train.cuda()
        idx_val = idx_val.cuda()
        idx_test = idx_test.cuda()

    return [adj, adj_orig] if need_orig else adj, features, labels, idx_train, idx_val, idx_test, Laplace


def sgc_precompute(features, adj, degree):
    t = perf_counter()
    adj_index = adj.coalesce().indices()
    adj_value = adj.coalesce().values()
    features_index = features.coalesce().indices()
    features_value = features.coalesce().values()
    m = adj.shape[0]
    n = adj.shape[1]
    k = features.shape[1]

    for i in range(degree):
        #features = torch.spmm(adj, features)
        features_index, features_value = torch.spspmm(adj_index, adj_value, features_index, features_value, m, n, k)
    precompute_time = perf_counter()-t
    return torch.sparse.FloatTensor(features_index, features_value, torch.Size(features.shape)), precompute_time



class WebKB_new(InMemoryDataset):
    r"""The WebKB datasets used in the
    `"Geom-GCN: Geometric Graph Convolutional Networks"
    <https://openreview.net/forum?id=S1e2agrFvS>`_ paper.
    Nodes represent web pages and edges represent hyperlinks between them.
    Node features are the bag-of-words representation of web pages.
    The task is to classify the nodes into one of the five categories, student,
    project, course, staff, and faculty.
    Args:
        root (string): Root directory where the dataset should be saved.
        name (string): The name of the dataset (:obj:`"Cornell"`,
            :obj:`"Texas"` :obj:`"Wisconsin"`).
        transform (callable, optional): A function/transform that takes in an
            :obj:`torch_geometric.data.Data` object and returns a transformed
            version. The data object will be transformed before every access.
            (default: :obj:`None`)
        pre_transform (callable, optional): A function/transform that takes in
            an :obj:`torch_geometric.data.Data` object and returns a
            transformed version. The data object will be transformed before
            being saved to disk. (default: :obj:`None`)
    """

    url = 'https://raw.githubusercontent.com/graphdml-uiuc-jlu/geom-gcn/master'

    def __init__(self, root, name, transform=None, pre_transform=None):
        self.name = name.lower()
        assert self.name in ['cornell', 'texas', 'wisconsin']

        super(WebKB_new, self).__init__(root, transform, pre_transform)
        self.data, self.slices = torch.load(self.processed_paths[0])

    @property
    def raw_dir(self):
        return osp.join(self.root, self.name, 'raw')

    @property
    def processed_dir(self):
        return osp.join(self.root, self.name, 'processed')

    @property
    def raw_file_names(self):
        return ['out1_node_feature_label.txt', 'out1_graph_edges.txt'] + [
            '{}_split_0.6_0.2_{}.npz'.format(self.name, i) for i in range(10)
        ]

    @property
    def processed_file_names(self):
        return 'data.pt'

    def download(self):
        for f in self.raw_file_names[:2]:
            download_url(f'{self.url}/new_data/{self.name}/{f}', self.raw_dir)
        for f in self.raw_file_names[2:]:
            download_url(f'{self.url}/splits/{f}', self.raw_dir)

    def process(self):
        with open(self.raw_paths[0], 'r') as f:
            data = f.read().split('\n')[1:-1]
            x = [[float(v) for v in r.split('\t')[1].split(',')] for r in data]
            x = torch.tensor(x, dtype=torch.float)

            y = [int(r.split('\t')[2]) for r in data]
            y = torch.tensor(y, dtype=torch.float)

        with open(self.raw_paths[1], 'r') as f:
            data = f.read().split('\n')[1:-1]
            data = [[int(v) for v in r.split('\t')] for r in data]
            edge_index = torch.tensor(data, dtype=torch.long).t().contiguous()
            edge_index = to_undirected(edge_index)
            edge_index, _ = coalesce(edge_index, None, x.size(0), x.size(0))

        train_masks, val_masks, test_masks = [], [], []
        for f in self.raw_paths[2:]:
            tmp = np.load(f)
            train_masks += [torch.from_numpy(tmp['train_mask']).to(torch.bool)]
            val_masks += [torch.from_numpy(tmp['val_mask']).to(torch.bool)]
            test_masks += [torch.from_numpy(tmp['test_mask']).to(torch.bool)]
        train_mask = torch.stack(train_masks, dim=1)
        val_mask = torch.stack(val_masks, dim=1)
        test_mask = torch.stack(test_masks, dim=1)
        data = Data(x=x, edge_index=edge_index, y=y, train_mask=train_mask,
                    val_mask=val_mask, test_mask=test_mask)
        data = data if self.pre_transform is None else self.pre_transform(data)
        torch.save(self.collate([data]), self.processed_paths[0])

    def __repr__(self):
        return '{}()'.format(self.name)

class WikipediaNetwork(InMemoryDataset):
    r"""The Wikipedia networks introduced in the
    `"Multi-scale Attributed Node Embedding"
    <https://arxiv.org/abs/1909.13021>`_ paper.
    Nodes represent web pages and edges represent hyperlinks between them.
    Node features represent several informative nouns in the Wikipedia pages.
    The task is to predict the average daily traffic of the web page.
    Args:
        root (string): Root directory where the dataset should be saved.
        name (string): The name of the dataset (:obj:`"chameleon"`,
            :obj:`"crocodile"`, :obj:`"squirrel"`).
        geom_gcn_preprocess (bool): If set to :obj:`True`, will load the
            pre-processed data as introduced in the `"Geom-GCN: Geometric
            Graph Convolutional Networks" <https://arxiv.org/abs/2002.05287>_`,
            in which the average monthly traffic of the web page is converted
            into five categories to predict.
            If set to :obj:`True`, the dataset :obj:`"crocodile"` is not
            available.
        transform (callable, optional): A function/transform that takes in an
            :obj:`torch_geometric.data.Data` object and returns a transformed
            version. The data object will be transformed before every access.
            (default: :obj:`None`)
        pre_transform (callable, optional): A function/transform that takes in
            an :obj:`torch_geometric.data.Data` object and returns a
            transformed version. The data object will be transformed before
            being saved to disk. (default: :obj:`None`)
    """

    raw_url = 'https://graphmining.ai/datasets/ptg/wiki'
    processed_url = ('https://raw.githubusercontent.com/graphdml-uiuc-jlu/'
                     'geom-gcn/f1fc0d14b3b019c562737240d06ec83b07d16a8f')

    def __init__(self, root: str, name: str, geom_gcn_preprocess: bool = True,
                 transform: Optional[Callable] = None,
                 pre_transform: Optional[Callable] = None):
        self.name = name.lower()
        self.geom_gcn_preprocess = geom_gcn_preprocess
        assert self.name in ['chameleon', 'crocodile', 'squirrel']
        if geom_gcn_preprocess and self.name == 'crocodile':
            raise AttributeError("The dataset 'crocodile' is not available in "
                                 "case 'geom_gcn_preprocess=True'")
        super().__init__(root, transform, pre_transform)
        self.data, self.slices = torch.load(self.processed_paths[0])

    @property
    def raw_dir(self) -> str:
        if self.geom_gcn_preprocess:
            return osp.join(self.root, self.name, 'geom_gcn', 'raw')
        else:
            return osp.join(self.root, self.name, 'raw')

    @property
    def processed_dir(self) -> str:
        if self.geom_gcn_preprocess:
            return osp.join(self.root, self.name, 'geom_gcn', 'processed')
        else:
            return osp.join(self.root, self.name, 'processed')

    @property
    def raw_file_names(self) -> str:
        if self.geom_gcn_preprocess:
            return (['out1_node_feature_label.txt', 'out1_graph_edges.txt'] +
                    [f'{self.name}_split_0.6_0.2_{i}.npz' for i in range(10)])
        else:
            return f'{self.name}.npz'

    @property
    def processed_file_names(self) -> str:
        return 'data.pt'

    def download(self):
        if self.geom_gcn_preprocess:
            for filename in self.raw_file_names[:2]:
                url = f'{self.processed_url}/new_data/{self.name}/{filename}'
                download_url(url, self.raw_dir)
            for filename in self.raw_file_names[2:]:
                url = f'{self.processed_url}/splits/{filename}'
                download_url(url, self.raw_dir)
        else:
            download_url(f'{self.raw_url}/{self.name}.npz', self.raw_dir)

    def process(self):
        if self.geom_gcn_preprocess:
            with open(self.raw_paths[0], 'r') as f:
                data = f.read().split('\n')[1:-1]
            x = [[float(v) for v in r.split('\t')[1].split(',')] for r in data]
            x = torch.tensor(x, dtype=torch.float)
            y = [int(r.split('\t')[2]) for r in data]
            y = torch.tensor(y, dtype=torch.long)

            with open(self.raw_paths[1], 'r') as f:
                data = f.read().split('\n')[1:-1]
                data = [[int(v) for v in r.split('\t')] for r in data]
            edge_index = torch.tensor(data, dtype=torch.long).t().contiguous()
            edge_index, _ = coalesce(edge_index, None, x.size(0), x.size(0))

            train_masks, val_masks, test_masks = [], [], []
            for filepath in self.raw_paths[2:]:
                f = np.load(filepath)
                train_masks += [torch.from_numpy(f['train_mask'])]
                val_masks += [torch.from_numpy(f['val_mask'])]
                test_masks += [torch.from_numpy(f['test_mask'])]
            train_mask = torch.stack(train_masks, dim=1).to(torch.bool)
            val_mask = torch.stack(val_masks, dim=1).to(torch.bool)
            test_mask = torch.stack(test_masks, dim=1).to(torch.bool)

            data = Data(x=x, edge_index=edge_index, y=y, train_mask=train_mask,
                        val_mask=val_mask, test_mask=test_mask)

        else:
            data = np.load(self.raw_paths[0], 'r', allow_pickle=True)
            x = torch.from_numpy(data['features']).to(torch.float)
            edge_index = torch.from_numpy(data['edges']).to(torch.long)
            edge_index = edge_index.t().contiguous()
            edge_index, _ = coalesce(edge_index, None, x.size(0), x.size(0))
            y = torch.from_numpy(data['target']).to(torch.float)

            data = Data(x=x, edge_index=edge_index, y=y)

        if self.pre_transform is not None:
            data = self.pre_transform(data)

        torch.save(self.collate([data]), self.processed_paths[0])

# class WikipediaNetwork(InMemoryDataset):
#     r"""The Wikipedia networks used in the
#     `"Geom-GCN: Geometric Graph Convolutional Networks"
#     <https://openreview.net/forum?id=S1e2agrFvS>`_ paper.
#     Nodes represent web pages and edges represent hyperlinks between them.
#     Node features represent several informative nouns in the Wikipedia pages.
#     The task is to classify the nodes into five categories in term of the
#     number of average monthly traffic of the web page.
#     Args:
#         root (string): Root directory where the dataset should be saved.
#         name (string): The name of the dataset (:obj:`"Chameleon"`,
#             :obj:`"Squirrel"`).
#         transform (callable, optional): A function/transform that takes in an
#             :obj:`torch_geometric.data.Data` object and returns a transformed
#             version. The data object will be transformed before every access.
#             (default: :obj:`None`)
#         pre_transform (callable, optional): A function/transform that takes in
#             an :obj:`torch_geometric.data.Data` object and returns a
#             transformed version. The data object will be transformed before
#             being saved to disk. (default: :obj:`None`)
#     """

#     url = 'https://raw.githubusercontent.com/graphdml-uiuc-jlu/geom-gcn/master'

#     def __init__(self, root, name, transform=None, pre_transform=None):
#         self.name = name.lower()
#         assert self.name in ['chameleon', 'squirrel']

#         super(WikipediaNetwork, self).__init__(root, transform, pre_transform)
#         self.data, self.slices = torch.load(self.processed_paths[0])

#     @property
#     def raw_dir(self):
#         return osp.join(self.root, self.name, 'raw')

#     @property
#     def processed_dir(self):
#         return osp.join(self.root, self.name, 'processed')

#     @property
#     def raw_file_names(self):
#         return ['out1_node_feature_label.txt', 'out1_graph_edges.txt'] + [
#             '{}_split_0.6_0.2_{}.npz'.format(self.name, i) for i in range(10)
#         ]

#     @property
#     def processed_file_names(self):
#         return 'data.pt'

#     def download(self):
#         for f in self.raw_file_names[:2]:
#             download_url(f'{self.url}/new_data/{self.name}/{f}', self.raw_dir)
#         for f in self.raw_file_names[2:]:
#             download_url(f'{self.url}/splits/{f}', self.raw_dir)

#     def process(self):
#         with open(self.raw_paths[0], 'r') as f:
#             data = f.read().split('\n')[1:-1]
#             x = [[float(v) for v in r.split('\t')[1].split(',')] for r in data]
#             x = torch.tensor(x, dtype=torch.float)

#             y = [int(r.split('\t')[2]) for r in data]
#             y = torch.tensor(y, dtype=torch.long)

#         with open(self.raw_paths[1], 'r') as f:
#             data = f.read().split('\n')[1:-1]
#             data = [[int(v) for v in r.split('\t')] for r in data]
#             edge_index = torch.tensor(data, dtype=torch.long).t().contiguous()
#             edge_index, _ = coalesce(edge_index, None, x.size(0), x.size(0))

#         train_masks, val_masks, test_masks = [], [], []
#         for f in self.raw_paths[2:]:
#             tmp = np.load(f)
#             train_masks += [torch.from_numpy(tmp['train_mask']).to(torch.bool)]
#             val_masks += [torch.from_numpy(tmp['val_mask']).to(torch.bool)]
#             test_masks += [torch.from_numpy(tmp['test_mask']).to(torch.bool)]
#         train_mask = torch.stack(train_masks, dim=1)
#         val_mask = torch.stack(val_masks, dim=1)
#         test_mask = torch.stack(test_masks, dim=1)

#         data = Data(x=x, edge_index=edge_index, y=y, train_mask=train_mask,
#                     val_mask=val_mask, test_mask=test_mask)
#         data = data if self.pre_transform is None else self.pre_transform(data)
#         torch.save(self.collate([data]), self.processed_paths[0])

#     def __repr__(self):
#         return '{}()'.format(self.name)


def get_heterophilic_dataset_IDM(dataset_name, data_path, idx_split):
    dataset_name = dataset_name.lower()
    assert dataset_name in ['cornell', 'texas', 'wisconsin', 'chameleon', 'squirrel']
    if dataset_name in ['cornell', 'texas', 'wisconsin']:
        dataset = WebKB_new(data_path, dataset_name, transform=T.NormalizeFeatures())
    elif dataset_name in ['chameleon', 'squirrel']:
            dataset = WikipediaNetwork(data_path, dataset_name, transform=T.NormalizeFeatures())
    data = dataset[0]
    train_mask, val_mask, test_mask = data.train_mask[:, idx_split], data.val_mask[:, idx_split], \
                                      data.test_mask[:, idx_split]
    edge_index, x, y = data.edge_index, data.x, data.y
    y = y.long()
    row, col = edge_index[0, :], edge_index[1, :]
    val = np.ones(len(row))
    adj = sp.coo_matrix((val, (row,col)), shape=(x.size(0), x.size(0)))
    sp_adj = AugNorm(adj)
    adj = sp_mtx_to_sp_tnsr(sp_adj, device='cuda')
    return adj, sp_adj, x, y, train_mask, val_mask, test_mask