import os
import importlib
import warnings
from pprint import pformat
from typing import (
    Any,
    Dict,
    List,
    Optional,
    Set,
    Tuple,
    Union,
    get_args,
)

import torch
import numpy as np
import anndata as ad
import pandas as pd
from collections import abc
from einops import rearrange
from functools import partial
from omegaconf import DictConfig

import multiprocessing as mp
from threading import Thread
from queue import Queue

from inspect import isfunction
from PIL import Image, ImageDraw, ImageFont

from celldiff.typing import TensorArray


def as_tensor(x: TensorArray, assert_type: bool = False):
    if isinstance(x, np.ndarray):
        x = torch.from_numpy(x)
    elif not isinstance(x, torch.Tensor) and assert_type:
        raise TypeError(f"Expecting tensor or numpy array, got, {type(x)}")
    return x


def as_1d_vec(x: torch.Tensor) -> torch.Tensor:
    if len(x.shape) == 1:
        x = x.unsqueeze(-1)
    elif len(x.shape) == 1:
        raise ValueError(f"input must be one or two dimensional tensor, got {x.shape}")
    return x


def as_array(x: TensorArray, assert_type: bool = False):
    if isinstance(x, torch.Tensor):
        x = x.detach().cpu().numpy()
    elif not isinstance(x, np.ndarray) and assert_type:
        raise TypeError(f"Expecting tensor or numpy array, got, {type(x)}")
    return x


def check_str_option(
    input_name: str,
    input_opt: Optional[str],
    available_opts: Union[List[str], Tuple[str], Set[str], Any],
    optional: bool = True,
    warn_fail: bool = True,
) -> str:
    """Pass through an input option and raise ValueError if it is in invalid."""
    if not isinstance(available_opts, (list, tuple, set)):
        try:
            available_opts = get_args(available_opts)
        except Exception:
            if warn_fail:
                warnings.warn(
                    f"Fail to check option for {input_name}",
                    RuntimeWarning,
                    stacklevel=2,
                )

    if input_opt is None:
        if not optional:
            raise ValueError(f"{input_name} can not be None.")
    elif input_opt not in available_opts:
        raise ValueError(
            f"Unknown option {input_opt!r} for {input_name}. "
            f"Available options are: {available_opts}.",
        )

    return input_opt


def ensure_dir(path):
    os.makedirs(path, exist_ok=True)
    return path


def split_molecules(
    umis: np.ndarray,
    data_split: float,
    overlap_factor: float = 0.0,
    random_state: np.random.RandomState = None,
):
    """Splits molecules into two (potentially overlapping) groups.

    :param umis: Array of molecules to split
    :param data_split: Proportion of molecules to assign to the first group
    :param overlap_factor: Overlap correction factor, if desired
    :param random_state: For reproducible sampling
    :return: umis_X and umis_Y, representing ``split`` and ``~(1 - split)`` counts
             sampled from the input array
    """
    if random_state is None:
        random_state = np.random.RandomState()

    umis_X_disjoint = random_state.binomial(umis, data_split - overlap_factor)
    umis_Y_disjoint = random_state.binomial(
        umis - umis_X_disjoint, (1 - data_split) / (1 - data_split + overlap_factor)
    )
    overlap_factor = umis - umis_X_disjoint - umis_Y_disjoint
    umis_X = umis_X_disjoint + overlap_factor
    umis_Y = umis_Y_disjoint + overlap_factor

    return umis_X, umis_Y

def log_txt_as_img(wh, xc, size=10):
    # wh a tuple of (width, height)
    # xc a list of captions to plot
    b = len(xc)
    txts = list()
    for bi in range(b):
        txt = Image.new("RGB", wh, color="white")
        draw = ImageDraw.Draw(txt)
        font = ImageFont.truetype('data/DejaVuSans.ttf', size=size)
        nc = int(40 * (wh[0] / 256))
        lines = "\n".join(xc[bi][start:start + nc] for start in range(0, len(xc[bi]), nc))

        try:
            draw.text((0, 0), lines, fill="black", font=font)
        except UnicodeEncodeError:
            print("Cant encode string for logging. Skipping.")

        txt = np.array(txt).transpose(2, 0, 1) / 127.5 - 1.0
        txts.append(txt)
    txts = np.stack(txts)
    txts = torch.tensor(txts)
    return txts


def ismap(x):
    if not isinstance(x, torch.Tensor):
        return False
    return (len(x.shape) == 4) and (x.shape[1] > 3)


def isimage(x):
    if not isinstance(x, torch.Tensor):
        return False
    return (len(x.shape) == 4) and (x.shape[1] == 3 or x.shape[1] == 1)


def exists(x):
    return x is not None


def default(val, d):
    if exists(val):
        return val
    return d() if isfunction(d) else d


def list_exclude(input_list: List[Any], exclude_list: Optional[Union[List[Any], Set[Any]]]) -> List[Any]:
    if exclude_list is not None:
        exclude_set = set(exclude_list)
        input_list = [i for i in input_list if i not in exclude_set]
    return input_list


def mean_flat(tensor):
    """
    https://github.com/openai/guided-diffusion/blob/27c20a8fab9cb472df5d6bdd6c8d11c8f430b924/guided_diffusion/nn.py#L86
    Take the mean over all non-batch dimensions.
    """
    return tensor.mean(dim=list(range(1, len(tensor.shape))))


def count_params(model, verbose=False):
    total_params = sum(p.numel() for p in model.parameters())
    if verbose:
        print(f"{model.__class__.__name__} has {total_params * 1.e-6:.2f} M params.")
    return total_params


def instantiate_from_config(
    config: Union[Dict, DictConfig, str],
    _target_key: str = "target",
    _params_key: str = "params",
    _catch_conflict: bool = True,
    **extra_kwargs: Any,
):
    # Check target specificiation and handel special conditions
    if _target_key not in config:
        if config == "__is_first_stage__":
            return None
        elif config == "__is_unconditional__":
            return None
        raise KeyError(f"Expected key `{_target_key}` to instantiate.")

    # Obtain target object and kwargs
    cls = get_obj_from_str(config["target"])
    kwargs = config.get(_params_key, dict())

    # Check conflict and merge kwargs
    if (common_keys := sorted(set(kwargs) & set(extra_kwargs))):
        diff_keys = []
        for key in common_keys:
            if kwargs[key] != extra_kwargs[key]:
                diff_keys.append(key)

        if diff_keys and _catch_conflict:
            conflicting_config_kwargs = {i: kwargs[i] for i in diff_keys}
            conflicting_extra_kwargs = {i: extra_kwargs[i] for i in diff_keys}
            raise ValueError(
                "Conflicting parameters between configs and those that are "
                "additionally specified. Please resolve or set _catch_conflict "
                f"to False to bypass this issue.\n{conflicting_config_kwargs=}\n"
                f"{conflicting_extra_kwargs=}\n",
            )
    kwargs = {**kwargs, **extra_kwargs}

    # Instantiate object and handel exception during instantiation
    try:
        return cls(**kwargs)
    except Exception as e:
        raise RuntimeError(f"Failed to instantiate {cls!r} with kwargs:\n{pformat(kwargs)}") from e


def get_obj_from_str(string, reload=False):
    module, cls = string.rsplit(".", 1)
    if reload:
        module_imp = importlib.import_module(module)
        importlib.reload(module_imp)
    return getattr(importlib.import_module(module, package=None), cls)


def _do_parallel_data_prefetch(func, Q, data, idx, idx_to_fn=False):
    # create dummy dataset instance

    # run prefetching
    if idx_to_fn:
        res = func(data, worker_id=idx)
    else:
        res = func(data)
    Q.put([idx, res])
    Q.put("Done")


def parallel_data_prefetch(
        func: callable, data, n_proc, target_data_type="ndarray", cpu_intensive=True, use_worker_id=False
):
    # if target_data_type not in ["ndarray", "list"]:
    #     raise ValueError(
    #         "Data, which is passed to parallel_data_prefetch has to be either of type list or ndarray."
    #     )
    if isinstance(data, np.ndarray) and target_data_type == "list":
        raise ValueError("list expected but function got ndarray.")
    elif isinstance(data, abc.Iterable):
        if isinstance(data, dict):
            print(
                f'WARNING:"data" argument passed to parallel_data_prefetch is a dict: Using only its values and disregarding keys.'
            )
            data = list(data.values())
        if target_data_type == "ndarray":
            data = np.asarray(data)
        else:
            data = list(data)
    else:
        raise TypeError(
            f"The data, that shall be processed parallel has to be either an np.ndarray or an Iterable, but is actually {type(data)}."
        )

    if cpu_intensive:
        Q = mp.Queue(1000)
        proc = mp.Process
    else:
        Q = Queue(1000)
        proc = Thread
    # spawn processes
    if target_data_type == "ndarray":
        arguments = [
            [func, Q, part, i, use_worker_id]
            for i, part in enumerate(np.array_split(data, n_proc))
        ]
    else:
        step = (
            int(len(data) / n_proc + 1)
            if len(data) % n_proc != 0
            else int(len(data) / n_proc)
        )
        arguments = [
            [func, Q, part, i, use_worker_id]
            for i, part in enumerate(
                [data[i: i + step] for i in range(0, len(data), step)]
            )
        ]
    processes = []
    for i in range(n_proc):
        p = proc(target=_do_parallel_data_prefetch, args=arguments[i])
        processes += [p]

    # start processes
    print(f"Start prefetching...")
    import time

    start = time.time()
    gather_res = [[] for _ in range(n_proc)]
    try:
        for p in processes:
            p.start()

        k = 0
        while k < n_proc:
            # get result
            res = Q.get()
            if res == "Done":
                k += 1
            else:
                gather_res[res[0]] = res[1]

    except Exception as e:
        print("Exception: ", e)
        for p in processes:
            p.terminate()

        raise e
    finally:
        for p in processes:
            p.join()
        print(f"Prefetching complete. [{time.time() - start} sec.]")

    if target_data_type == 'ndarray':
        if not isinstance(gather_res[0], np.ndarray):
            return np.concatenate([np.asarray(r) for r in gather_res], axis=0)

        # order outputs
        return np.concatenate(gather_res, axis=0)
    elif target_data_type == 'list':
        out = []
        for r in gather_res:
            out.extend(r)
        return out
    else:
        return gather_res

def stratified_sample_genes_by_sparsity(data, boundaries=None, seed=10, sample_per_group=25):
    df = data.to_df()
    zero_rates = 1 - df.astype(bool).sum(axis=0) / df.shape[0]
    if boundaries is None:
        # boundaries = [0, zero_rates.mean() - zero_rates.std(), zero_rates.mean(), 
        #               min(zero_rates.mean() + zero_rates.std(), 1)]
        boundaries = [0, 0.75, 0.9, 0.95, 1]
    gene_group = pd.cut(zero_rates, boundaries, labels=False)
    # gene_df = pd.DataFrame({'zero_rates': zero_rates, 'gene_group': gene_group})
    zero_rates = zero_rates.groupby(gene_group, group_keys=False)
    samples = zero_rates.apply(lambda x: x.sample(min(len(x), sample_per_group), random_state=seed))
    return list(samples.index)
