#!/bin/bash --login
# Uncomment specific block of interest and run
# $ bash script.sh
#
# Specify CUDA devices
# $ CUDA_VISIBLE_DEVICES=1 bash script.sh
#
# Specify log directory
# $ LOGDIR=/localscrtach/celldiff bash script.sh
#
# To view the generated script without executing, pass the TEST_FLAG envar as 1
# $ TEST_FLAG=1 bash script.sh

trap "echo ERROR && exit 1" ERR

# --------------------
export CUDA_VISIBLE_DEVICES=${CUDA_VISIBLE_DEVICES:-0}
LOGDIR=${LOGDIR:-logs}
NAME=v7.5

CONFIG_PREFIX="configs/eval"

TEST_FLAG=${TEST_FLAG:-0}

OFFLINE_SETTINGS="--wandb f"
# --------------------

HOMEDIR=$(dirname $(dirname $(realpath $0)))
cd $HOMEDIR
echo HOMEDIR=$HOMEDIR

launch () {
    full_settings=($@)
    task=${full_settings[0]}
    name=${full_settings[1]}
    seed=${full_settings[2]}

    if [[ $task == denoising ]] || [[ $task == perturbation ]]; then
        if [[ $name == jurkat ]]; then
            dataset_name=Jurkat
        elif [[ $name == pbmc1k ]]; then
            dataset_name=PBMC1K
        elif [[ $name == 293t ]]; then
            dataset_name=293T
        elif [[ $name == pbmc ]]; then
            dataset_name=pbmc
        elif [[ $name == hpoly ]]; then
            dataset_name=hpoly
        elif [[ $name == salmonella ]]; then
            dataset_name=salmonella
        else
            echo Unknown ${task} dataset ${name} && exit 1
        fi

        data_settings="data.params.train.params.dataset=${dataset_name} data.params.train.params.fname=${dataset_name}_processed.h5ad"
        data_settings+=" data.params.test.params.dataset=${dataset_name} data.params.test.params.fname=${dataset_name}_processed.h5ad"

    elif [[ $task == genepert ]]; then
        task_specific_settings="--load_best t --ckpt_every_n_epochs 5"
        if [[ $name == adamson ]]; then
            dataset_name=adamson
        task_specific_settings+=" lightning.trainer.max_epochs=50"
        elif [[ $name == dixit ]]; then
            dataset_name=dixit
        task_specific_settings+=" lightning.trainer.max_epochs=150"
        elif [[ $name == norman ]]; then
            dataset_name=norman
        task_specific_settings+=" lightning.trainer.max_epochs=100"
        else
            echo Unknown ${task} dataset ${name} && exit 1
        fi

        data_settings="data.params.train.params.dataset=${dataset_name}  data.params.validation.params.dataset=${dataset_name} data.params.test.params.dataset=${dataset_name}"

    elif [[ $task == annotation ]]; then
        if [[ $name == pbmc12k ]]; then
            dataset_name=PBMC12K
        elif [[ $name == pbmc106k ]]; then
            dataset_name=PBMC106K
        elif [[ $name == immune ]]; then
            dataset_name=Immune
            task_specific_settings+="model.base_learning_rate=1e-7"
        elif [[ $name == lung ]]; then
            dataset_name=LungAtlas
        elif [[ $name == pancreas ]]; then
            dataset_name=Pancreas
        else
            echo Unknown ${task} dataset ${name} && exit 1
        fi

        data_settings="data.params.train.target=celldiff.data.${name}.${dataset_name}Train"
        data_settings+=" data.params.train.params.fname=${dataset_name}_processed.h5ad"
        data_settings+=" data.params.test.target=celldiff.data.${name}.${dataset_name}Test"
        data_settings+=" data.params.test.params.fname=${dataset_name}_processed.h5ad"

    elif [[ $task == annotation_cellxgene ]]; then
        if [[ $name == hlcasub ]]; then
            dataset_name=HLCA_sub
        elif [[ $name == immunesub ]]; then
            dataset_name=Immune_sub
        elif [[ $name == brain ]]; then
            dataset_name=Brain
        elif [[ $name == liver ]]; then
            dataset_name=Liver
        else
            echo Unknown ${task} dataset ${name} && exit 1
        fi

        data_settings="data.params.train.params.dataset=${dataset_name} data.params.test.params.dataset=${dataset_name}"

    else
        echo Unknown task $task && exit 1
    fi

    script="python main.py -b ${CONFIG_PREFIX}_${task}.yaml --name ${NAME} --wandb t --seed ${seed}"
    [[ $task == annotation_cellxgene ]] && task=annotation
    script+=" --logdir ${LOGDIR} --postfix ${task}_${name}_r.${seed} ${OFFLINE_SETTINGS} ${data_settings} ${task_specific_settings}"

    echo ${script} && [[ $TEST_FLAG == 0 ]] && eval ${script}
}


task=$1
name=$2
SEED=${SEED:-10}

echo task=$task name=$name seed=$SEED
launch ${task} ${name} ${SEED}
